/*
 * Created on Jul 6, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.dataaccess.ConnectionTypeDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.MessageDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI;
import com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI;
import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.ConnectionType;
import com.tandbergtv.watchpoint.studio.dto.IPersistable;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IMessageUIDGenerator;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceErrorCode;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.service.ServiceValidationException;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * The IResourceTypeService implementation.
 * 
 * @author Vijay Silva
 */
public class ResourceTypeService extends ServiceImpl implements IResourceTypeService
{
	private static final int MAX_UIDGEN_RETRY_COUNT = 100;
	
	private ResourceTypeFileSystemManager resourceTypeFileSystemManager;
	
	/**
	 * Constructor
	 */
	public ResourceTypeService() {
		super();
		resourceTypeFileSystemManager = new ResourceTypeFileSystemManager();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#getResourceTypeList()
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceType> getResourceTypeList()
	{
		Class<?>[] types = {};
		return (List<ResourceType>) this.performOperation(types);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#getResourceType(long)
	 */
	public ResourceType getResourceType(long resourceTypeId)
	{
		Class<?>[] types = { long.class };
		return (ResourceType) this.performOperation(types, resourceTypeId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#getResourceTypeByName(java.lang.String)
	 */
	public ResourceType getResourceTypeByName(String name)
	{
		Class<?>[] types = { String.class };
		return (ResourceType) this.performOperation(types, name);
	}
	
	public ResourceType getResourceTypeByPath(String path)
	{
		Class<?>[] types = { String.class };
		return (ResourceType) super.performOperation(types, path);
	}
	
	protected ResourceType getResourceTypeByPath(String path, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return resourceTypeDAO.findByPath(path);
	}
	
	@SuppressWarnings("unchecked")
	public List<ResourceType> getResourceTypeByProject(String projectName) {
		Class<?>[] types = {String.class};
		return (List<ResourceType>) this.performOperation(types, projectName);
	}
	
	protected List<ResourceType> getResourceTypeByProject(String projectName, IPersistenceContext context) {
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return resourceTypeDAO.findByProjectName(projectName);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#getResourceTypeBySystemId(java.lang.String)
	 */
	public ResourceType getResourceTypeBySystemId(String systemId)
	{
		Class<?>[] types = { String.class };
		return (ResourceType) this.performOperation(types, systemId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#createResourceType(com.tandbergtv.watchpoint.studio.dto.ResourceType)
	 */
	public ResourceType createResourceTypeInDataBase(ResourceType resourceType)
	{
		Class<?>[] types = { ResourceType.class };
		return (ResourceType) this.performOperation(types, resourceType);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#updateResourceType(com.tandbergtv.watchpoint.studio.dto.ResourceType)
	 */
	public ResourceType updateResourceTypeInFileSystem(ResourceType resourceType)
	{
		ResourceType updated = null;
		try {
			// Updates the resource type in file system
			resourceTypeFileSystemManager.updateResourceType(resourceType);
			updated = resourceTypeFileSystemManager.getResourceType(resourceType.getProjectName());
		} catch (Exception e) {
			throw new ServiceException(ServiceErrorCode.RT_UPDATE_IN_FILE_SYSTEM_ERROR, e);
		}
		return updated;
	}
	
	public ResourceType updateResourceTypeInDataBase(ResourceType resourceType){
		Class<?>[] types = { ResourceType.class };
		return (ResourceType) this.performOperation(types, resourceType);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#deleteResourceType(long)
	 */
	public void deleteResourceType(long resourceTypeId)
	{
		Class<?>[] types = { long.class };
		this.performOperation(types, resourceTypeId);
	}
	
	public ResourceType getResourceTypeFromFileSystem(String projectName){
		return resourceTypeFileSystemManager.getResourceType(projectName);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#validateResourceType(long)
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateResourceType(long resourceTypeId)
	{
		Class<?>[] types = { long.class };
		return (List<ValidationMessage>) this.performOperation(types, resourceTypeId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#isResourceTypeNameUnique(java.lang.String)
	 */
	public boolean isResourceTypeNameUnique(String name)
	{
		Class<?>[] types = { String.class };
		return (Boolean) this.performOperation(types, name);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#isResourceTypeSystemIdUnique(java.lang.String)
	 */
	public boolean isResourceTypeSystemIdUnique(String systemId)
	{
		Class<?>[] types = { String.class };
		return (Boolean) this.performOperation(types, systemId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#getUnusedResourceTypeMessages(long)
	 */
	@SuppressWarnings("unchecked")
	public List<Message> getUnusedResourceTypeMessages(long resourceTypeId)
	{
		Class<?>[] types = { long.class };
		return (List<Message>) this.performOperation(types, resourceTypeId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#getUnusedResourceTypeMessages(long, long)
	 */
	@SuppressWarnings("unchecked")
	public List<Message> getUnusedResourceTypeMessages(long resourceTypeId, long nodeDefinitionId)
	{
		Class<?>[] types = { long.class, long.class };
		return (List<Message>) this.performOperation(types, resourceTypeId, nodeDefinitionId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#findMessageByUID(java.lang.String)
	 */
	public Message findMessageByUID(String messageUID)
	{
		Class<?>[] types = { String.class };
		return (Message) this.performOperation(types, messageUID);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#getAllConnectionTypes()
	 */
	@SuppressWarnings("unchecked")
	public List<ConnectionType> getAllConnectionTypes()
	{
		Class<?>[] types = {};
		return (List<ConnectionType>) this.performOperation(types);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#createMessage(com.tandbergtv.watchpoint.studio.dto.Message)
	 */
	public Message createMessage(Message message)
	{
		Class<?>[] types = { Message.class };
		return (Message) this.performOperation(types, message);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceTypeService#deleteMessage(long)
	 */
	public void deleteMessage(long messageId)
	{
		Class<?>[] types = { long.class };
		this.performOperation(types, messageId);
	}

    // ========================================================================
	// ====== BUSINESS LOGIC IMPLEMENTATION (WITHOUT CONTEXT MANAGEMENT)
	// ========================================================================

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#getResourceTypeList()
	 */
	protected List<ResourceType> getResourceTypeList(IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return resourceTypeDAO.findAll();
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#getResourceType(long)
	 */
	protected ResourceType getResourceType(long resourceTypeId, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return this.findResourceType(resourceTypeId, resourceTypeDAO);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#getResourceTypeByName(java.lang.String)
	 */
	protected ResourceType getResourceTypeByName(String name, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return resourceTypeDAO.findByName(name);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#getResourceTypeBySystemId(String)
	 */
	protected ResourceType getResourceTypeBySystemId(String systemId, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return resourceTypeDAO.findBySystemId(systemId);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#createResourceType(com.tandbergtv.watchpoint.studio.dto.ResourceType)
	 */
	protected ResourceType createResourceTypeInDataBase(ResourceType resourceType, IPersistenceContext context)
	{
		resourceType.setId(IPersistable.DEFAULT_ID);

		/* Set the initial Version data */
		if(resourceType.getVersion() == 0)
			resourceType.setVersion(1);
		
		resourceType.setDirty(true);

		/* Set the Message UID Sequence Generator and clear all messages */
		resourceType.setMessageSequenceNumber(1);
		resourceType.setMessages(new HashSet<Message>());

		/* Trim the Name */
		String name = resourceType.getName();
		if (name != null)
			resourceType.setName(name.trim());

		/* Trim the System Id */
		String systemId = resourceType.getSystemId();
		if (systemId != null)
			resourceType.setSystemId(systemId.trim());

		/* Create the new Resource Type */
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return resourceTypeDAO.create(resourceType);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#updateResourceType(com.tandbergtv.watchpoint.studio.dto.ResourceType)
	 */
	protected ResourceType updateResourceTypeInDataBase(ResourceType resourceType, IPersistenceContext context)
	{
		long resourceTypeId = resourceType.getId();
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		ResourceType currentResourceType = this.findResourceType(resourceTypeId, resourceTypeDAO);

		/* Validate that the Resource Type is not out-of-the-box */
		if(currentResourceType.isOutOfTheBox())
		{
			String message = "Modifying an out-of-the-box Resource Type is not allowed.";
			throw new ServiceException(ServiceErrorCode.RESOURCE_TYPE_CANNOT_UPDATE_OOBRESTYPE, message);
		}

		/* Set the Version data */
		int currentVersion = currentResourceType.getVersion();
		if (!currentResourceType.isDirty())
			currentVersion++;

		resourceType.setVersion(currentVersion);
		resourceType.setDirty(true);

		/* Set the Sequence number and keep only existing messages */
		resourceType.setMessageSequenceNumber(currentResourceType.getMessageSequenceNumber());
		resourceType.setMessages(this.getMessagesForUpdate(resourceType, currentResourceType));
		
		/* Update the Resource Type and its Messages in the DB */
		MessageDAI messageDAO = this.createMessageDAO(context);
		for (Message message : resourceType.getMessages())
			messageDAO.update(message);

		return resourceTypeDAO.update(resourceType);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#deleteResourceType(long)
	 */
	protected void deleteResourceType(long resourceTypeId, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		MessageDAI messageDAO = this.createMessageDAO(context);

		/* Validate that the Resource Type can be deleted */
		ResourceType resourceType = this.findResourceType(resourceTypeId, resourceTypeDAO);
		this.validateDeletion(resourceType, context);

		/* Delete the Resource Type and its Messages from the DB (physical delete) */
		Set<Message> messages = resourceType.getMessages();
		if (messages != null)
		{
			for (Message message : messages)
				messageDAO.delete(message.getId());
		}

		resourceTypeDAO.delete(resourceTypeId);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#exportResourceType(long,
	 *      java.util.Map)
	 */
	protected ResourceType exportResourceType(long resourceTypeId, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		ResourceType resourceType = this.findResourceType(resourceTypeId, resourceTypeDAO);

		validateExport(resourceType);
		
		/* Update the version data and save */
		if (!resourceType.isDirty())
		{ // Even if not 'dirty', update the version number since libraries / config could change
			resourceType.setVersion(resourceType.getVersion() + 1);
		}
		resourceType.setDirty(false);
		resourceType = resourceTypeDAO.update(resourceType);

		return resourceType;
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#validateResourceType(long)
	 */
	protected List<ValidationMessage> validateResourceType(long resourceTypeId,
			IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		ResourceType resourceType = this.findResourceType(resourceTypeId, resourceTypeDAO);

		return this.validateResourceType(resourceType);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#isResourceTypeNameUnique(String)
	 */
	protected boolean isResourceTypeNameUnique(String name, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return (resourceTypeDAO.getCountByName(name) == 0);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#isResourceTypeSystemIdUnique(String)
	 */
	protected boolean isResourceTypeSystemIdUnique(String systemId, IPersistenceContext context)
	{
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		return (resourceTypeDAO.getCountBySystemId(systemId) == 0);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#isMessageNameUnique(long,
	 *      java.lang.String)
	 */
	protected boolean isMessageNameUnique(long resourceTypeId, String name,
			IPersistenceContext context)
	{
		MessageDAI messageDAO = this.createMessageDAO(context);
		return (messageDAO.getCountByResourceTypeAndName(resourceTypeId, name) == 0);
	}
	
	/**
	 * Implementation of the business logic for the getUnusedResourceTypeMessages method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#getUnusedResourceTypeMessages(long)
	 */
	protected List<Message> getUnusedResourceTypeMessages(long resourceTypeId,
			IPersistenceContext context)
	{
		return this.createMessageDAO(context).findUnusedByResourceType(resourceTypeId);
	}

	/**
	 * Implementation of the business logic for the getUnusedResourceTypeMessages method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#getUnusedResourceTypeMessages(long, long)
	 */
	protected List<Message> getUnusedResourceTypeMessages(long resourceTypeId, long nodeDefinitionId, 
			IPersistenceContext context)
	{
		return this.createMessageDAO(context).findUnusedByResourceType(resourceTypeId, nodeDefinitionId);
	}

	/**
	 * Implementation of the business logic for the findMessageByUID method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#findMessageByUID(java.lang.String)
	 */
	protected Message findMessageByUID(String messageUID, IPersistenceContext context)
	{
		return this.createMessageDAO(context).findByUID(messageUID);
	}

	/**
	 * Implementation of the business logic for the getAllConnectionTypes method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#getAllConnectionTypes()
	 */
	protected List<ConnectionType> getAllConnectionTypes(IPersistenceContext context)
	{
		ConnectionTypeDAI connectionTypeDAO = this.createConnectionTypeDAO(context);
		return connectionTypeDAO.findAll();
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#createMessage(Message)
	 */
	protected Message createMessage(Message message, IPersistenceContext context)
	{
		/* Get the Resource Type for the Message and clone it */
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		long resourceTypeId = message.getResourceType().getId();
		ResourceType currentResourceType = this.findResourceType(resourceTypeId, resourceTypeDAO);
		
		/* Validate that the Resource Type is not out-of-the-box */
		if(currentResourceType.isOutOfTheBox())
		{
			String errMsg = "Cannot create a new Message for an out-of-the-box Resource Type.";
			throw new ServiceException(ServiceErrorCode.MESSAGE_CANNOT_CREATE_OOBRESTYPE, errMsg);
		}

		ResourceType clonedResourceType = currentResourceType.clone();
		clonedResourceType.addMessage(message);

		/* Generate the UID for the message if required */
		boolean generateUID = message.isUIDGenerationRequired();
		if (generateUID)
		{
			this.generateMessageUID(clonedResourceType, message);
			int sequence = clonedResourceType.getMessageSequenceNumber();
			currentResourceType.setMessageSequenceNumber(sequence);
		}

		/* Trim the Name and UID */
		String name = message.getName();
		if (name != null)
			message.setName(name.trim());

		String uid = message.getUid();
		if (uid != null)
			message.setUid(uid.trim());

		/* Update the Version data */
		int currentVersion = currentResourceType.getVersion();
		if (!currentResourceType.isDirty())
			currentVersion++;

		currentResourceType.setVersion(currentVersion);
		currentResourceType.setDirty(true);
		resourceTypeDAO.update(currentResourceType);

		/* Create the Message */
		MessageDAI messageDAO = this.createMessageDAO(context);
		return messageDAO.create(message);
	}

	/**
	 * Business Logic implementation.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceTypeService#deleteMessage(long)
	 */
	protected void deleteMessage(long messageId, IPersistenceContext context)
	{
		MessageDAI messageDAO = this.createMessageDAO(context);
		Message message = messageDAO.find(messageId);
		if (message == null)
			return;

		/* Lock the Resource Type before deleting the Message */
		long resourceTypeId = message.getResourceType().getId();
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		ResourceType resourceType = this.findResourceType(resourceTypeId, resourceTypeDAO);

		/* Validate the Message before deleting */
		this.validateMessageDeletion(message, resourceType, context);
		
		resourceType.removeMessage(message);

		/* Update the Version data */
		if (!resourceType.isDirty())
		{
			resourceType.setVersion(resourceType.getVersion() + 1);
			resourceType.setDirty(true);
		}

		/* Delete the Message */
		messageDAO.delete(messageId);
	}

	// ========================================================================
	// ==================== VALIDATION METHODS
	// ========================================================================

	/*
	 * Validate the Resource Type being deleted. The Resource Type must not have been previously
	 * exported, and must not be used by any Resource Group or Node Definition.
	 */
	private void validateDeletion(ResourceType resourceType, IPersistenceContext context)
	{
		/* Validate that the Resource Type is not out-of-the-box */
		if(resourceType.isOutOfTheBox())
		{
			String message = "Cannot delete an out-of-the-box Resource Type.";
			throw new ServiceException(ServiceErrorCode.RT_DELETE_OOB_ERROR, message);
		}
		
		long resourceTypeId = resourceType.getId();
		/* Validate that the Resource Type is not used by any existing Node Definitions */
		NodeDefinitionDTODAI nodeDefinitionDAI = this.createNodeDefinitionDAO(context);
		int nodeDefinitionCount = nodeDefinitionDAI.getCountByResourceType(resourceTypeId);
		if (nodeDefinitionCount > 0)
		{
			String msg = "The resource type is being used by existing node definitions, "
					+ "cannot delete this resource type.";
			throw new ServiceException(ServiceErrorCode.RT_DELETE_USED_BY_NODE_DEFN_ERROR, msg);
		}
	}

	/*
	 * Validates Resource Type for Export.
	 * 
	 * Rules:
	 * 1) If the Adaptor Type is NOT NONE, there should be messages
	 * 	  associated with the Resource Type.
	 */
	private void validateExport(ResourceType resourceType)
	{
		if(resourceType.getAdaptorType() != AdaptorType.NONE) {
			Set<Message> messages = resourceType.getMessages();
			if(messages == null || messages.isEmpty()) {
				List<ValidationMessage> validationMessages = new ArrayList<ValidationMessage>();
				ValidationMessageAdder.getInstance().addValidationMessage(validationMessages, resourceType, ValidationMessageCode.RESOURCE_TYPE_CANNOT_EXPORT_NO_MESSAGES);
				
				String errMsg = "The resource type has adaptor defined, but does not have messages. " +
						"Cannot export this resource type."; 
				throw new ServiceValidationException(errMsg, validationMessages);
			}
		}
	}
	
    /*
	 * Performs validation of a Resource Type using the Validation Framework
	 */
	private List<ValidationMessage> validateResourceType(ResourceType resourceType)
	{
		ValidationServiceFactory factory = ValidationServiceFactory.createFactory();
		IValidationService service = factory.createValidationService();
		List<ValidationMessage> resourceTypeErrorMessages = service.validateResourceType(resourceType);

		//run validation for each node definition
		IValidationService validationService = ValidationServiceFactory.createFactory().createValidationService();
		Set<NodeDefinitionDTO> nodes = resourceType.getNodes();
		List<ValidationMessage> nodeDefinitionErrorMessages = new ArrayList<ValidationMessage>();
		for (NodeDefinitionDTO nodeDTO: nodes) {
			try {
				
				NodeDefinition nodeDef = SemanticElementUtil.createNodeDefinition(nodeDTO.getXml());
				List<ValidationMessage> messages = validationService.validateNodeDefinition(nodeDef);
				if (!messages.isEmpty()){
					nodeDefinitionErrorMessages.addAll(messages);
				}
            } catch (Exception e) {
				String errMsg = "Error creating Node Definition from NodeDefinitionDTO xml";
				throw new ServiceException(ServiceErrorCode.GENERAL_VALIDATION_ERROR, errMsg);
			}
		}
		List<ValidationMessage> allMessages = new ArrayList<ValidationMessage>();
		allMessages.addAll(resourceTypeErrorMessages);
		allMessages.addAll(nodeDefinitionErrorMessages);
		
		return allMessages;
	}

	/*
	 * Validate the Message being deleted (message cannot be used)
	 */
	private void validateMessageDeletion(Message message, ResourceType resourceType, IPersistenceContext context)
	{
		/* Validate that the ResourceType is not out-of-the-box */
		if(resourceType.isOutOfTheBox())
		{
			String errMsg = "Cannot delete a Message that belongs to an out-of-the-box Resource Type.";
			throw new ServiceException(ServiceErrorCode.RT_MESSAGE_DELETE_OOBRESTYPE_ERROR, errMsg);
		}
		
		/* Validate that the Message is not used by any existing Node Definitions */
		NodeDefinitionDTODAI nodeDefinitionDAI = this.createNodeDefinitionDAO(context);
		int nodeDefinitionCount = nodeDefinitionDAI.getCountByMessage(message.getId());
		if (nodeDefinitionCount > 0)
		{
			String msg = "The message is being used by existing node definitions, "
					+ "cannot delete this message.";
			throw new ServiceException(ServiceErrorCode.RT_MESSAGE_DELETE_USED_BY_NODE_DEFN_ERROR,
					msg);
		}
	}

	/*
	 * Lock the Resource Type returning the locked resource type, throwing an error if the resource
	 * type is already deleted.
	 */
	private ResourceType findResourceType(long resourceTypeId, ResourceTypeDAI resourceTypeDAO)
	{
		ResourceType resourceType = resourceTypeDAO.find(resourceTypeId);

		// Validate that the Resource Type has not been previously deleted
		if (resourceType == null)
		{
			String msg = "The Resource Type has already been deleted, cannot perform operation on this Resource Type.";
			throw new ServiceException(ServiceErrorCode.WT_OPERATION_ON_DELETED_TEMPLATE, msg);
		}

		return resourceType;
	}

	/*
	 * Create the data access object for the ResourceType
	 */
	private ResourceTypeDAI createResourceTypeDAO(IPersistenceContext context)
	{
		Class<ResourceType> entityClass = ResourceType.class;
		DataAccessInterface<ResourceType, ?> dao = this.daFactory.createDataAccessObject(
				entityClass, context);
		return (ResourceTypeDAI) dao;
	}

	/*
	 * Create the data access object for the NodeDefinitionDTO
	 */
	private NodeDefinitionDTODAI createNodeDefinitionDAO(IPersistenceContext context)
	{
		Class<NodeDefinitionDTO> entityClass = NodeDefinitionDTO.class;
		DataAccessInterface<NodeDefinitionDTO, ?> dao = this.daFactory.createDataAccessObject(
				entityClass, context);
		return (NodeDefinitionDTODAI) dao;
	}

	/*
	 * Create the data access object for the ConnectionType
	 */
	private ConnectionTypeDAI createConnectionTypeDAO(IPersistenceContext context)
	{
		Class<ConnectionType> entityClass = ConnectionType.class;
		DataAccessInterface<ConnectionType, ?> dao = this.daFactory.createDataAccessObject(
				entityClass, context);
		return (ConnectionTypeDAI) dao;
	}

	/*
	 * Create the data access object for the Message
	 */
	private MessageDAI createMessageDAO(IPersistenceContext context)
	{
		Class<Message> entityClass = Message.class;
		DataAccessInterface<Message, ?> dao = this.daFactory.createDataAccessObject(entityClass,
				context);
		return (MessageDAI) dao;
	}

	/*
	 * Get the set of messages in the Resource Type that are already present in the current Resource
	 * Type present in the persistence layer.
	 */
	private Set<Message> getMessagesForUpdate(ResourceType resourceType,
			ResourceType currentResourceType)
	{
		Set<Message> messagesForUpdate = new HashSet<Message>();

		Set<Message> messages = resourceType.getMessages();
		Set<Message> currentMessages = currentResourceType.getMessages();

		if (messages == null || currentMessages == null)
			return messagesForUpdate;

		for (Message message : messages)
		{
			if (currentMessages.contains(message))
				messagesForUpdate.add(message);
		}

		return messagesForUpdate;
	}

	/*
	 * Method to generate all the Message UIDs for the Messages
	 */
	public void generateMessageUID(ResourceType resourceType, Message messageForGeneration)
	{
		/* Collect all existing Message UIDs */
		Set<String> usedUIDs = new HashSet<String>();
		for (Message message : resourceType.getMessages())
		{
			if (!message.isUIDGenerationRequired())
			{ // Collect all the used Message UIDs
				usedUIDs.add(message.getUid());
			}
		}

		/* Generate the UID required */
		ServiceFactory factory = ServiceFactory.createFactory();
		IMessageUIDGenerator uidGenerator = factory.createMessageUIDGenerator();
		int sequenceNumber = resourceType.getMessageSequenceNumber();
		int attemptCount = 0;

		while (messageForGeneration.isUIDGenerationRequired()
				&& attemptCount < MAX_UIDGEN_RETRY_COUNT)
		{
			String uid = uidGenerator.generateMessageUID(messageForGeneration, resourceType);
			resourceType.setMessageSequenceNumber(++sequenceNumber);
			attemptCount++;
			if (!usedUIDs.contains(uid))
			{
				messageForGeneration.setUid(uid);
				messageForGeneration.setUIDGenerationRequired(false);
			}
		}

		if (messageForGeneration.isUIDGenerationRequired())
		{ // Failed to generate the UID
			String msg = "Failed to generate a Message UID for the resource type.";
			throw new ServiceException(ServiceErrorCode.RT_UID_GENERATION_ERROR, msg);
		}
	}
}