/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.editor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.gef.palette.ConnectionCreationToolEntry;
import org.eclipse.gef.palette.CreationToolEntry;
import org.eclipse.gef.palette.MarqueeToolEntry;
import org.eclipse.gef.palette.PaletteEntry;
import org.eclipse.gef.palette.PaletteGroup;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.palette.SelectionToolEntry;
import org.eclipse.gef.palette.ToolEntry;
import org.eclipse.jface.resource.ImageDescriptor;
import org.jbpm.gd.common.editor.CreationFactory;
import org.jbpm.gd.jpdl.Plugin;

/**
 * The palette class for the node definition editor. This palette is only used when the node
 * definition is operating on a super state node definition.
 * 
 * @author Imran Naqvi
 * 
 */
public class NodeDefinitionPaletteRoot extends PaletteRoot {

	private static final String PALETTE_EXTENSION_POINT = "org.jbpm.gd.jpdl.palette";

	private NodeDefinitionEditor editor;

	private List<IConfigurationElement> categoryList = new ArrayList<IConfigurationElement>();

	private Map<String, List<?>> categoryMap = new HashMap<String, List<?>>();

	/**
	 * Ctor.
	 * 
	 * @param editor
	 */
	public NodeDefinitionPaletteRoot(NodeDefinitionEditor editor) {
		this.editor = editor;
		addControls();
	}

	/*
	 * Adds the controls to show in this palette. Those objects are loaded from the plugin.xml file
	 * which are associated with this editor.
	 */
	private void addControls() {
		add(createDefaultControls());
		IConfigurationElement[] configurationElements = Platform.getExtensionRegistry()
				.getConfigurationElementsFor(PALETTE_EXTENSION_POINT);
		for (int i = 0; i < configurationElements.length; i++) {
			String editorClass = configurationElements[i].getAttribute("editorClass");
			if (editorClass != null && editorClass.equals(editor.getClass().getName())) {
				processPaletteContribution(configurationElements[i]);
			}
		}
		for (int i = 0; i < categoryList.size(); i++) {
			IConfigurationElement categoryElement = (IConfigurationElement) categoryList.get(i);
			String categoryName = categoryElement.getNamespaceIdentifier() + "."
					+ categoryElement.getAttribute("id");
			List<?> entries = (List<?>) categoryMap.get(categoryName);
			if (entries != null && !entries.isEmpty()) {
				PaletteGroup paletteGroup = new PaletteGroup(categoryName);
				for (int j = 0; j < entries.size(); j++) {
					IConfigurationElement entryElement = (IConfigurationElement) entries.get(j);
					String icon = entryElement.getAttribute("icon");
					String entryName = entryElement.getNamespaceIdentifier() + "."
							+ entryElement.getAttribute("id");
					String label = entryElement.getAttribute("label");
					boolean isNode = !"false".equals(entryElement.getAttribute("node"));
					String object = entryElement.getAttribute("object");
					String tooltip = entryElement.getAttribute("tooltip");
					PaletteEntry entry;
					CreationFactory factory = new CreationFactory(object,
							editor.getSemanticElementFactory(), editor.getNotationElementFactory());
					if (isNode) {
						entry = new CreationToolEntry(label, tooltip, factory,
								getIconDescriptor(icon), null);
					}
					else {
						entry = new ConnectionCreationToolEntry(label, tooltip, factory,
								getIconDescriptor(icon), null);
					}
					entry.setId(entryName);
					paletteGroup.add(entry);
				}
				add(paletteGroup);
			}
		}
	}

	private ImageDescriptor getIconDescriptor(String icon) {
		return ImageDescriptor.createFromURL(Plugin.getDefault().getBundle().getEntry("/" + icon));
	}

	@SuppressWarnings("unchecked")
	private void processPaletteContribution(IConfigurationElement configurationElement) {
		IConfigurationElement[] elements;
		elements = configurationElement.getChildren("category");
		for (int i = 0; i < elements.length; i++) {
			categoryList.add(elements[i]);
		}
		elements = configurationElement.getChildren("entry");
		for (int i = 0; i < elements.length; i++) {
			List<IConfigurationElement> list = (List<IConfigurationElement>) categoryMap.get(elements[i].getAttribute("category"));
			if (list == null) {
				list = new ArrayList<IConfigurationElement>();
				categoryMap.put(elements[i].getAttribute("category"), list);
			}
			list.add(elements[i]);
		}
	}

	/*
	 * Creates a palette group for this palette and adds any tools to it.
	 * 
	 * @return a palette group for this palette
	 */
	private PaletteGroup createDefaultControls() {
		PaletteGroup controls = new PaletteGroup("Default Tools");
		controls.setId("org.jbpm.palette.DefaultTools");
		addSelectionTool(controls);
		addMarqueeTool(controls);
		return controls;
	}

	/*
	 * Adds the marquee tool to the palettegroup associated with this palette. A marquee tool
	 * selects multiple objects inside a rectangular area of a Graphical Viewer.
	 * 
	 * @param controls the palettegroup associated with this palette
	 */
	private void addMarqueeTool(PaletteGroup controls) {
		ToolEntry tool = new MarqueeToolEntry();
		tool.setId("org.jbpm.ui.palette.Marquee");
		controls.add(tool);
	}

	/*
	 * Adds the selection tool to the palettegroup associated with this palette. A selection tool is
	 * used to select and manipulate figures.
	 * 
	 * @param controls the palettegroup associated with this palette
	 */
	private void addSelectionTool(PaletteGroup controls) {
		ToolEntry tool = new SelectionToolEntry();
		tool.setId("org.jbpm.ui.palette.Selection");
		controls.add(tool);
		setDefaultEntry(tool);
	}

}
