package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.NodeDefinitionEditor;
import com.tandbergtv.watchpoint.studio.ui.model.DataCollectionManager;
import com.tandbergtv.watchpoint.studio.ui.model.DataCollectionReader;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.ResourceGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageType;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;

/**
 * Composite UI for showing the node definition type section.
 * 
 * @author Imran Naqvi
 * 
 */
public class NodeDefinitionTypeSectionComposite {

	private static final String UNDEFINED = "undefined";
    private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;
    private Composite composite;

    private Label resourceTypeLabel;
	private CCombo resourceTypeCombo;

	private Label nodeDefinitionLabel;
	private CCombo nodeDefinitionCombo;
	
	private NodeDefinition model;
    private EditorPart editor;

    public NodeDefinitionTypeSectionComposite(Composite parent, TabbedPropertySheetWidgetFactory widgetFactory) {
		this.widgetFactory = widgetFactory;
		this.parent = parent;

		createComponents();
	}

    private void createComponents() {
        composite = widgetFactory.createFlatFormComposite(parent);

        createResourceTypeCombo();
        createNodeDefinitionCombo();
    }

    private void createResourceTypeCombo() {
        resourceTypeLabel = widgetFactory.createLabel(composite, "Resource Type");
        FormData data = new FormData();
        data.left = new FormAttachment(0);
        data.top = new FormAttachment(0);
        resourceTypeLabel.setLayoutData(data);

        resourceTypeCombo = widgetFactory.createCCombo(composite, SWT.DROP_DOWN | SWT.SINGLE
                | SWT.V_SCROLL | SWT.H_SCROLL | SWT.READ_ONLY);
        data = new FormData();
        data.left = new FormAttachment(resourceTypeLabel, 0, SWT.RIGHT);
        data.top = new FormAttachment(resourceTypeLabel, 0, SWT.TOP);
        data.width = 330;
        resourceTypeCombo.setLayoutData(data);
        resourceTypeCombo.setVisibleItemCount(10);
        resourceTypeCombo.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                setNodeDefinitionComboInput(model);
            }
        });
    }

    private void createNodeDefinitionCombo() {
        nodeDefinitionLabel = widgetFactory.createLabel(composite, "Message Node");
        FormData data = new FormData();
        data.left = new FormAttachment(resourceTypeLabel, 0, SWT.LEFT);
        data.top = new FormAttachment(resourceTypeCombo, 0, SWT.BOTTOM);
        nodeDefinitionLabel.setLayoutData(data);

        nodeDefinitionCombo = widgetFactory.createCCombo(composite, SWT.DROP_DOWN | SWT.SINGLE
                | SWT.V_SCROLL | SWT.H_SCROLL | SWT.READ_ONLY);
        data = new FormData();
        data.left = new FormAttachment(resourceTypeCombo, 0, SWT.LEFT);
        data.top = new FormAttachment(resourceTypeCombo, 0, SWT.BOTTOM);
        data.width = 330;
        nodeDefinitionCombo.setLayoutData(data);
        nodeDefinitionCombo.setVisibleItemCount(10);
        nodeDefinitionCombo.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                if (nodeDefinitionCombo.getSelectionIndex() >= 0) {
                    if (NodeDefinitionType.MessageNode.equals(model.getNodeType())) {
                        if (nodeDefinitionCombo.getData(nodeDefinitionCombo.getText()) != null) {
                            String uid = (String) nodeDefinitionCombo.getData(nodeDefinitionCombo.getText());
                            NodeDefinition sampleMessageNode = DataCollectionManager.getInstance().getReader()
                                    .getNewMessageNode(uid);
    
                            selectNodeDefinition(sampleMessageNode);
                        }

                    } else {
                        NodeDefinition sampleSuperstate = DataCollectionManager.getInstance().getReader().getNewSuperstate(nodeDefinitionCombo.getText());

                        selectNodeDefinition(sampleSuperstate);
                    }
                }
            }
        });
    }

	private boolean isNodeDefinitionValid(NodeDefinition nodeDefinition) {
	    List<ValidationMessage> messages = ValidationServiceFactory.createFactory().createValidationService()
                .validateNodeDefinition(nodeDefinition);
        for (ValidationMessage message : messages) {
            if (ValidationMessageType.Error.equals(message.getType())) {
                return false;
            }
        }
	    return true;
	}

    private void selectNodeDefinition(NodeDefinition selectedNodeDefinition) {
        if (!isNodeDefinitionValid(selectedNodeDefinition)) {
            MessageDialog.openError(Display.getCurrent().getActiveShell(), "Error",
                    new StringBuilder().append("Message Node ").append(resourceTypeCombo.getText())
                            .append(":").append(nodeDefinitionCombo.getText()).append(" is not valid.").toString());
            return;
        }

        model.setNode(selectedNodeDefinition.getNode());
        model.setDefinitionName(selectedNodeDefinition.getName());

        if (editor instanceof NodeDefinitionEditor) {
            /* Remove timer actions and durations from node since nested node can not create timers */
            model.setDueDate(null);

        } else if (NodeDefinitionType.MessageNode.equals(model.getNodeType())){
            IResourceGroupService groupService = ServiceFactory.createFactory()
                    .createResourceGroupService();

            ResourceType disposableRt = new ResourceType();
            disposableRt.setSystemId(Utility.getSystemIDByMessageUID(selectedNodeDefinition.getUid()));

            com.tandbergtv.watchpoint.studio.dto.ResourceGroup defaultGroup = groupService
                    .getDefaultResourceGroupForResourceType(disposableRt);
            if (defaultGroup != null) {
                ResourceGroup resourceGroup = (ResourceGroup) model.getFactory().createById(
                        SemanticElementConstants.RESOURCEGROUP_SEID);
                resourceGroup.setName(defaultGroup.getName());
                model.setResourceGroup(resourceGroup);
            }

        } else if (NodeDefinitionType.SuperState.equals(model.getNodeType())) {
            for (WPVariable var : selectedNodeDefinition.getVariables()) {
                model.addVariable(var);
            }
        }

        resourceTypeCombo.setEnabled(false);
        nodeDefinitionCombo.setEnabled(false);
    }
    
    public void setInput(NodeDefinition model, EditorPart editor) {
        this.model = model;
        this.editor = editor;

        setResourceTypeComboInput(editor, model);

        if (!resourceTypeCombo.getText().equals("")) {
            setNodeDefinitionComboInput(model);
        }

        if (model.getDefinitionName() != null) {
            resourceTypeCombo.setEnabled(false);
            nodeDefinitionCombo.setEnabled(false);
        }

        if (NodeDefinitionType.SuperState.equals(model.getNodeType())) {
            nodeDefinitionLabel.setText("Super state");
        }
    }

    private void setResourceTypeComboInput(EditorPart editor, NodeDefinition model) {
        NodeDefinitionEditor nodeEditor = Utility.get(editor, NodeDefinitionEditor.class);

        List<String> resourceTypes = getResourceTypes(nodeEditor);
        Collections.sort(resourceTypes);
        resourceTypeCombo.setItems(resourceTypes.toArray(new String[0]));

        if (nodeEditor == null) {
            String resourceTypeText = getResourceTypeSelection(model);
            resourceTypeCombo.setText(resourceTypeText);

        } else {
            resourceTypeCombo.setText(resourceTypeCombo.getItem(0));
            resourceTypeCombo.setEnabled(false);
        }
    }

    private String getResourceTypeSelection(NodeDefinition model) {
        if (model.getDefinitionName() != null) {
            String name = null;
            if (NodeDefinitionType.MessageNode.equals(model.getNodeType())) {
                name = DataCollectionManager.getInstance().getReader().getResourceTypeNameByMessageNodeUid(model.getUid());

            } else {
                NodeGroup group = (NodeGroup) model.getNode();
                for (NodeElement ssElement : group.getNodeElements()) {
                    if (ssElement instanceof NodeDefinition) {
                        NodeDefinition nd = (NodeDefinition) ssElement;
                        name = DataCollectionManager.getInstance().getReader().getResourceTypeNameByMessageNodeUid(nd.getUid());
                        break;
                    }
                }
            }
            return name != null ? name : UNDEFINED;
        } else {
            return "";
        }
    }

    private List<String> getResourceTypes(NodeDefinitionEditor nodeEditor) {
        List<String> result = new ArrayList<String>();
        if (nodeEditor != null) {
            IFileEditorInput editorInput = (IFileEditorInput) nodeEditor.getEditorInput();
            Map<String, String> map = DataCollectionManager.getInstance().getReader().getResourceTypeUrlByNameMap();
            for (String resourceTypeName : map.keySet()) {
                if (map.get(resourceTypeName).contains(editorInput.getFile().getProject().getName() + "/")) {
                    result.add(resourceTypeName);
                    break;
                }
            }

        } else {
            Collection<String> resourceTypeNames = DataCollectionManager.getInstance().getReader().getResourceTypeUrlByNameMap().keySet();
            result.addAll(resourceTypeNames);
        }
        
        return result;
    }

    private void setNodeDefinitionComboInput(NodeDefinition model) {
        if (!resourceTypeCombo.getText().equals(UNDEFINED)) {
            nodeDefinitionCombo.removeAll();
            nodeDefinitionCombo.add("");
            DataCollectionReader reader = DataCollectionManager.getInstance().getReader();
            Map<String, String> uidToName = new TreeMap<String, String>();
            
            if (NodeDefinitionType.MessageNode.equals(model.getNodeType())) {
                for (String uid : reader.getMessageNodeUidsByResourceType(resourceTypeCombo.getText())) {
                    String nodeDefinitionName = reader.extractFileNameFromUrl(reader.getMessageNodeUrlByUidMap().get(uid));
                    
                    uidToName.put(nodeDefinitionName, uid);
                }
                
                for (String key : uidToName.keySet()) {
                	nodeDefinitionCombo.setData(key, uidToName.get(key));
                    nodeDefinitionCombo.add(key);
                }
            } else {
            	List<String> names = reader.getSuperStateNamesByResourceType(resourceTypeCombo.getText());
            	Collections.sort(names);
                for (String superStateName : names) {
                    nodeDefinitionCombo.add(superStateName);
                }
            }
        }
        if (model.getDefinitionName() != null) {
            nodeDefinitionCombo.setText(model.getDefinitionName());
        }
    }
}