/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties.template;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement;
import com.tandbergtv.watchpoint.studio.ui.model.ProtectionKey;
import com.tandbergtv.watchpoint.studio.ui.model.ResourceGroup;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Item wrapper for ProtectionKeys in tables.
 * 
 * @author Imran Naqvi
 * 
 */
public class ProtectionKeyTableItemWrapper implements DisposeListener, SelectionListener,
		FocusListener {

	protected TableItem tableItem;

	protected CCombo nameCombo;

	protected ProtectionKey protectionKey;

	protected PKContainerConfigurationComposite parent;
	
	private AbstractPropertySection parentPropertySection;

	public ProtectionKeyTableItemWrapper(AbstractPropertySection parentSection, PKContainerConfigurationComposite parent, Table table,
			ProtectionKey variable) {
		this.protectionKey = (ProtectionKey) variable;
		this.parent = parent;
		this.parentPropertySection = parentSection;

		initializeTableItem(table);
		initializeNameCombo(table);
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e) {
		cancelEditing();
		if (e.widget == this.nameCombo) {
			String upper = this.nameCombo.getText().toUpperCase();
			this.protectionKey.setName(upper);
			this.nameCombo.setText(upper);
		}
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e) {
		if (e.widget == this.nameCombo) {
			this.tableItem.getParent().setSelection(this.tableItem);
			this.parent.setRemoveButtonEnable(true);
		}
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == nameCombo) {
			protectionKey.setName(nameCombo.getText());
			applyName();
		}
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {

	}

	/**
	 * @see org.eclipse.swt.events.DisposeListener#widgetDisposed(org.eclipse.swt.events.DisposeEvent)
	 */
	public void widgetDisposed(DisposeEvent e) {
		if (nameCombo == null || nameCombo.isDisposed())
			return;

		nameCombo.removeSelectionListener(this);
		nameCombo.removeFocusListener(this);
		nameCombo.dispose();
	}

	/**
	 * Gets the table item.
	 * 
	 * @return table item for this item wrapper
	 */
	public TableItem getTableItem() {
		return tableItem;
	}

	/**
	 * Gets the protection key associated with this wrapper.
	 * 
	 * @return variable
	 */
	public ProtectionKey getProtectionKey() {
		return protectionKey;
	}

	public void editCell(int column) {
		if (column == 1) {
			editName();
		}
	}

	/**
	 * Initializes the table item.
	 * 
	 * @param table
	 */
	protected void initializeTableItem(Table table) {
		tableItem = new TableItem(table, SWT.NONE);
		tableItem.addDisposeListener(this);
		tableItem.setData(this);
	}

	/**
	 * Initializes the mapped name text box.
	 * 
	 * @param table
	 *            the table in which the text box is being added
	 */
	protected void initializeNameCombo(Table table) {
		nameCombo = createCombo(table, 1);
		nameCombo.setTextLimit(50);
		FormData data = new FormData();
		data.width = 85;
		nameCombo.setLayoutData(data);
		nameCombo.addSelectionListener(this);
		nameCombo.addFocusListener(this);
		nameCombo.setText(getName());
		nameCombo.setEditable(true);
		fillProtectionKeyCombo();
		if (this.nameCombo.getSelectionIndex() == -1)
			this.nameCombo.setText(getName());
	}

	/**
	 * Creates a combo box in the given table in the column specified by the given index.
	 * 
	 * @param table
	 *            the table in which the combo box is being created
	 * @param index
	 *            index of the column in which to create the combo box
	 * @return the combo box created
	 */
	protected CCombo createCombo(Table table, int index) {
		TableEditor textEditor = new TableEditor(table);
		CCombo result = new CCombo(table, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL
				| SWT.READ_ONLY);
		result.pack();
		textEditor.horizontalAlignment = SWT.LEFT;
		textEditor.grabHorizontal = true;

		textEditor.setEditor(result, tableItem, index);
		return result;
	}

	/**
	 * Edits the protection key name text
	 */
	protected void editName() {
		nameCombo.setVisible(true);
		nameCombo.setText(getName());
		nameCombo.setFocus();
	}

	protected void cancelEditing() {
		applyName();
	}

	/**
	 * Apply protection key name changes.
	 */
	protected void applyName() {
		if (this.nameCombo == null)
			return;

		this.tableItem.setText(1, getName());
	}

	/**
	 * Get the protection key name.
	 * 
	 * @return protection key name
	 */
	protected String getName() {
		return protectionKey.getName() == null ? "" : protectionKey.getName();
	}

	private void fillProtectionKeyCombo() {
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		int index = 0;
		for (String groupName : getResourceGroups()) {
			com.tandbergtv.watchpoint.studio.dto.ProtectionKey protectionKey = service.getProtectionKeyByResourceGroupName(groupName);
			
			if (protectionKey != null) {
				String pk = protectionKey.getName();
				if(arrayContains(this.nameCombo.getItems(), pk))
					continue;
				this.nameCombo.add(pk);
				if (pk.equals(this.protectionKey.getName()))
					this.nameCombo.select(index);
				index++;
			}
		}
	}

	private Set<String> getResourceGroups() {
		Set<String> resourceGroups = new HashSet<String>();
		IWorkbenchPart part = this.parentPropertySection.getPart();
		NodeElementContainer nodeContainer = Utility.getNodeElementContainer(part);
		
		if (nodeContainer == null)
			throw new UIException(
					"Current editor does not have a node element container associated with it.");
		for (NodeElement nodeElement : nodeContainer.getNodeElements()) {
			if (!(nodeElement instanceof IResourceGroupElement))
				continue;
			ResourceGroup resourceGroup = ((IResourceGroupElement) nodeElement).getResourceGroup();
			if (resourceGroup != null && resourceGroup.getName() != null
					&& !resourceGroup.getName().trim().equals(""))
				resourceGroups.add(((IResourceGroupElement) nodeElement).getResourceGroup()
						.getName());
		}
		return resourceGroups;
	}
	
	private boolean arrayContains(String[] strArray, String val){
		for(String str : strArray)
			if(str.equals(val))
				return true;
		return false;
	}
}
