/**
 * ResourceTypeCreateCommand.java
 * Created May 13, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.resource;

import static javax.xml.xpath.XPathConstants.NODESET;

import java.util.HashSet;
import java.util.Set;

import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.dto.IPersistable;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.SynchronizationException;
import com.tandbergtv.watchpoint.studio.ui.sync.command.AbstractDiffCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * Creates a resource type
 * 
 * @author Sahil Verma
 */
public class ResourceTypeCreateCommand extends AbstractDiffCommand {

	protected ResourceType model;

	/**
	 * Creates the command
	 * 
	 * @param name
	 * @param diff
	 * @param context
	 */
	public ResourceTypeCreateCommand(String name, IDiff diff, ISynchronizationContext context) {
		super(name, diff, context);
		this.model = (ResourceType) diff.getModel();
	}

	/**
	 * {@inheritDoc}
	 */
	public void execute() {
		model.setId(IPersistable.DEFAULT_ID);
		model.setDirty(true);

		context.createResourceType(model);

		Set<NodeDefinitionDTO> superstates = new HashSet<NodeDefinitionDTO>();

		for (NodeDefinitionDTO node : model.getNodes()) {
			if (node.getType() == NodeDefinitionType.SuperState) {
				/* Defer superstate creation */
				superstates.add(node);
				continue;
			}

			updateNodeDefinitionXml(node);
			context.createNode(node);
		}

		for (NodeDefinitionDTO superstate : superstates) {
			updateSuperStateXml(superstate);
			context.createNode(superstate);
		}
	}

	/**
	 * Updates the node definition xml with the resource type id
	 * 
	 * @param node
	 */
	protected void updateNodeDefinitionXml(NodeDefinitionDTO node) {
		Document document = Utility.getDocumentFromString(node.getXml());
		Element e = document.getDocumentElement();

		e.setAttribute("resourceType", String.valueOf(node.getResourceType().getId()));

		try {
			String xml = FileUtil.convertDocument(document);
			node.setXml(xml);
		} catch (TransformerException ex) {
			throw new SynchronizationException("Failed to convert node XML", ex);
		}
	}

	/**
	 * Updates the node definition xml with the resource type id
	 * 
	 * @param superstate
	 */
	protected void updateSuperStateXml(NodeDefinitionDTO superstate) {
		try {
			Document document = Utility.getDocumentFromString(superstate.getXml());
			Element e = document.getDocumentElement();

			e.setAttribute("resourceType", String.valueOf(superstate.getResourceType().getId()));

			XPath xpath = XPathFactory.newInstance().newXPath();
			NodeList nodelist = (NodeList) xpath.evaluate("//super-state/nodeDefinition", e, NODESET);

			for (int i = 0; i < nodelist.getLength(); i++) {
				Element dom = (Element) nodelist.item(i);
				String name = dom.getAttribute("definitionName");

				dom.setAttribute("resourceType", String.valueOf(superstate.getResourceType().getId()));

				for (NodeDefinitionDTO child : superstate.getChildren()) {
					if (child.getName().equals(name)) {
						dom.setAttribute("id", String.valueOf(child.getId()));
						break;
					}
				}
			}

			String xml = FileUtil.convertDocument(document);

			superstate.setXml(xml);
		} catch (XPathException e) {
			throw new SynchronizationException("Failed to convert node XML", e);
		} catch (TransformerException e) {
			throw new SynchronizationException("Failed to convert node XML", e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "Resource Type " + model + " " + diff.getKind();
	}
}
