/*
 * Created on Oct 30, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.view;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Event;
import org.eclipse.ui.part.ViewPart;

import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * The Action triggered for the WatchPoint Views
 * 
 * @author Vijay Silva
 */
public abstract class ViewAction extends Action
{
	protected ViewPart viewPart;

	/**
	 * Default Constructor. Creates a new action with no text and no image.
	 * 
	 * @param viewPart
	 *            The View Part that triggers this action.
	 */
	public ViewAction(ViewPart viewPart)
	{
		this.viewPart = viewPart;
	}

	/**
	 * Creates a new action with the given text and no image.
	 * 
	 * @param viewPart
	 *            The View Part that triggers this action.
	 * @param text
	 *            the action text
	 * 
	 * @see org.eclipse.jface.action.Action#Action(String)
	 */
	public ViewAction(ViewPart viewPart, String text)
	{
		super(text);
		this.viewPart = viewPart;
	}

	/**
	 * Creates a new action with the given text and image.
	 * 
	 * @param viewPart
	 *            The View Part that triggers this action.
	 * @param text
	 *            the action text
	 * @param image
	 *            the action image
	 * 
	 * @see org.eclipse.jface.action.Action#Action(String, ImageDescriptor)
	 */
	public ViewAction(ViewPart viewPart, String text, ImageDescriptor image)
	{
		super(text, image);
		this.viewPart = viewPart;
	}

	/**
	 * Creates a new action with the given text and style.
	 * 
	 * @param viewPart
	 *            The View Part that triggers this action.
	 * @param text
	 *            the action text
	 * @param style
	 *            The action style
	 * 
	 * @see org.eclipse.jface.action.Action#Action(String, int)
	 */
	public ViewAction(ViewPart viewPart, String text, int style)
	{
		super(text, style);
		this.viewPart = viewPart;
	}

	/**
	 * @see org.eclipse.jface.action.Action#run()
	 */
	@Override
	public final void run()
	{
		try
		{
			this.performAction();
		}
		catch (Exception ex)
		{
			this.handleActionError(ex);
		}
	}

	/**
	 * @see org.eclipse.jface.action.Action#runWithEvent(org.eclipse.swt.widgets.Event)
	 */
	@Override
	public final void runWithEvent(Event event)
	{
		try
		{
			this.performActionForEvent(event);
		}
		catch (Exception ex)
		{
			this.handleActionError(ex);
		}
	}

	/**
	 * Performs the action.
	 * 
	 * @see org.eclipse.jface.action.IAction#run()
	 */
	protected void performAction()
	{
	}

	/**
	 * Performs the action caused by the given event. The default implementation ignores the event
	 * and calls performAction().
	 * 
	 * @param event
	 *            the SWT event which triggered this action being run
	 * 
	 * @see org.eclipse.jface.action.IAction#runWithEvent(Event)
	 */
	protected void performActionForEvent(Event event)
	{
		this.performAction();
	}

	/**
	 * Handle the exception that was caused when performing the action.
	 * 
	 * @param error
	 *            The Exception that occurred when performing the action.
	 */
	protected void handleActionError(Exception error)
	{
		Utility.reportError("Error", error);
	}
}
