/**
 * ImportWizardListPage.java
 * Created Feb 10, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.imports;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.DTONameDecorator;
import com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWizardPage;

/**
 * Shows list of available plugins. The plugins that need to be imported can be selected.
 * 
 * @author Sahil Verma
 */
public class ImportWizardListPage extends AbstractWizardPage {

	private ImportWizardFirstPage page;
	
	private Label counterLabel;
	
	private Collection<? extends IWatchPointDTO> models;
	
	private CheckboxTableViewer viewer;
	private boolean overwriteExistingProjects = true;
	
	public void setOverwriteExistingProjects(boolean overwriteExistingProjects) {
		this.overwriteExistingProjects = overwriteExistingProjects;
	}

	public boolean isOverwriteExistingProjects() {
		return overwriteExistingProjects;
	}

	/**
	 * @param pageName
	 */
	public ImportWizardListPage(String pageName, ImportWizardFirstPage page, IStructuredSelection selection) {
		super(pageName);
		setTitle("Selection");
		setMessage("Select items to import");
		this.page = page;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void saveSettings(IDialogSettings settings) {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		
		if (visible) {
			String location = page.getBaseDirectory();
			this.models = page.loadModel(location);
			computeModelsToImport();
			pageChanged();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void createControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		
		layout.numColumns = 1;
		layout.horizontalSpacing = 5;
		layout.verticalSpacing = 0;
		
		container.setLayout(layout);
		container.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		createTablePart(container);
		createButtons(container);
		
		initialize();
		setControl(container);
		Dialog.applyDialogFont(container);
	}
	
	/**
	 * Creates the table part
	 * 
	 * @param container
	 */
	private void createTablePart(Composite parent) {
		Composite container = new Composite(parent, SWT.NONE);
		container.setLayout(new GridLayout());
		container.setLayoutData(new GridData(GridData.FILL_BOTH));

		createImportList(container);
		
		this.counterLabel = new Label(container, SWT.NONE);
		this.counterLabel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
	}

	/**
	 * Sets up the table containing all the available model items
	 * 
	 * @param container
	 */
	private void createImportList(Composite parent) {
		Composite container = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		container.setLayoutData(new GridData(GridData.FILL_BOTH));

		this.viewer = CheckboxTableViewer.newCheckList(container, SWT.BORDER);
		
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.widthHint = 225;
		
		this.viewer.getTable().setLayoutData(gd);
		this.viewer.setContentProvider(new ItemContentProvider());
		this.viewer.setLabelProvider(new ItemLabelProvider());
		this.viewer.setComparator(new ItemViewerComparator());
		this.viewer.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent ev) {
				pageChanged();
			}
		});
		
		parent.pack();
	}

	/**
	 * Creates the buttons used for selecting and deselecting
	 * 
	 * @param container
	 */
	private void createButtons(Composite container) {
		Composite buttonComp = new Composite(container, SWT.NONE);
		GridLayout layout = new GridLayout(2, true);
		
		layout.marginHeight = 0;
		layout.marginBottom = 10;
		layout.verticalSpacing = 0;
		layout.marginRight = 4;
		
		buttonComp.setLayout(layout);
		buttonComp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Button selectAll = new Button(buttonComp, SWT.PUSH);
		
		selectAll.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		selectAll.setText("Select All");
		selectAll.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewer.setAllChecked(true);
				pageChanged();
			}
		});
		
		Button deselectAll = new Button(buttonComp, SWT.PUSH);
		deselectAll.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		deselectAll.setText("Deselect All");
		deselectAll.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewer.setAllChecked(false);
				pageChanged();
			}
		});
		
		Button overwriteProjects = new Button(buttonComp, SWT.CHECK );
		GridData test = new GridData(GridData.FILL_HORIZONTAL);
		test.verticalIndent = 10;
		overwriteProjects.setLayoutData(test);
		overwriteProjects.setText("Overwrite existing resource type projects");
		overwriteProjects.setSelection(true);
		overwriteProjects.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				overwriteExistingProjects = ((Button)e.widget).getSelection();
			}
			public void widgetDefaultSelected(SelectionEvent e) {
				overwriteExistingProjects = ((Button)e.widget).getSelection();
			}
		});
	}
	
	protected void initialize() {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isPageComplete() {
		return this.viewer.getCheckedElements().length > 0;
	}
	
	/**
	 * Returns the list of selected items
	 * 
	 * @return
	 */
	public Collection<? extends IWatchPointDTO> getSelectedModels() {
		Object [] items = this.viewer.getCheckedElements();
		Collection<IWatchPointDTO> collection = new HashSet<IWatchPointDTO>();
		
		for (int i = 0; i < items.length; i++)
			collection.add((IWatchPointDTO) items[i]);
		
		return collection;
	}
	
	protected void pageChanged() {
		updateCount();
		setPageComplete(isPageComplete());
	}
	
	private void computeModelsToImport() {
		this.viewer.getTable().removeAll();
		
		this.viewer.add(this.models.toArray());
	}

	private void updateCount() {
		String number = Integer.toString(this.viewer.getCheckedElements().length);
		String totalNumber = Integer.toString(this.viewer.getTable().getItemCount());
		this.counterLabel.setText(NLS.bind("{0} of {1} selected", (new String[] {number, totalNumber})));
		this.counterLabel.getParent().layout();
	}
	
	/**
	 * Why do I need this?
	 */
	class ItemContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object arg0) {
			return models.toArray();
		}

		public void dispose() {
		}

		public void inputChanged(Viewer arg0, Object arg1, Object arg2) {
		}
	}
	
	/**
	 * Provides rendering information - text and icon - for each table item
	 */
	class ItemLabelProvider extends LabelProvider implements ITableLabelProvider {
		private ImageRegistry registry;
		private static final String RT_ICON = "Resource Type Icon";
		private static final String RG_ICON = "Resource Group Icon";
		private static final String TEMPLATE_ICON = "Template Icon";
		
		ItemLabelProvider() {
			super();
			this.registry = new ImageRegistry();
			this.registry.put(RT_ICON, Utility.getImageDescriptor("resource_type.png"));
			this.registry.put(RG_ICON, Utility.getImageDescriptor("resource_group.png"));
			this.registry.put(TEMPLATE_ICON, Utility.getImageDescriptor("template.png"));
		}

		public Image getColumnImage(Object obj, int arg1) {
			if (obj instanceof ResourceType)
				return this.registry.get(RT_ICON);
			else if (obj instanceof ResourceGroup)
				return this.registry.get(RG_ICON);
			else if (obj instanceof WorkflowTemplateDTO) 
				return this.registry.get(TEMPLATE_ICON);
			
			throw new IllegalArgumentException();
		}
		
		public String getColumnText(Object obj, int arg1) {
			if (obj instanceof ResourceType) {
				ResourceType item = ResourceType.class.cast(obj);
				return item.getName() + " [" + item.getSystemId() + "] - v" + item.getVersion();
			} else if (obj instanceof ResourceGroup) {
				ResourceGroup item = ResourceGroup.class.cast(obj);
				return item.getName() + " [" + item.getSystemId() + "] - v" + item.getVersion();
			} else if (obj instanceof WorkflowTemplateDTO) {
				WorkflowTemplateDTO dto = WorkflowTemplateDTO.class.cast(obj);
				return dto.getName() + " - v" + dto.getVersion();
			}
			
			throw new IllegalArgumentException();
		}
	}
	
	/**
	 * Used by the table viewer to compare two table items, ensures that the list is sorted by name
	 */
	class ItemViewerComparator extends ViewerComparator {
		@SuppressWarnings("unchecked")
		@Override
		public int compare(Viewer viewer, Object e1, Object e2) {
			String name1 = getName(e1);
			String name2 = getName(e2);
			
			return getComparator().compare(name1, name2);
		}
		
		private String getName(Object obj) {
			return new DTONameDecorator((IWatchPointDTO) obj).getName();
		}
	}
}
