package com.tandbergtv.watchpoint.studio.util;

import java.util.Arrays;
import java.util.List;

import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.jpdl.model.Controller;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.Task;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Auto map the node definition variables to template variables
 * @author Patrik
 *
 */
public final class MapNodeVariablesUtil {
	public static final String VARIABLE_DEFAULT_TYPE = "STRING";
	
	/**
	 * Map the node definition variables with the start node variables. <br>
	 * If a node definition variable is not defined in the start node, 
	 * this method creates the variable and map it.
	 * 
	 * @param template The template which variables have to be created/mapped
	 * @param nodeDefinition The node definition which variables have to be mapped
	 */
	public static void mapNodeVariables(WorkflowTemplate template, NodeDefinition nodeDefinition){
		
		StartState startState = template.getStartState();
		//  Check if the task and controller is null. This can happen if the template was just created.
        if(startState.getTask() == null){
            startState.setTask((Task) startState.getFactory().createById(
                    SemanticElementConstants.TASK_SEID));
            Task task = startState.getTask();
            if(task.getController() == null){
                task.setController((Controller) task.getFactory().createById(
                        SemanticElementConstants.CONTROLLER_SEID));
            }
		}

		List<WPVariable> variablesToMap = nodeDefinition.getVariables();
		for (WPVariable variableToMap : variablesToMap) {
            if (variableToMap.getName() == null || variableToMap.getName().isEmpty()) {
                Variable variable = findOrCreate(template, variableToMap);
                variableToMap.setName(variable.getName());
            }
		}
	}

	private static Variable findOrCreate(WorkflowTemplate template, WPVariable variableToMap) {
        //  Get start node variables
        List<Variable> startVariables = Arrays.asList(template.getStartState().getTask().getController().getVariables());

        //  Check if the variable exists in start node
        for (Variable variable : startVariables) {
            if (variable.getName().equalsIgnoreCase(variableToMap.getMappedName())) {
                return variable;
            }
        }
        return createVariable(template, variableToMap);
    }

    private static WPVariable createVariable(WorkflowTemplate template, WPVariable wpVariable){
		SemanticElementFactory factory = new SemanticElementFactory(Utility.TEMPLATE_EDITOR_ID);
		// Creates the variable element
		WPVariable variable = (WPVariable)factory.createById(SemanticElementConstants.VARIABLE_SEID);
		variable.setMappedName( createVariableDisplayName(wpVariable.getMappedName()) );
		variable.setName(wpVariable.getMappedName());
		variable.setAccess("read,write");
		variable.setType(getNewVariableType(wpVariable));
		// Add the variable to the start node
		template.getStartState().getTask().getController().addVariable(variable);

		return variable;
	}
	
	private static String getNewVariableType(WPVariable wpVariable){
		String newVariableType = "";
		if(wpVariable.getType() == null || wpVariable.getType().isEmpty()){
			newVariableType = VARIABLE_DEFAULT_TYPE;
		}else{
			newVariableType = wpVariable.getType();
		}
		return newVariableType;
	}
	
	private static String createVariableDisplayName(String baseName){
		String[] nameParts = baseName.split("(?=\\p{Upper})");
		
		String displayName = "";
		for (String string : nameParts) {
			if(string.trim().length() == 0)
				continue;
			if(Character.isLowerCase(string.charAt(0)) ){
				string = Character.toUpperCase(string.charAt(0)) + string.substring(1);
			}
			displayName = displayName + string.trim() + " ";
		}
		return displayName.trim();
	}
}
