package com.tandbergtv.watchpoint.studio.validation.rules.expression;

import java.util.Hashtable;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.CompilationUnit;

import com.tandbergtv.watchpoint.studio.ui.editor.expression.ExpressionUtil;
import com.tandbergtv.watchpoint.studio.ui.editor.expression.SnippetCompletionProcessor;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.validation.impl.UsedVariablesCollectorVisitor;

/**
 * 		Validates the expressions.
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class VariableExpressionValidator {
	
	// The Logger
	private static final Logger logger = Logger.getLogger(VariableExpressionValidator.class);
	
	private String contextVariablesScript;

	private IProject project;

    private DataType returnType = DataType.OBJECT;

	private IProblem[] problems;

    /**
	 * 		Generates a temporary Java class based 
	 * on the input.
	 * 		The structure of the generated Java class
	 * will be similar to:
	 * 
	 *   	public class ExpressionSnippet {
	 *   		[String] [templateVar1];
	 *   		[int] [templateVar2];		
	 *   		
	 *   		public [Boolean] get() {
	 *   			return [userExpression];
	 * 		}
	 * Where:
	 * 		[String] is the variable type
	 * 		[templateVar1] is the variable name
	 * 
	 * 		[Boolean] is the target variable type
	 * 		[userExpression] is the user inputed expression.
	 * @param expression 
	 */
	protected String generateTempClassSnippet(String expression) {
		StringBuilder classSnippet = new StringBuilder();
		
		StringBuilder defaultImports = new StringBuilder();
		for (String defaultImport : ExpressionUtil.DEFAULT_IMPORTS) {
			defaultImports.append("import ");
			defaultImports.append(defaultImport);
			defaultImports.append(";");
		}
		
		final String classStr = "public class ExpressionSnippet { ";
		
		classSnippet.append(defaultImports);
		classSnippet.append(classStr);
		
		/* Declares the method with the return type the same as the 
		 * variable type to force the Java compiler to ensure that
		 * the expression returns an object of the same type
		 * as the target variable.
		 */
        final String methodStr = new StringBuilder().append(" public " + returnType.getJavaMatchingType())
                .append(" get() { try {").append(SnippetCompletionProcessor.DEFAULT_VARS).append(contextVariablesScript).toString();

		classSnippet.append(methodStr);
		
		/* Example of contents the Beanshell runtime should allow:
		 * - Simple statement with implicit return:
		 * 		parentFolder.toLowerCase() + "/" + fileName.toLowerCase()
		 * - Explicit return:
		 * 		return new String(parentFolder + fileName)
		 * - Multiple returns (conditional):
		 * 		if (parentFolder != null) { return parentFolder + fileName; } else { return fileName; } 
		 *
		 */
		if (!expression.contains("return")) {
			expression = "return " + expression;
		}
		expression = expression + "; } catch (Throwable t) { return null; }";
		classSnippet.append(expression);
		
		// closing method body and class body 
		classSnippet.append("}}");
		
		return classSnippet.toString();
	}
	
	/**
	 * 		Validates the Expression using the AST Parser
	 * @return 
	 */
	public Set<String> validateExpression(String expression) {
		ASTParser parser = ASTParser.newParser(AST.JLS3);
		parser.setKind(ASTParser.K_COMPILATION_UNIT);
		parser.setResolveBindings(true);
		if (project != null) {
			IJavaProject javaProject = JavaCore.create(project);
			parser.setProject(javaProject);
			
			try {
				IClasspathEntry[] entries = javaProject.getResolvedClasspath(true);
				for (IClasspathEntry entry : entries) {
					if (entry.getEntryKind() == IClasspathEntry.CPE_SOURCE) {
						// Must call this method for the parser to resolve the bindings...
						parser.setUnitName(entry.getPath().toPortableString() + "/ExpressionSnippet.java");
						break;
					}
				}
			} catch (JavaModelException e) {
				logger.error("Error reading project classpath.", e);
			}
			
		}
		
		parser.setSource(generateTempClassSnippet(expression).toCharArray());
        Hashtable options = JavaCore.getOptions();
        options.put(JavaCore.COMPILER_PB_UNUSED_IMPORT, JavaCore.IGNORE);
        options.put(JavaCore.COMPILER_PB_UNUSED_LOCAL, JavaCore.IGNORE);
        parser.setCompilerOptions(options);
		
		// Compiles the code and shows the error in the field 
		CompilationUnit result = (CompilationUnit) parser.createAST(null);

        UsedVariablesCollectorVisitor astVisitor = new UsedVariablesCollectorVisitor();
        result.accept(astVisitor);

		problems = result.getProblems();

		return astVisitor.getUsedVariables();
	}
	
	/**
	 * 		Returns if errors were found after the last invocation of the validateExpression method.
	 * 
	 * @return
	 */
	public boolean hasErrors() {
		for (IProblem problem : problems) {
			if (problem.isError()) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * 		Returns the problems found as a result of the last invocation of the validateExpression method.
	 * 
	 * @return
	 */
	public IProblem[] getProblems() {
		return problems;
	}

	/**
	 * 		Sets the context variables to be defined in the expression context if any.
	 * 
	 * @param contextVariablesScript
	 */
	public void setContextVariables(String contextVariablesScript) {
		this.contextVariablesScript = contextVariablesScript;
	}
	
	/**
	 * This is used specially for defining if the expression actually resolves to the same type
	 * that the variable is expecting.
	 */
	public void setReturnType(DataType returnType) {
		this.returnType = returnType;
	}

	public void setProject(IProject project) {
	    this.project = project;
	}
}
