package com.tandbergtv.watchpoint.studio.validation.rules.resourcegroup;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Validation Rule that ensures Resource Group properties are 
 * 1) not null and not blank
 * 2) of allowed text length 
 * 
 * @author Raj Prakash
 */
public class RGBasicPropertiesRule extends ValidationRule<ResourceGroup>
{
	/**
	 * Ensures that Resource Group Name properties are 1) not null and not blank 2) of allowed text length. 
	 * 
	 * @param target
	 *            The Object to validate
	 * 
	 * @return The list of Validation Messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ResourceGroup target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		//name
		validateNotNullAndNotBlank(target, target.getName(), ValidationMessageCode.RESOURCEGROUP_NAME_BLANK, messages);
		validateTextMaxLength(target.getName(), 50, target,
							  ValidationMessageCode.RESOURCEGROUP_NAME_TOO_LARGE, messages);
		
		//resource type
		String systemID = target.getSystemId();
		//if resource type is null or id is not set
		if(systemID == null || systemID.equals(String.valueOf(ResourceType.DEFAULT_ID)))
		{
		    ValidationMessageAdder.getInstance().addValidationMessage(messages, target, ValidationMessageCode.RESOURCEGROUP_RESOURCETYPE_NOTSET);
		} else {
			IResourceTypeService rtService = ServiceFactory.createFactory().createResourceTypeService();
			ResourceType resourceType = rtService.getResourceTypeBySystemId(systemID);
			if (resourceType == null) {
				List<String> params = Arrays.asList(new String[] {systemID});
                ValidationMessageAdder.getInstance().addValidationMessage(messages, target, ValidationMessageCode.RESOURCEGROUP_RESOURCETYPE_NOT_FOUND, params);
			} else {
				if (resourceType.isHasErrors()) {
					List<String> params = Arrays.asList(new String[] {resourceType.getName()});
					ValidationMessageAdder.getInstance().addValidationMessage(messages, target, ValidationMessageCode.RESOURCEGROUP_RESOURCETYPE_INVALID, params);
				}
			}
		}
		
		//functional type
		validateTextMaxLength(target.getFunctionalType(), 256, target,
							  ValidationMessageCode.RESOURCEGROUP_FUNTYPE_TOO_LARGE, messages);
		
		//allocation strategy
		validateTextMaxLength(target.getAllocationStrategyClassName(), 256, target,
							  ValidationMessageCode.RESOURCEGROUP_ALLOCSTRAT_TOO_LARGE, messages);
		
		//image
		validateTextMaxLength(target.getImageFileName(), 150, target,
							  ValidationMessageCode.RESOURCEGROUP_IMAGEFILENAME_TOO_LARGE, messages);
				
		//protection key
		if(target.getProtectionKey() == null)
		{
		    ValidationMessageAdder.getInstance().addValidationMessage(messages, target, ValidationMessageCode.RESOURCEGROUP_PROTKEY_BLANK);
		}
		else if(target.getProtectionKey().getId() == ProtectionKey.DEFAULT_ID)
		{
			validateNotNullAndNotBlank(target, target.getProtectionKey().getName(),
									   ValidationMessageCode.RESOURCEGROUP_PROTKEY_BLANK, messages);
			validateTextMaxLength(target.getProtectionKey().getName(), 50, target,
								  ValidationMessageCode.RESOURCEGROUP_PROTKEY_TOO_LARGE, messages);
		}
		
		// duplication
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		List<ResourceGroup> homonymous = service.getResourceGroupByName(target.getName());
		if (homonymous.size() > 1) {
		    ValidationMessageAdder.getInstance().addValidationMessage(messages, target, ValidationMessageCode.RESOURCEGROUP_NAME_NOT_UNIQUE);
		}
			
		return messages;
	}
	
    private void validateTextMaxLength(String text, int maxSize, ResourceGroup target, ValidationMessageCode errorCode,
            List<ValidationMessage> messages) {
        if (text != null && text.length() > maxSize) {
            ValidationMessageAdder.getInstance().addValidationMessage(messages, target, errorCode);
        }
    }
	
	private void validateNotNullAndNotBlank(ResourceGroup target, String property,
										 ValidationMessageCode errorCode,
										 List<ValidationMessage> messages)
	{
		if(isNullOrBlank(property))
		{
		    ValidationMessageAdder.getInstance().addValidationMessage(messages, target, errorCode);
		}
	}
	
	private boolean isNullOrBlank(String str)
	{
		if(str == null || str.trim().length() == 0)
			return true;
		else
			return false;
	}
}
