/**
 * Loop.java
 * Created Oct 27, 2011
 */
package com.tandbergtv.watchpoint.studio.debugger.core.graph;

import static java.lang.Boolean.parseBoolean;
import static org.jbpm.jpdl.el.impl.JbpmExpressionEvaluator.evaluate;

import java.util.List;

import org.dom4j.Element;
import org.jbpm.JbpmException;
import org.jbpm.graph.def.Action;
import org.jbpm.graph.def.Event;
import org.jbpm.graph.def.Node;
import org.jbpm.graph.def.SuperState;
import org.jbpm.graph.def.Transition;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.jpdl.xml.JpdlXmlReader;

/**
 * A node that supports iteration over child nodes until a specified condition is met. Supports
 * basic for, while-condition and for-each idioms. 
 * 
 * @author Sahil Verma
 */
public class Loop2 extends SuperState {

	private static final long serialVersionUID = -2696666366298339153L;
	
	private String variable;
	
	private String index;
	
	private int initialValue;

	private String expression;

	public Loop2() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void read(Element element, JpdlXmlReader jpdlReader) {
		super.read(element, jpdlReader);
		
		variable = element.attributeValue("variable"); // FIXME Check type == LIST
		index = element.attributeValue("index");
		
		if (index == null || index.isEmpty())
			jpdlReader.addError("Node " + name + " does not have an index variable");
		
		initialValue = Integer.valueOf(element.attributeValue("initialValue", "0"));
		expression = element.attributeValue("expression");
		
		if (nodes == null || nodes.isEmpty())
			jpdlReader.addError("Node " + name + " does not have child nodes");
	
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Transition getDefaultLeavingTransition() {
		if (superState != null && superState.getParent() instanceof Loop2) {
			List<?> transitions = superState.getLeavingTransitions();
			
			// This node's parent NodeGroup is the last child of an outer loop, need a fake transition
			if (transitions == null || transitions.isEmpty())
				return createLeavingTransition();
		}
		
		return super.getDefaultLeavingTransition();
	}

	public Node first() {
		for (Object obj : this.nodes) {
			Node node = (Node) obj;

			if (node.getArrivingTransitions() == null || node.getArrivingTransitions().isEmpty())
				return node;
		}
		
		throw new JbpmException(this.name + " has no nodes to delegate execution");
	}
	
	public Node last() {
		for (Object obj : this.nodes) {
			Node node = (Node) obj;

			if (node.getLeavingTransitions() == null || node.getLeavingTransitions().isEmpty())
				return node;
		}
		
		throw new JbpmException(this.name + " has no nodes to finish execution");
	}

	public void incrementIndex(ExecutionContext context) {
		Integer i = (Integer) context.getVariable(index);

		context.setVariable(index, i + 1);
	}
	
	public boolean evaluateExpression(ExecutionContext context) {
		return parseBoolean(evaluate(expression, context).toString());
	}
	
	protected Transition getArrivingTransition(Node node) {
		if (node.getArrivingTransitions() != null && !node.getArrivingTransitions().isEmpty())
			return (Transition) node.getArrivingTransitions().iterator().next();
		
		final Node target = (node instanceof SuperState2) ? SuperState2.class.cast(node).first() : node;
		
		Transition transition = new Transition() {
			private static final long serialVersionUID = 1L;
			@Override
			public void take(ExecutionContext context) {
				fireEvent(Event.EVENTTYPE_TRANSITION, context);
			}
		};
		
		transition.setTo(target);
		transition.setProcessDefinition(processDefinition);
		
		/* Avoid attaching event if this loop is a child of a NodeGroup, that case is already taken care of */
		if (!(getSuperState() instanceof SuperState2))
			addEvent(transition, "resource-request");
		
		return transition;
	}
	
	protected Transition createLeavingTransition() {
		Transition transition = new Transition() {
			private static final long serialVersionUID = 1L;
			@Override
			public void take(ExecutionContext context) {
				/* As always, be sure to explicitly leave parent NodeGroup to cleanup state etc. */
				superState.leave(context);
				fireEvent(Event.EVENTTYPE_TRANSITION, context);
			}
		};
		
		transition.setFrom(this);
		transition.setProcessDefinition(processDefinition);
		
		return transition;
	}
	
	protected void addEvent(Transition transition, String action) {
		// FIXME Read delegation from configuration, let delegation do this magic trick
		Event event = new Event(transition, Event.EVENTTYPE_TRANSITION);
		Action a = transition.getProcessDefinition().getAction(action);
		
		event.addAction(new Action(a.getActionDelegation())); // Be sure to attach unpersisted entity
		transition.addEvent(event);
	}

	public String getIndex() {
		return index;
	}

	public int getInitialValue() {
		return initialValue;
	}

	public String getExpression() {
		return expression;
	}

}
