/**
 * ProcessEventListener.java
 * Created Aug 23, 2012
 */
package com.tandbergtv.watchpoint.studio.debugger.runtime.exe;

import static com.tandbergtv.watchpoint.studio.debugger.launching.TemplateLauncherConstants.CONFIG;
import static com.tandbergtv.watchpoint.studio.debugger.runtime.parser.RuntimeTemplateParser.parse;
import static com.tandbergtv.workflow.core.event.ColleaguePriority.LOW;

import java.io.File;
import java.util.Map;

import org.apache.log4j.Logger;

import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.interpreterwrapper.EventDispatcher;
import com.tandbergtv.workflow.core.WorkflowProcess;
import com.tandbergtv.workflow.core.event.ColleaguePriority;
import com.tandbergtv.workflow.core.event.IColleague;
import com.tandbergtv.workflow.core.event.WorkflowEvent;
import com.tandbergtv.workflow.driver.event.WorkflowProcessEvent;

/**
 * Listens for process-related events of interest from the runtime and generates notifications
 * 
 * @author Sahil Verma
 */
public class ProcessEventListener implements IColleague {
	
	private static final Logger logger = Logger.getLogger(ProcessEventListener.class);
    private EventDispatcher eventDispatcher;

	public ProcessEventListener(EventDispatcher eventDispatcher) {
		super();
		this.eventDispatcher = eventDispatcher;
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	public String getColleagueName() {
		return "debugger-events";
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	public ColleaguePriority getColleaguePriority() {
		return LOW;
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	public void receive(WorkflowEvent event) {
		if (!(event instanceof WorkflowProcessEvent))
			return;
		
		WorkflowProcessEvent e = (WorkflowProcessEvent) event;

		logger.debug(e.getToken() + " event " + e.getType());

		switch (e.getType()) {
			case PAUSED: paused(e.getProcess()); break;
			case RESUMED: resumed(e.getProcess()); break;
			case STARTED: started(e.getProcess()); break;
            case CREATED: created(e.getProcess()); break;
		}
	}
	
	private void created(WorkflowProcess process) {
        eventDispatcher.started();
    }

    /**
	 * Sets variables in a child process. This process isn't launched with a specified configuration,
	 * we have to find one - if it exists.
	 * 
	 * @param process
	 */
	private void started(WorkflowProcess process) {
        if (!hasParent(process)) {
            logger.debug("started event received, but process has no parents");
			return;
        }

		WorkflowProcess parent = process.getSuperProcessToken().getProcessInstance();
		String dir = (String) parent.getContextInstance().getTransientVariable("__dir");
		
		process.getRuntimeActions().clear();
		process.getContextInstance().setTransientVariable("__dir", dir);
		
		Map<String, NodeSimulationConfig> map = parse(new File(dir), process.getProcessDefinition());

		process.getContextInstance().setTransientVariable(CONFIG, map);
	}

	private void paused(WorkflowProcess process) {
		if (hasParent(process)) {
		    logger.debug("paused event received, but it has parents");
			return;
		}

		eventDispatcher.suspended(process.getRootToken().getCurrentNode().getName());
	}
	
	private void resumed(WorkflowProcess process) {
		if (hasParent(process)) {
            logger.debug("resumed event received, but it has parents");
			return;
		}
		
		eventDispatcher.resumed();
	}
	
	private boolean hasParent(WorkflowProcess process) {
		return process.getSuperProcessToken() != null;
	}
}
