/**
 * JPDLParser.java
 * Created Mar 22, 2012
 */
package com.tandbergtv.watchpoint.studio.debugger.runtime.template;

import static org.jbpm.graph.def.Event.EVENTTYPE_NODE_ENTER;
import static org.jbpm.graph.def.Event.EVENTTYPE_NODE_LEAVE;
import static org.jbpm.graph.def.Event.EVENTTYPE_PROCESS_END;
import static org.jbpm.graph.def.Event.EVENTTYPE_SUPERSTATE_ENTER;
import static org.jbpm.graph.def.Event.EVENTTYPE_SUPERSTATE_LEAVE;
import static org.jbpm.graph.def.Event.EVENTTYPE_TRANSITION;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.log4j.Logger;
import org.dom4j.Element;
import org.jbpm.graph.def.Action;
import org.jbpm.graph.def.Event;
import org.jbpm.graph.def.GraphElement;
import org.jbpm.graph.def.Node;
import org.jbpm.graph.def.NodeCollection;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.node.Decision;
import org.jbpm.graph.node.NodeTypes;
import org.jbpm.graph.node.ProcessState;
import org.jbpm.graph.node.StartState;
import org.jbpm.instantiation.Delegation;

import com.tandbergtv.watchpoint.studio.debugger.runtime.Services;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exe.ManualTaskHandler;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exe.NodeEnterHandler;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exe.NodeLeaveHandler;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exe.ProcessEndHandler;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exe.SimulationDataHandler;
import com.tandbergtv.workflow.core.WorkflowTemplate;
import com.tandbergtv.workflow.driver.DriverException;
import com.tandbergtv.workflow.driver.service.ITemplateLoaderService;
import com.tandbergtv.workflow.driver.template.ExtendedJpdlXmlReader;
import com.tandbergtv.workflow.driver.template.TemplateReaderException;

/**
 * Template parser
 * 
 * @author Sahil Verma
 */
public class JPDLParser extends ExtendedJpdlXmlReader {

	private File dir;
	
	private static final long serialVersionUID = -8644316196789750523L;
	
	private static final Logger logger = Logger.getLogger(JPDLParser.class);

	public JPDLParser(String name) throws FileNotFoundException {
		super(new FileInputStream(name));
		dir = new File(name).getParentFile().getParentFile();
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	public ProcessDefinition readProcessDefinition() {
		super.readProcessDefinition();
		
		Event event = new Event(processDefinition, EVENTTYPE_PROCESS_END);
		
		event.addAction(new Action(new Delegation(ProcessEndHandler.class.getName())));
		processDefinition.addEvent(event);
		
		save(processDefinition);
		
		return processDefinition;
	}

	@Override
	protected void parseProtectionKeys(Element root) {
	}

	@Override
	public Map<String, String> getPredefinedActions() {
		return new HashMap<String, String>();
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	public void readNodes(Element element, NodeCollection nodeCollection) {
		Iterator<?> i = element.elementIterator();
		
		while (i.hasNext()) {
			Element e = (Element) i.next();
			String name = e.getName();
			Class<?> type = NodeTypes.getNodeType(name);

			if (type == null)
				continue;

			Node node = null;
			
			try {
				node = (Node) type.newInstance();
			} catch (Exception e1) {
				addError("Couldn't instantiate node " + name +" of type " + type.getName(), e1);
				continue;
			}

			node.setProcessDefinition(processDefinition);

			if ((node instanceof StartState) && (processDefinition.getStartState() != null)) {
				addError("At most one start-state allowed in a process");
				continue;
			}
			
			readNode(e, node, nodeCollection);
			
			if (node instanceof ProcessState) {
				//Look for ...jpdl/templatename/processdefinition.xml
				File f = new File(dir, e.element("sub-process").attributeValue("name"));
				
				parse(new File(f, "processdefinition.xml").getAbsolutePath());
			} else if (node instanceof Decision) {
				//FIXME Fixed transition
			}
			
			node.read(e, this);
		}
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	protected void addAction(GraphElement graphElement, String eventType, Action action) {
		if (EVENTTYPE_TRANSITION.equals(eventType)) 
			return;
		
		if (EVENTTYPE_NODE_LEAVE.equals(eventType) || EVENTTYPE_SUPERSTATE_LEAVE.equals(eventType)) {
			action = new Action(new Delegation(NodeLeaveHandler.class.getName()));
		} else if (EVENTTYPE_NODE_ENTER.equals(eventType) || EVENTTYPE_SUPERSTATE_ENTER.equals(eventType)) {
			action = new Action(new Delegation(NodeEnterHandler.class.getName()));
		}
		
		super.addAction(graphElement, eventType, action);
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	public Action readSingleAction(Element e) {
		Action action = super.readSingleAction(e);

		if (action != null && action.getActionDelegation() != null) {
			Delegation d = action.getActionDelegation();
			String className = d.getClassName();
			
			if (className.endsWith("MessageEmitter") || className.endsWith("MessageReceiver")) { //FIXME YUCK
				d.setClassName(SimulationDataHandler.class.getName());
				d.setConfiguration(null);
			} else if (className.endsWith("TaskNotificationHandler")) {
				d.setClassName(ManualTaskHandler.class.getName());
				d.setConfiguration(null);
			} else {
				logger.info(className);
			}
		}
		
		return action;
	}

	/*
	 * {@inheritDoc}
	 */
	@Override
	protected void readExceptionHandler(Element element, GraphElement graphElement) {
		/* FIXME Only support ignore, fail and maybe retry? */
		super.readExceptionHandler(element, graphElement);
	}
	
	protected void parse(String name) {
		try {
			new JPDLParser(name).readProcessDefinition();
		} catch (FileNotFoundException e) {
			throw new TemplateReaderException(name, e);
		}
	}
	
	protected void save(ProcessDefinition template) {
		try {
			Services.getService(ITemplateLoaderService.class).update((WorkflowTemplate) template);
		} catch (DriverException e) {
			throw new TemplateReaderException(":(", e);
		}		
	}
}
