package com.tandbergtv.watchpoint.studio.debugger.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Widget;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.ProcessState;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.model.ParameterizableSimulationData;
import com.tandbergtv.watchpoint.studio.debugger.model.SimulationType;
import com.tandbergtv.watchpoint.studio.ui.model.ManualTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;

public class ParameterizableNodeSimulationComposite extends AbstractNodeSimulationComposite implements SimulationModifyListener {
	
	protected VariableExpressionValuesTable variableValuesTable;
	protected Label lblReturnValues;
	protected Button btnFail;
	protected Label lblFail;
	
	private static final String SEMICOLON = ";";
	
	public ParameterizableNodeSimulationComposite(Composite parent, NodeSimulationConfig node) {
		super(parent, node);
	}

	@Override
	protected void createAdditionalControls(Composite parent) {
		createFailWidget(parent);
		createNodeSimulationTable(parent);
	}
	
	protected void createFailWidget(Composite composite) {
		lblFail = new Label(composite, SWT.NONE);
		lblFail.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		lblFail.setText("Fail :");
		lblFail.setEnabled(false);
		
		btnFail = new Button(composite, SWT.CHECK);
		btnFail.setEnabled(false);
		btnFail.addSelectionListener(this);
	}
	
	protected void createNodeSimulationTable(Composite composite) {
		lblReturnValues = new Label(composite, SWT.NONE);
		lblReturnValues.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 2, 1));
		lblReturnValues.setText("Return Values :");
		
		variableValuesTable = new VariableExpressionValuesTable(composite);
		variableValuesTable.addSimulationModifyListener(this);
	}

	@Override
	public void load(NodeSimulationConfig nodeSimulationConfig) {
		super.load(nodeSimulationConfig);
		
		updateWidgets();
		
		ParameterizableSimulationData simulationData = getSimulationData();
		btnFail.setSelection(simulationData.isFail());
		
		Collection<Variable> variables = Collections.emptyList();
		if (nodeSimulationConfig.getNodeElement() instanceof NodeDefinition) {
			NodeDefinition nd = (NodeDefinition) nodeSimulationConfig.getNodeElement();
			variables = new ArrayList<Variable>(nd.getVariables());
			
			if (nd.getNode() instanceof NodeGroup) {
				// is superstate, show the 'variable name' instead of the 'mapped name'
				variableValuesTable.setLabelProvider(new VariableNameTableLabelProvider());
				
				// caches the variables by mapped-name
				Map<String, Variable> mappedNameMap = new HashMap<String, Variable>(variables.size());
				for (Variable variable : variables) {
					mappedNameMap.put(variable.getMappedName(), variable);
				}
				// loops through all superstate nodes to figure out
				// the true variable 'access'
				NodeGroup group = (NodeGroup) nd.getNode();
				for (NodeElement ssElement : group.getNodeElements()) {
					if (ssElement instanceof NodeDefinition) {
						NodeDefinition ssNode = (NodeDefinition) ssElement;
						for (WPVariable ssNodeVar : ssNode.getVariables()) {
							Variable variable = mappedNameMap.get(ssNodeVar.getName());
							if (variable != null) {
								variable.setAccess(ssNodeVar.getAccess());
							}
						}
					}
				}
			} 
			
		} else if (nodeSimulationConfig.getNodeElement() instanceof ProcessState) {
			ProcessState process = (ProcessState) nodeSimulationConfig.getNodeElement();
			variables = new ArrayList<Variable>(Arrays.asList(process.getVariables()));
		} else if (nodeSimulationConfig.getNodeElement() instanceof ManualTaskNode) {
			ManualTaskNode manual = (ManualTaskNode) nodeSimulationConfig.getNodeElement();
			variables = new ArrayList<Variable>(manual.getVariables());
		}
		
		for (Variable var : variables) {
			WPVariable v = (WPVariable) var;
			if (v.getAccess().contains("write")) {
				v.setValue((String) simulationData.getVariable(v.getName()));
			} else {
				v.setValue(v.getName());
			}
		}
		variableValuesTable.setVariablesData(variables);
        if (template.getStartState().getTask() != null) {
            variableValuesTable.setContextVariables(Arrays.asList(template.getStartState().getTask().getController().getVariables()));
        }
        variableValuesTable.setProject(ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(nodeSimulationConfig.getTemplatePath())).getProject());
        variableValuesTable.initializeExpressionEditor();
	}
	
	protected void updateWidgets() {
		boolean isSimulatedValues = SimulationType.SIMULATED_VALUES.equals(nodeSimulationConfig.getSimulationType());
		variableValuesTable.setEnabled(isSimulatedValues);
		lblReturnValues.setEnabled(isSimulatedValues);
		lblFail.setEnabled(isSimulatedValues);
		btnFail.setEnabled(isSimulatedValues);
	}


	@Override
	protected SimulationType[] getSimulationTypes() {
		return new SimulationType[] {SimulationType.SKIP_NODE, SimulationType.SIMULATED_VALUES};
	}

	/**
	 * 		Notifies when a widget is selected. Subclasses may override.
	 */
	@Override
	public void widgetSelected(SelectionEvent e) {
		Widget widget = (Widget) e.getSource();
		if (SimulationType.SIMULATED_VALUES.equals(widget.getData()) 
		&& !SimulationType.SIMULATED_VALUES.equals(nodeSimulationConfig.getSimulationType())) {
			Button btn = (Button) widget;
			variableValuesTable.setEnabled(btn.getSelection());
			lblReturnValues.setEnabled(btn.getSelection());
			lblFail.setEnabled(btn.getSelection());
			btnFail.setEnabled(btn.getSelection());
			nodeSimulationConfig.setSimulationType(SimulationType.SIMULATED_VALUES);
			updateWidgets();
			fireSimulationModifiedEvent(nodeSimulationConfig);
		} else if (SimulationType.SKIP_NODE.equals(widget.getData())) {
			nodeSimulationConfig.setSimulationType(SimulationType.SKIP_NODE);
			updateWidgets();
			fireSimulationModifiedEvent(nodeSimulationConfig);
		} else if (widget == btnFail) {
			getSimulationData().setFail(btnFail.getSelection());
			fireSimulationModifiedEvent(nodeSimulationConfig);
		}
	}
	
	public ParameterizableSimulationData getSimulationData() {
		ParameterizableSimulationData result = null;
		if (!(nodeSimulationConfig.getSimulationData() instanceof ParameterizableSimulationData)) {
			result = new ParameterizableSimulationData(nodeSimulationConfig.getSimulationData());
		} else {
			result = (ParameterizableSimulationData) nodeSimulationConfig.getSimulationData();
		}
		nodeSimulationConfig.setSimulationData(result);
		return result;
	}

	
	@Override
	public void addSimulationModifyListener(SimulationModifyListener listener) {
		super.addSimulationModifyListener(listener);
	}

	
	public void modifySimulation(Object o) {
		WPVariable var = (WPVariable) o;
		getSimulationData().setVariable(var.getName(), var.getValue());
		fireSimulationModifiedEvent(nodeSimulationConfig);
	}

}
