package com.tandbergtv.watchpoint.studio.interpreterwrapper;

import jargs.gnu.CmdLineParser;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.ServerSocket;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jbpm.graph.def.ProcessDefinition;

import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.runtime.RuntimeLauncher;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exe.Lifecycle;
import com.tandbergtv.watchpoint.studio.debugger.runtime.parser.RuntimeTemplateParser;
import com.tandbergtv.watchpoint.studio.debugger.runtime.template.JPDLParser;

/**
 * The main class that orchestrates the process for creating the ports and attaching process to them.
 * This class should be changed to use real EventDispatchers and real RuntimeCommandFacades.
 *  
 * @author beroubach
 */
public class InterpreterWrapperServer {

    private Integer commandPort;
    private Integer eventPort;
    private String templateFile;
    private String configFile;
    private String presentationTemplateFile;
    
    private static final Logger logger = Logger.getLogger(InterpreterWrapperServer.class);

    public InterpreterWrapperServer(Integer commandPort, Integer eventPort, String templateFile, String presentationTemplateFile, String configFile) {
        this.commandPort = commandPort;
        this.eventPort = eventPort;
        this.templateFile = templateFile;
        this.presentationTemplateFile = presentationTemplateFile;
        this.configFile = configFile;
    }

    public static void main(String[] args) {
        Object[] parsedArgs = readArgs(args);
        InterpreterWrapperServer intepreter = new InterpreterWrapperServer((Integer) parsedArgs[0],
                (Integer) parsedArgs[1], parsedArgs[2].toString(), parsedArgs[3].toString(), parsedArgs[4].toString());
        intepreter.start();
    }

    private static Object[] readArgs(String[] args) {
        InterpreterCommandLineParser parser = new InterpreterCommandLineParser();
        CmdLineParser.Option commandPortOption = parser.addHelp(parser.addIntegerOption('c', "commandPort"), "The port number to listen for commands issued from remote debuggers.");
        CmdLineParser.Option eventPortOption = parser.addHelp(parser.addIntegerOption('e', "eventPort"), "The port number opened to send debug events to clients connected.");
        CmdLineParser.Option templateFileOption = parser.addHelp(parser.addStringOption('t', "templateFile"), "The template file to be processed.");
        CmdLineParser.Option presentationTemplateFileOption = parser.addHelp(parser.addStringOption('p', "presentationTemplateFile"), "The template file to used by the plugin, usually a file that is not named like 'processdefinition.xml'.");
        CmdLineParser.Option configFileOption = parser.addHelp(parser.addStringOption('o', "configFile"), "The config file with simulation data to be used.");
        try {
            parser.parse(args);

            Object[] result = new Object[] {parser.getOptionValue(commandPortOption), parser.getOptionValue(eventPortOption), parser.getOptionValue(templateFileOption), parser.getOptionValue(presentationTemplateFileOption), parser.getOptionValue(configFileOption)};
            if (result[0] == null || result[1] == null || result[2] == null || result[3] == null || result[4] == null) {
                throw new IllegalStateException();
            }

            return result;
        } catch (IllegalStateException e) {
            treatParserException(parser);
        } catch (CmdLineParser.OptionException e) {
            treatParserException(parser);
        }
        return null;
    }

    private static void treatParserException(InterpreterCommandLineParser parser) {
        parser.printUsage();
        System.exit(-1);
    }
    
    private static void treatException(String message) {
        System.err.println(message);
        System.exit(-1);
    }

    private void start() {
        try {
            // creates server sockets
            ServerSocket commandServerSocket = new ServerSocket(commandPort);
            ServerSocket eventServerSocket = new ServerSocket(eventPort);

            // creates listener
            EventListener eventListener = new EventListener(eventServerSocket);
            EventDispatcher dispatcher = new EventDispatcher(eventListener);
            new Thread(eventListener, "event-listener").start();

            logger.debug("Event listener started at port " + eventPort);

            // starts runtime
            startRuntime();

            logger.debug("Runtime started");

            // creates command facade
            IRuntimeCommandFacade runtimeFacade = new RuntimeCommandFacade(presentationTemplateFile, dispatcher);
            CommandPortListener commandReceiver = new CommandPortListener(commandServerSocket, runtimeFacade);
            new Thread(commandReceiver, "command-listener").start();

            logger.debug("Command listener started at port " + commandPort);

        } catch (IOException e) {
            treatException(new StringBuilder().append("Could not listen on ports: ").append(commandPort).append(" or ").append(eventPort).toString());
        } catch (InterruptedException e) {
            treatException(e.getMessage());
        } catch (Exception e) {
            treatException(e.getMessage());
        }
    }

    private ProcessDefinition startRuntime() throws InterruptedException, FileNotFoundException, Exception {
        Lifecycle.start();
        // TODO: Wait for template service to start - ideally register a colleague
        Thread.sleep(2000L);

        ProcessDefinition template = new JPDLParser(templateFile).readProcessDefinition();
        Map<String, NodeSimulationConfig> configs = RuntimeTemplateParser.parse(configFile, template);
        new RuntimeLauncher(template, configs, new File(configFile).getParent()).startSimulation();

        return template;
    }
}

class InterpreterCommandLineParser extends CmdLineParser {
    List<String> optionHelpStrings = new ArrayList<String>();

    public Option addHelp(Option option, String helpString) {
        optionHelpStrings.add(" -" + option.shortForm() + "/--" + option.longForm() + ": " + helpString);
        return option;
    }

    public void printUsage() {
        System.err.println("usage: InterpreterWrapperServer [options]");
        for (Iterator<String> i = optionHelpStrings.iterator(); i.hasNext();) {
            System.err.println(i.next());
        }
    }
}