package com.tandbergtv.watchpoint.studio.external.wpexport.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.util.HashMap;
import java.util.Map;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes.TargetFilesystem;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

public class ResourceGroupExporter extends WatchPointPluginExporter<ResourceGroup>
{
	// The logger
	private static final Logger logger = Logger.getLogger(ResourceGroupExporter.class);
	
	private static final String EXTENSION_ID_SUFFIX = " Resource Group";
	private static final String DEFAULT_TASK_IMAGE_FILE_NAME = "taskImage";

	@Override
	public Map<String, Object> export(ResourceGroup exportable, Map<String, Object> exportInputs) throws ExportFailureException { 
		ExporterInputAttributes.TargetFilesystem targetFs = (TargetFilesystem) exportInputs.get(ExporterInputAttributes.TARGET_FILESYSTEM);
		// Super class will export to filesystem only. 
		if (!ExporterInputAttributes.TargetFilesystem.ECLIPSE_WORKSPACE.equals(targetFs)) {
			return super.export(exportable, exportInputs);
		} 
		
		// Build the plug-in XML document.
		Document document = this.buildPluginDocument(exportable, exportInputs);
		
		// Add the Plug-in dependencies
		this.addPluginDependencies(exportable, exportInputs, document, null);
		
		// Add the Extensions / Extension Points
		this.addPluginPointsAndExtensions(exportable, exportInputs, document, null);

		// Writes the Plug-in document to the root folder
		this.writePluginDocument(exportable, exportInputs, document, null);


		return new HashMap<String, Object>();
	}
	@Override
	protected String getPluginId(ResourceGroup exportable, Map<String, Object> inputs) {
		return exportable.getName() + " Resource Group";
	}

	@Override
	protected String getPluginVersion(ResourceGroup exportable, Map<String, Object> inputs) {
		return String.valueOf(exportable.getVersion());
	}

	@Override
	protected void addPluginPointsAndExtensions(ResourceGroup exportable, Map<String, Object> inputs, Document pluginDocument, File pluginFolder) throws ExportFailureException {
		//adding extension to the plugin document
		String extensionId = exportable.getName() + EXTENSION_ID_SUFFIX;
		Element extensionElement = JPFExportUtil.addExtension(pluginDocument, extensionId,
															  WatchPointPluginConstants.PLUGIN_ID,
															  WatchPointPluginConstants.EXTENSION_POINT_ID_RESOURCE_GROUP);
		
		//adding parameters to the extension element
		//name
		if(isNotNullAndNotBlank(exportable.getName()))
		{
			JPFExportUtil.addParameter(extensionElement, WatchPointPluginConstants.RG_PARAM_NAME,
					exportable.getName());
		}

		//systemId
		if(isNotNullAndNotBlank(exportable.getSystemId()))
		{
			JPFExportUtil.addParameter(extensionElement, WatchPointPluginConstants.RG_PARAM_SYSTEM_ID,
						exportable.getSystemId());
		}

		//functionalType
		if(isNotNullAndNotBlank(exportable.getFunctionalType()))
		{
			JPFExportUtil.addParameter(extensionElement, WatchPointPluginConstants.RG_PARAM_FUNCTIONAL_TYPE,
					exportable.getFunctionalType());
		}
		
		//accessLevel (protection key)
		if(exportable.getProtectionKey() != null && isNotNullAndNotBlank(exportable.getProtectionKey().getName()))
		{
				JPFExportUtil.addParameter(extensionElement, WatchPointPluginConstants.RG_PARAM_PROTECTION_KEY,
						exportable.getProtectionKey().getName());
		}

		//isInternallyAcquired
		JPFExportUtil.addParameter(extensionElement, WatchPointPluginConstants.RG_PARAM_INTERNALLY_ACQUIRED,
				String.valueOf(exportable.isInternallyAcquired()));
		
		//allocationStrategyClass
		if(isNotNullAndNotBlank(exportable.getAllocationStrategyClassName()))
		{
			JPFExportUtil.addParameter(extensionElement, WatchPointPluginConstants.RG_PARAM_ALLOCATION_STRATEGY,
					exportable.getAllocationStrategyClassName());
		}
		
		//imagePath
		if(exportable.getImageFileName() != null && !exportable.getImageFileName().trim().isEmpty())
		{
			//write image content to file
			String imageFileName = (isNotNullAndNotBlank(exportable.getImageFileName()))
										? exportable.getImageFileName()
										: DEFAULT_TASK_IMAGE_FILE_NAME;
			
			//add the file path as an extension parameter
			JPFExportUtil.addParameter(extensionElement, WatchPointPluginConstants.RG_PARAM_IMAGE_PATH, imageFileName);
		}
		
		// Recursively exports all the files in the ResourceGroup folder to the filesystem. 
		ExporterInputAttributes.TargetFilesystem targetFs = (TargetFilesystem) inputs.get(ExporterInputAttributes.TARGET_FILESYSTEM);
		if (!ExporterInputAttributes.TargetFilesystem.ECLIPSE_WORKSPACE.equals(targetFs)) {

			IFile sourceFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(exportable.getPath()));
			IFolder sourceContainer = (IFolder) sourceFile.getParent();
			
			try
			{
				FileUtil.exportWorkspaceFolderToFilesystem(sourceContainer, pluginFolder, true);
			} catch (Exception e)
			{
				String message = "IO Exception caught while exporting the image (writing image content to file). " +
									GENERIC_EXCEPTION_MESSAGE;
				throw new ExportFailureException(message, e);
			}
		}
	}

	private boolean isNotNullAndNotBlank(String str)
	{
		return (str != null && str.trim().length() > 0);
	}
	
	/**
	 * Write the plug-in document to file 'plugin.xml' in the root folder for the plug-in
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * @param pluginDocument
	 *            The plug-in XML Document
	 * @param pluginRootFolder
	 *            The root folder for the plug-in
	 * 
	 * @throws ExportFailureException
	 *             Exception writing the plug-in document
	 */
	protected void writePluginDocument(ResourceGroup exportable, Map<String, Object> inputs, Document pluginDocument, File pluginRootFolder) throws ExportFailureException {
		ExporterInputAttributes.TargetFilesystem targetFs = (TargetFilesystem) inputs.get(ExporterInputAttributes.TARGET_FILESYSTEM);
		// Super class will export to filesystem only. 
		if (!ExporterInputAttributes.TargetFilesystem.ECLIPSE_WORKSPACE.equals(targetFs)) {
			super.writePluginDocument(exportable, inputs, pluginDocument, pluginRootFolder);
			return;
		} 
		String filePath = inputs.get(ExporterInputAttributes.EXPORT_FOLDER_PATH).toString();
		OutputStreamWriter writer = null;
		try {
			IFile targetFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(filePath));

			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();

			// Set the DocType and indentation
			transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, JPFConstants.DOCTYPE_PUBLIC_ID);
			transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, JPFConstants.DOCTYPE_SYSTEM_ID);
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			writer = new OutputStreamWriter(baos, "UTF-8");
			
			transformer.transform(new DOMSource(pluginDocument), new StreamResult(writer));
			
			InputStream in = new ByteArrayInputStream(baos.toByteArray());

			if (targetFile.exists()) {
				targetFile.setContents(in, IFile.FORCE, null);
			} else {
				targetFile.create(in, IFile.FORCE, null);
			}
		}
		catch (Exception ex)
		{
			String msg = "Failed to write the Plugin document to the file: " + filePath + ". " + GENERIC_EXCEPTION_MESSAGE;
			throw new ExportFailureException(msg, ex);
		}
		finally
		{
			if (writer != null)
			{
				try
				{
					writer.close();
				}
				catch (IOException ex)
				{
					String msg = "Failed to close file stream after writing WatchPoint Plugin file: "
							+ filePath;
					logger.warn(msg, ex);
				}
			}
		}
	}
}
