/*
 * Created on Sep 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.SCHEMA_FILE_NAME;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.FormPartDirtyModifyListener;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.TextTrimFocusListener;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.fields.IResourceTypeGeneralPageInputField;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.ProjectPreferencesUtil;

/**
 * The General Section Part containing the basic properties for a Resource Type
 * 
 * @author Vijay Silva
 */
public class GeneralSectionPart extends ResourceTypeSectionPart
{
	private static final String RT_GENERAL_PAGE_FIELDS_EXT_NAME = "org.jbpm.gd.jpdl.resourceTypeGeneralPageFields";
	
	Label nameLabel, systemIdLabel, schemaFileLabel, versionLabel;

	Text nameText, systemIdText, schemaFileText, versionText;

	List<IResourceTypeGeneralPageInputField> extFields = new ArrayList<IResourceTypeGeneralPageInputField>();
	
	
	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite
	 * @param toolkit
	 *            The toolkit to create the widgets
	 * @param style
	 *            The Style bits to create the Section
	 */
	public GeneralSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit,
			int style)
	{
		super(parentPart, parent, toolkit, style);
	}

	// ========================================================================
	// ===================== OVERRIDDEN SECTION PART METHODS
	// ========================================================================

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#commit(boolean)
	 */
	@Override
	public void commit(boolean onSave)
	{
		super.commit(onSave);

		/* Update the Resource Type with the data in the widgets */
		ResourceType resourceType = this.getResourceType();
		if (resourceType != null)
		{
			resourceType.setName(this.nameText.getText());
			resourceType.setSystemId(this.systemIdText.getText());
			
			/* Sets the values to the resource type object */
			for (IResourceTypeGeneralPageInputField extField : extFields) {
				extField.setObjectValues(resourceType);
			}
		}
	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	/*
	 * Create the Controls displayed in the composite
	 */
	@Override
	protected void createSectionClientControls(IManagedForm managedForm)
	{
		FormToolkit toolkit = managedForm.getToolkit();

		/* The Name Label and Text */
		this.nameLabel = toolkit.createLabel(this.sectionClient, "Name: ");
		this.nameText = toolkit.createText(this.sectionClient, "");
		this.nameText.setTextLimit(50);
		this.nameText.addFocusListener(new TextTrimFocusListener());
		this.nameText.addModifyListener(new FormPartDirtyModifyListener(this));

		/* The System ID Label and Text */
		this.systemIdLabel = toolkit.createLabel(this.sectionClient, "System ID: ");
		this.systemIdText = toolkit.createText(this.sectionClient, "", SWT.SINGLE | SWT.READ_ONLY);
		this.systemIdText.setTextLimit(10);
		
		/* The schema file name */
		this.schemaFileLabel = toolkit.createLabel(this.sectionClient, "Schema: ");
		this.schemaFileText = toolkit.createText(this.sectionClient, "");
		this.schemaFileText.setEditable(false);
		
		/* Load extension point controls */
		loadExtensionControls();
		
		/* Create extension point controls in the section client */
		createExtensionControls(this.sectionClient);
		
		/* The version label and text */
		this.versionLabel = toolkit.createLabel(this.sectionClient, "Resource Type Version: ");
		this.versionText = toolkit.createText(this.sectionClient, "", SWT.SINGLE | SWT.READ_ONLY);
	}
	
	/**
	 * Load the controls defined by the extension point.<br>
	 * The classes representing the controls are loaded in a list.
	 */
	private void loadExtensionControls(){
		IConfigurationElement[] configurationElements = Platform.getExtensionRegistry()
		.getConfigurationElementsFor(RT_GENERAL_PAGE_FIELDS_EXT_NAME);
		
		extFields.clear();
		
		/* Loads each extension point field according to the order defined in plugin.xml */
		for (int i = 0; i < configurationElements.length; i++) {
			try {
				// Creates an instance for the class defined by the 'class' parameter from the
				// extension point calling the 0-argument public constructor
				Object classObject = configurationElements[i].createExecutableExtension("class");
				IResourceTypeGeneralPageInputField extField = (IResourceTypeGeneralPageInputField)classObject;
				extFields.add(extField);
			} catch (CoreException e) {
				Utility.reportError("Error loading extension point resource type general fields.", e);
			}
		}
	}
	
	/**
	 * Creates the controls defined by extension points. <br>
	 * For each class defined by the extension point is given a composite where the control 
	 * shall be created, a label layout data and a control layout data. <br>
	 * Each control can modify the layout data for specific control requirements, 
	 * for example, to define a custom control width .   
	 * @param parent The composite to be used for control creation
	 */
	private void createExtensionControls(Composite parent){
		/* Add the fields loaded by extension points */
		for (int i = 0; i < extFields.size(); i++) {
			IResourceTypeGeneralPageInputField current = extFields.get(i);
			// Layout data for each control
			GridData labelData = createLabelGridData();
			GridData controlData = createGridData();
	
			current.createControls(parent, labelData, controlData);
			// Sets a modify listener for each control, then the page is notified if it should be
			// marked like dirty.
			current.setFormPartDirtyModifyListener(new FormPartDirtyModifyListener(this));
		}
	}
	
	/* Prepare the Layout of the Section Part */
	@Override
	protected void initializeSectionClientLayout(IManagedForm managedForm)
	{
		/* Set the Layout for the Section Client in the Form */
		this.sectionClient.setLayout(this.createGridLayout(2, false));

		/* Set the default Layout data for the widgets */
		this.nameLabel.setLayoutData(this.createLabelGridData());
		this.systemIdLabel.setLayoutData(this.createLabelGridData());
		this.systemIdText.setLayoutData(this.createGridData());
		this.schemaFileLabel.setLayoutData(this.createGridData());
		this.versionLabel.setLayoutData(this.createLabelGridData());
		this.versionText.setLayoutData(this.createGridData());
		
		/* Set the layout data for the schema text */
		GridData schema = this.createGridData();
		schema.widthHint = 300;
		this.schemaFileText.setLayoutData(schema);

		/* Set the Layout data for the Name and Description Text */
		GridData layoutData = this.createGridData();
		layoutData.widthHint = 200;
		this.nameText.setLayoutData(layoutData);

	}

	// ========================================================================
	// ===================== DISPLAYING THE INPUT DATA
	// ========================================================================

	/*
	 * Display the Input data in the Section Widgets
	 */
	@Override
	protected void populateSectionData()
	{
		ResourceType resourceType = this.getResourceType();
		
		/* Initialize the values for the extension fields */
		for (IResourceTypeGeneralPageInputField extField : extFields) {
			extField.initValues();
		}

		/* Set all the Data in their respective fields */
		if (resourceType != null)
		{	
			/* Set name */
			this.nameText.setText(this.formatValue(resourceType.getName(), true));
			/* Set system id */
			this.systemIdText.setText(this.formatValue(resourceType.getSystemId(), true));
			/* Set schema file name */
			String schema = getResourceTypeSchema(resourceType);
			if(schema != null && !schema.isEmpty()){
				this.schemaFileText.setText(schema);
			}else{
				this.schemaFileText.setText("No schema file defined.");
				this.schemaFileText.setEnabled(false);
			}
			
			/* Set the current version number */
			this.versionText.setText(Integer.toString(resourceType.getVersion()));
			
			/* Set the current values for the extension fields */
			for (IResourceTypeGeneralPageInputField extField : extFields) {
				extField.setCurrentValue(resourceType);
			}
		}
		else
		{
			this.nameText.setText("");
			this.systemIdText.setText("");
		}
	}
	
	private String getResourceTypeSchema(ResourceType rt){
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(rt.determineNameForProject());
		String schemaFileName = null;
		try {
			boolean projectWasOpened = false;
			if(!project.isOpen()){
				project.open(null);
				projectWasOpened = true;
			}
			schemaFileName = ProjectPreferencesUtil.getProjectPreference(project, SCHEMA_FILE_NAME, "");
			if(projectWasOpened){
				project.close(null);
			}
		} catch (CoreException e) {
			Utility.reportError("Error loading the schema file name.", e);
		}
		return schemaFileName;
	}

	/*
	 * Set the permissions of the section widgets based on the input
	 */
	@Override
	protected void setWidgetPermissions()
	{
		boolean isEditable = this.isInputEditable();
		this.nameText.setEditable(isEditable);
		
		/* Set the permission for the extension fields */
		for (IResourceTypeGeneralPageInputField extField : extFields) {
			extField.setWidgetPermission(isEditable);
		}
	}
}
