/*
 * Created on Sep 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.dailog.ResourceTypeParameterDialog;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.dailog.ResourceTypeParameterNameValidator;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.dailog.ResourceTypeParameterValueValidator;

/**
 * A Container for the controls present in the ResourceType Parameters table composite.
 * 
 * @author Vijay Silva
 */
class ParametersCompositeContainer
{
	AbstractFormPart formPart;

	Composite parametersComposite;

	Table parameterTable;

	TableColumn nameColumn, valueColumn;

	Composite buttonComposite;

	Button addButton, removeButton, editButton;

	private Map<String, String> parameters;

	private boolean isEditable = true;

	private IEditorPart editorPart;

	/**
	 * Constructor
	 * 
	 * @param parentPart
	 *            The Parent Workbench Part that contains this composite
	 * @param formPart
	 *            The Parent Form Part that contains this composite
	 * @param toolkit
	 *            The Toolkit used to create the child widgets
	 * @param parametersComposite
	 *            The Parameters Composite wrapped by this container
	 */
	public ParametersCompositeContainer(IEditorPart parentPart, AbstractFormPart formPart,
			FormToolkit toolkit, Composite parametersComposite)
	{
		this.editorPart = parentPart;
		this.formPart = formPart;
		this.parametersComposite = parametersComposite;

		this.initialize(toolkit, parametersComposite);
	}

	// ========================================================================
	// ===================== PUBLIC METHODS
	// ========================================================================

	/**
	 * Gets the list of all Parameters as currently displayed in the Table
	 * 
	 * @return The parameters displayed in the Table
	 */
	public Map<String, String> getParameters()
	{
		Map<String, String> parameters = new HashMap<String, String>();
		int itemCount = this.parameterTable.getItemCount();
		for (int index = 0; index < itemCount; index++)
		{
			TableItem tableItem = this.parameterTable.getItem(index);
			String name = tableItem.getText(0);
			String value = tableItem.getText(1);
			parameters.put(name, value);
		}

		return parameters;
	}

	/**
	 * Set the Widgets to be enabled or disabled
	 * 
	 * @param enable
	 *            flag to indicate if the widgets should be enabled or disabled
	 */
	public void setEnabled(boolean enable)
	{
		this.parameterTable.setEnabled(enable);
		this.addButton.setEnabled(enable);
		this.removeButton.setEnabled(enable);
		this.editButton.setEnabled(enable);

		if (enable)
			this.setWidgetPermissions();
	}

	// ========================================================================
	// ===================== USER CONTROLS INITIALIZATION
	// ========================================================================

	/**
	 * Initialize the Composite
	 * 
	 */
	private void initialize(FormToolkit toolkit, Composite parent)
	{
		this.initializeTable(toolkit, parent);
		this.buttonComposite = toolkit.createComposite(parent);
		toolkit.paintBordersFor(this.buttonComposite);
		this.addButton = toolkit.createButton(this.buttonComposite, "Add...", SWT.PUSH);
		this.editButton = toolkit.createButton(this.buttonComposite, "Edit...", SWT.PUSH);
		this.removeButton = toolkit.createButton(this.buttonComposite, "Remove", SWT.PUSH);

		/* Add the button selection listeners */
		this.addButton.addSelectionListener(new ButtonSelectionListener());
		this.editButton.addSelectionListener(new ButtonSelectionListener());
		this.removeButton.addSelectionListener(new ButtonSelectionListener());

		this.initializeLayouts(parent);
		this.updateButtonStatus();
	}

	/**
	 * Initialize the table columns
	 */
	protected void initializeTable(FormToolkit toolkit, Composite parent)
	{
		this.parameterTable = toolkit.createTable(parent, SWT.SINGLE | SWT.FULL_SELECTION);

		this.nameColumn = new TableColumn(this.parameterTable, SWT.LEFT, 0);
		this.nameColumn.setText("Parameter Name");
		this.nameColumn.setToolTipText("Parameter Name");
		this.nameColumn.setResizable(false);
		this.nameColumn.setMoveable(false);
		this.nameColumn.setWidth(130);

		this.valueColumn = new TableColumn(this.parameterTable, SWT.LEFT, 1);
		this.valueColumn.setText("Parameter Value");
		this.valueColumn.setToolTipText("Parameter Value");
		this.valueColumn.setResizable(false);
		this.valueColumn.setMoveable(false);
		this.valueColumn.setWidth(200);

		this.parameterTable.setHeaderVisible(true);
		this.parameterTable.setSortColumn(this.nameColumn);
		this.parameterTable.setSortDirection(SWT.DOWN);
		this.parameterTable.setTopIndex(0);
		this.parameterTable.setLinesVisible(true);
		this.parameterTable.addSelectionListener(new TableSelectionListener());
	}

	/**
	 * Set the Layouts for the widgets
	 */
	protected void initializeLayouts(Composite parent)
	{
		GridLayout gridLayout = new GridLayout(2, false);
		gridLayout.marginHeight = gridLayout.marginWidth = 0;
		parent.setLayout(gridLayout);

		GridData gridData = new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false);
		gridData.heightHint = 85;
		this.parameterTable.setLayoutData(gridData);

		gridData = new GridData(SWT.BEGINNING, SWT.BEGINNING, false, true);
		this.buttonComposite.setLayoutData(gridData);

		RowLayout rowLayout = new RowLayout(SWT.VERTICAL);
		rowLayout.fill = true;
		rowLayout.pack = false;
		rowLayout.marginTop = rowLayout.marginBottom = 0;
		rowLayout.marginLeft = rowLayout.marginRight = 2;
		rowLayout.spacing = 5;
		this.buttonComposite.setLayout(rowLayout);

		this.addButton.setLayoutData(new RowData());
		this.editButton.setLayoutData(new RowData());
		this.removeButton.setLayoutData(new RowData());
	}

	// ========================================================================
	// ===================== INPUT INITIALIZATION
	// ========================================================================

	/**
	 * Set the Editor Input as the input for this control
	 * 
	 * @param parameters
	 *            the ResourceType Parameters
	 * @param isEditable
	 *            Indicates if the data is editable
	 */
	public void setInput(Map<String, String> parameters, boolean isEditable)
	{
		this.parameters = parameters;
		this.isEditable = isEditable;

		this.populateData();
		this.setWidgetPermissions();
	}

	/**
	 * Display the data in the Table
	 */
	protected void populateData()
	{
		this.parameterTable.setRedraw(false);

		try
		{
			this.parameterTable.removeAll();

			if (parameters != null)
			{
				List<String> parameterNames = new ArrayList<String>(this.parameters.keySet());
				Collections.sort(parameterNames, new ParameterNameComparator());
				for (String parameterName : parameterNames)
				{
					String parameterValue = this.parameters.get(parameterName);
					parameterName = (parameterName != null) ? parameterName.trim() : "";
					parameterValue = (parameterValue != null) ? parameterValue : "";
					TableItem tableItem = new TableItem(this.parameterTable, SWT.NULL);
					tableItem.setText(new String[] { parameterName, parameterValue });
				}
			}
		}
		finally
		{
			this.parameterTable.setRedraw(true);
		}
	}

	/**
	 * Set the widget permissions (editable / enabled)
	 */
	protected void setWidgetPermissions()
	{
		this.addButton.setEnabled(this.isEditable);
		this.editButton.setEnabled(this.isEditable);
		this.removeButton.setEnabled(this.isEditable);
		this.updateButtonStatus();
	}

	// ========================================================================
	// ===================== ACTIONS FOR THE TABLE
	// ========================================================================

	protected void updateButtonStatus()
	{
		if (!this.isEditable)
			return;

		boolean isItemSelected = (this.parameterTable.getSelectionIndex() != -1);
		this.editButton.setEnabled(isItemSelected);
		this.removeButton.setEnabled(isItemSelected);
	}

	// ========================================================================
	// ===================== ACTIONS FOR THE BUTTONS
	// ========================================================================

	/**
	 * Add new Parameter to the table
	 */
	protected void addParameter()
	{
		ResourceTypeParameterDialog dialog = this.createParameterDialog(null, null);
		if (dialog.open() == Window.OK)
		{
			String name = dialog.getParameterName();
			String value = dialog.getParameterValue();
			this.insertParameterInTable(name, value);
			this.markFormPartDirty();
		}
	}

	/**
	 * Edit an existing Parameter in the table
	 */
	protected void editParameter()
	{
		int selectionIndex = this.parameterTable.getSelectionIndex();
		if (selectionIndex != -1)
		{
			TableItem tableItem = this.parameterTable.getItem(selectionIndex);
			String name = tableItem.getText(0);
			String value = tableItem.getText(1);
			ResourceTypeParameterDialog dialog = this.createParameterDialog(name, value);
			if (dialog.open() == Window.OK)
			{
				String newName = dialog.getParameterName();
				String newValue = dialog.getParameterValue();

				// Check if the name or value have not been modified
				if (name.equals(newName) && value.equals(newValue))
					return;

				if (name.equals(newName))
				{ // Name is not changed, no need to sort
					tableItem.setText(1, newValue);
				}
				else
				{ // Need to sort the rows by Name Column
					this.parameterTable.remove(selectionIndex);
					this.insertParameterInTable(newName, newValue);
				}

				this.markFormPartDirty();
			}
		}
	}

	/**
	 * Remove the existing Parameter from the table
	 */
	protected void removeParameter()
	{
		int selectionIndex = this.parameterTable.getSelectionIndex();
		if (selectionIndex != -1)
		{
			this.parameterTable.remove(selectionIndex);

			int itemCount = this.parameterTable.getItemCount();
			if (selectionIndex >= itemCount)
				selectionIndex = itemCount - 1;
			this.parameterTable.select(selectionIndex);
			this.updateButtonStatus();

			this.markFormPartDirty();
		}
	}

	/*
	 * Creates a ResourceType Parameter Editor Dialog with the given parameter name and value.
	 */
	private ResourceTypeParameterDialog createParameterDialog(String name, String value)
	{
		ResourceTypeParameterDialog dialog = null;

		Shell shell = this.parametersComposite.getShell();
		String pluginId = this.editorPart.getSite().getPluginId();
		dialog = new ResourceTypeParameterDialog(shell, pluginId, name, value);
		Set<String> names = this.getParameterNames();
		dialog.setParameterNameValidator(new ResourceTypeParameterNameValidator(names, name));
		dialog.setParameterValueValidator(new ResourceTypeParameterValueValidator());

		return dialog;
	}

	/*
	 * Get all the current names displayed in the table
	 */
	private Set<String> getParameterNames()
	{
		Set<String> names = new HashSet<String>();

		for (int i = 0; i < this.parameterTable.getItemCount(); i++)
			names.add(this.parameterTable.getItem(i).getText(0));

		return names;
	}

	/* Inserts the new Table Item in a sorted manner */
	private void insertParameterInTable(String name, String value)
	{
		int index = this.parameterTable.getItemCount();
		for (int i = 0; i < this.parameterTable.getItemCount(); i++)
		{
			String currentName = this.parameterTable.getItem(i).getText(0);
			if ((currentName.compareToIgnoreCase(name) > 0)
					|| ((currentName.compareToIgnoreCase(name) == 0) && currentName.compareTo(name) > 0))
			{
				index = i;
				break;
			}
		}

		TableItem tableItem = new TableItem(this.parameterTable, SWT.NONE, index);
		tableItem.setText(new String[] { name, value });
		this.parameterTable.select(index);
		this.updateButtonStatus();
	}

	/*
	 * Mark the Form Part as dirty
	 */
	private void markFormPartDirty()
	{
		if (this.formPart != null && !this.formPart.isDirty())
			this.formPart.markDirty();
	}

	// ========================================================================
	// ===================== EVENT LISTENERS
	// ========================================================================

	/*
	 * Selection Listener for the Table
	 */
	private class TableSelectionListener extends SelectionAdapter
	{
		/*
		 * @see org.eclipse.swt.events.SelectionAdapter#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		@Override
		public void widgetDefaultSelected(SelectionEvent e)
		{
			ParametersCompositeContainer.this.updateButtonStatus();
		}

		/*
		 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		@Override
		public void widgetSelected(SelectionEvent e)
		{
			ParametersCompositeContainer.this.updateButtonStatus();
		}
	}

	/*
	 * Selection Listeners for the Buttons
	 */
	private class ButtonSelectionListener extends SelectionAdapter
	{
		/*
		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetSelected(SelectionEvent event)
		{
			if (event.getSource() == ParametersCompositeContainer.this.addButton)
			{
				ParametersCompositeContainer.this.addParameter();
			}
			else if (event.getSource() == ParametersCompositeContainer.this.editButton)
			{
				ParametersCompositeContainer.this.editParameter();
			}
			else if (event.getSource() == ParametersCompositeContainer.this.removeButton)
			{
				ParametersCompositeContainer.this.removeParameter();
			}
		}
	}

	// ========================================================================
	// ===================== COMPARATOR
	// ========================================================================

	/*
	 * Internal Comparator for sorting Parameter Names
	 */
	private class ParameterNameComparator implements Comparator<String>
	{
		public int compare(String s1, String s2)
		{
			int n1 = s1.length(), n2 = s2.length();
			for (int i1 = 0, i2 = 0; i1 < n1 && i2 < n2; i1++, i2++)
			{
				char c1 = s1.charAt(i1);
				char c2 = s2.charAt(i2);
				if (c1 != c2)
				{
					c1 = Character.toUpperCase(c1);
					c2 = Character.toUpperCase(c2);
					if (c1 != c2)
					{
						c1 = Character.toLowerCase(c1);
						c2 = Character.toLowerCase(c2);
						if (c1 != c2)
						{
							return c1 - c2;
						}
					}
				}
			}

			int lengthDiff = n1 - n2;
			if (lengthDiff == 0)
				return s1.compareTo(s2);

			return lengthDiff;
		}
	}
}
