package com.tandbergtv.watchpoint.studio.ui.properties;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.InvalidPropertiesFormatException;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Controller;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNodeType;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;

public class LoopNodeDetailsConfigurationComposite implements SelectionListener, FocusListener {
	private static final Logger logger = Logger.getLogger(LoopNodeDetailsConfigurationComposite.class);
	
	public static LoopNodeDetailsConfigurationComposite create(AbstractPropertySection parentPropertySection, TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		LoopNodeDetailsConfigurationComposite result = new LoopNodeDetailsConfigurationComposite();
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.parentPropertySection = parentPropertySection;
		result.create();
		return result;
	}

	private AbstractPropertySection parentPropertySection;
	private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;
	private LoopNode loopNode;
	
	private Label indexLabel;
	private CCombo indexCombo;
	
	private Button forEachRadioButton;
	private Button forRadioButton;
	private Button whileRadioButton;
	
	private Label listValueLabel;
	private CCombo listVariablesCombo;
	
	private Label finalValueLabel;
	private Text finalValueText;
	private Label expressionLabel;
	private Text expressionText;
	private Label initialValueLabel;
	private Text initialValueText;
	
	public void setLoopNode(LoopNode loopNode) {
		if (this.loopNode == loopNode) return;
		unhookListeners();
		this.loopNode = (LoopNode) loopNode;
		clearControls();
		updateControls();
		hookListeners();
	}
	
	private void hookListeners() {
		indexCombo.addFocusListener(this);
		listVariablesCombo.addFocusListener(this);
		finalValueText.addFocusListener(this);
		finalValueText.addVerifyListener(verifyListener);
		expressionText.addFocusListener(this);
		initialValueText.addFocusListener(this);
		initialValueText.addVerifyListener(verifyListener);
		
		forEachRadioButton.addSelectionListener(this);
		forRadioButton.addSelectionListener(this);
		whileRadioButton.addSelectionListener(this);
	}
	
	Listener modifyListener = new Listener() {
	    public void handleEvent(Event p_event) {
	    	setFinalValueVisible(true);
	    }
	  };

	VerifyListener verifyListener = new VerifyListener() {
		public void verifyText(VerifyEvent event) {
			// Get the character typed
			char myChar = event.character;
			// Allow 0-9
			if (!Character.isDigit(myChar))
				event.doit = false;
			// Allow backspace and delete
			if (myChar == '\b' || event.keyCode == 127 || myChar == Character.UNASSIGNED) {
				event.doit = true;
			}
		}
	};
	
	private void unhookListeners() {
		indexCombo.removeFocusListener(this);
		listVariablesCombo.removeFocusListener(this);
		finalValueText.removeFocusListener(this);
		finalValueText.removeVerifyListener(verifyListener);
		expressionText.removeFocusListener(this);
		initialValueText.removeFocusListener(this);
		initialValueText.removeVerifyListener(verifyListener);
		
		forEachRadioButton.removeSelectionListener(this);
		forRadioButton.removeSelectionListener(this);
		whileRadioButton.removeSelectionListener(this);
	}
	
	private void clearControls() {
		indexCombo.setText("");
		listVariablesCombo.setText("");
		finalValueText.setText("");
		expressionText.setText("");
		initialValueText.setText("");
	}
	
	private void updateControls() {
		fillCombos();
		indexCombo.setText(loopNode.getIndex() == null ? "" : loopNode.getIndex());
		if( LoopNodeType.FOR_EACH.equals(loopNode.getLoopType()) ){
			setButtonSelected(forEachRadioButton);
			setListVariablesVisible(true);
			setInitialValueVisible(false);
		} else if( LoopNodeType.FOR.equals(loopNode.getLoopType()) ){
			setButtonSelected(forRadioButton);
			setFinalValueVisible(true);
		} else if( LoopNodeType.WHILE.equals(loopNode.getLoopType()) ){
			setButtonSelected(whileRadioButton);
			setExpressionVisible(true);
		}
		expressionText.setText(loopNode.getExpression() == null ? "" : loopNode.getExpression());
		initialValueText.setText( Integer.toString(loopNode.getInitialValue()) );
	}
	
	private void setButtonSelected(Button button){
		whileRadioButton.setSelection(false);
		forEachRadioButton.setSelection(false);
		forRadioButton.setSelection(false);
		button.setSelection(true);
	}
	

	/*
	 * Fills the variable combo with node definition start variables.
	 */
	private void fillCombos() {
		indexCombo.removeAll();
		NodeElementContainer elementContainer = Utility.getNodeElementContainer(this.parentPropertySection.getPart());
		/* Get all variables on the template or superstate */
		List<? extends Variable> variables = Collections.emptyList();

		if (elementContainer instanceof WorkflowTemplate) {
			WorkflowTemplate template = (WorkflowTemplate) elementContainer;
			StartState startState = template.getStartState();
			if (startState != null && startState.getTask() != null) {
				Controller controller = startState.getTask().getController();
				variables = Arrays.asList(controller.getVariables());
			}
		} else if (elementContainer instanceof NodeDefinition) {
			NodeDefinition node = (NodeDefinition) elementContainer;
			variables = node.getVariables();
		}
		
		Collections.sort(variables, new Comparator<Variable>() {
			@Override
			public int compare(Variable var1, Variable var2) {
				String var1name = "";
				if (var1.getName() != null) {
					var1name = var1.getName();
				}
				String var2name = "";
				if (var2.getName() != null) {
					var2name = var2.getName();
				}
				return var1name.compareTo(var2name);
			}
		});
		
		this.indexCombo.add("");
		this.listVariablesCombo.add("");
		for (Variable element : variables) {
			WPVariable var = (WPVariable) element;
			String varType = var.getType();
			DataType type = DataType.STRING;
			if (varType != null && !varType.trim().isEmpty()) {
				type = DataType.valueOf(varType);
			}
			if ( type == DataType.INT ) {
				if( var.getName() != null ){
					this.indexCombo.add(var.getName());
				} else if (elementContainer instanceof NodeDefinition){
					this.indexCombo.add(var.getMappedName());
				}
			}
			if ( type == DataType.LIST ){
				if( var.getName() != null ){
					this.listVariablesCombo.add(var.getName());
				} else if (elementContainer instanceof NodeDefinition){
					this.listVariablesCombo.add(var.getMappedName());
				}
			}
		}
	}
	
	private void create() {
		indexLabel = widgetFactory.createLabel(parent, "Index");
		indexCombo = widgetFactory.createCCombo(parent, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL);
		
		forEachRadioButton = new Button(parent, SWT.RADIO );
		forEachRadioButton.setText("For each");
		forEachRadioButton.setBackground( new Color(null, 255, 255, 255) );
		forRadioButton = new Button(parent, SWT.RADIO);
		forRadioButton.setText("For");
		forRadioButton.setBackground( new Color(null, 255, 255, 255) );
		whileRadioButton = new Button(parent, SWT.RADIO);
		whileRadioButton.setText("While");
		whileRadioButton.setBackground( new Color(null, 255, 255, 255) );
		
		listValueLabel = widgetFactory.createLabel(parent, "List Variable");
		listVariablesCombo = widgetFactory.createCCombo(parent, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL);
		listVariablesCombo.setEnabled(false);
		
		finalValueLabel = widgetFactory.createLabel(parent, "Final Value");
		finalValueText = widgetFactory.createText(parent, "");
		finalValueText.setEnabled(false);
		
		expressionLabel = widgetFactory.createLabel(parent, "Expression");
		expressionText = widgetFactory.createText(parent, "");
		expressionText.setEnabled(false);
		
		initialValueLabel = widgetFactory.createLabel(parent, "Initial Value");
		initialValueText = widgetFactory.createText(parent, "");
		
		indexLabel.setLayoutData(createIndexLabelLayoutData());
		indexCombo.setLayoutData(createIndexComboLayoutData());
		
		forEachRadioButton.setLayoutData(createListButtonLayoutData());
		forRadioButton.setLayoutData(createFinalValueButtonLayoutData());
		whileRadioButton.setLayoutData(createExpressionButtonLayoutData());
		
		listValueLabel.setLayoutData(createListValueLabelLayoutData());
		listVariablesCombo.setLayoutData(createListVariablesComboLayoutData());
		
		finalValueLabel.setLayoutData(createFinalValueLabelLayoutData());
		finalValueText.setLayoutData(createFinalValueTextLayoutData());
		
		expressionLabel.setLayoutData(createExpressionLabelLayoutData());
		expressionText.setLayoutData(createExpressionTextLayoutData());
		
		initialValueLabel.setLayoutData(createInitialValueLabelLayoutData());
		initialValueText.setLayoutData(createInitialValueTextLayoutData());
	}
	
	private FormData createIndexLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(0, 3);
		return data;
	}
	
	private FormData createIndexComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 100);
		data.top = new FormAttachment(0, 2);
		data.width = 300;
		return data;
	}
	
	private FormData createListButtonLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 100);
		data.top = new FormAttachment(indexLabel, 2);
		data.width = 100;
		return data;
	}
	
	private FormData createFinalValueButtonLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(forEachRadioButton, 0);
		data.top = new FormAttachment(indexLabel, 2);
		data.width = 100;
		return data;
	}
	
	private FormData createExpressionButtonLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(forRadioButton, 0);
		data.top = new FormAttachment(indexLabel, 2);
		data.width = 100;
		return data;
	}
	
	private FormData createListValueLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(forEachRadioButton, 2);
		return data;
	}
	
	private FormData createListVariablesComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 100);
		data.top = new FormAttachment(forEachRadioButton, 2);
		data.width = 300;
		return data;
	}
	
	private FormData createInitialValueLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(listValueLabel, 2);
		return data;
	}
	
	private FormData createInitialValueTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 100);
		data.top = new FormAttachment(listValueLabel, 2);
		data.width = 50;
		return data;
	}
	
	private FormData createFinalValueLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(initialValueLabel, 2);
		return data;
	}
	
	private FormData createFinalValueTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 100);
		data.top = new FormAttachment(initialValueLabel, 2);
		data.width = 50;
		return data;
	}
	
	private FormData createExpressionLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(finalValueLabel, 2);
		return data;
	}
	
	private FormData createExpressionTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 100);
		data.top = new FormAttachment(finalValueLabel, 2);
		data.width = 290;
		return data;
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
	}
	
	public void widgetSelected(SelectionEvent e) {
		if(e.widget == forEachRadioButton && forEachRadioButton.getSelection()){
			loopNode.setLoopType(LoopNodeType.FOR_EACH);
			setListVariablesVisible(true);
			setFinalValueVisible(false);
			setExpressionVisible(false);
			setInitialValueVisible(false);
		}
		if(e.widget == forRadioButton && forRadioButton.getSelection()){
			loopNode.setLoopType(LoopNodeType.FOR);
			setListVariablesVisible(false);
			setFinalValueVisible(true);
			setExpressionVisible(false);
			setInitialValueVisible(true);
		}
		if(e.widget == whileRadioButton && whileRadioButton.getSelection()){
			loopNode.setLoopType(LoopNodeType.WHILE);
			setListVariablesVisible(false);
			setFinalValueVisible(false);
			setExpressionVisible(true);
			setInitialValueVisible(true);
		}
	}
	
	private void setListVariablesVisible(boolean visible){
		listVariablesCombo.setEnabled(visible);
		if(!visible){
			listVariablesCombo.setText("");
		} else{
			listVariablesCombo.setText(loopNode.getListName() == null ? "" : loopNode.getListName());
		}
	}
	
	private void setFinalValueVisible(boolean visible){
		finalValueText.setEnabled(visible);
		if(!visible){
			finalValueText.setText("");
		} else {
			int finalValue = loopNode.getFinalValue();
			String finalValueString = Integer.toString(finalValue);
			finalValueText.setText( (loopNode.getExpression() == null || finalValue < 0 ) ? "" : finalValueString );
		}
	}
	
	private void setExpressionVisible(boolean visible){
		expressionText.setEnabled(visible);
		expressionText.setText(loopNode.getExpression() == null ? "" : loopNode.getExpression());
	}
	
	private void setInitialValueVisible(boolean visible){
		initialValueText.setEnabled(visible);
		if(!visible){
			initialValueText.setText("");
		} else {
			initialValueText.setText( Integer.toString(loopNode.getInitialValue()) );
		}
	}
	
	private void updateIndex() {
		String str = indexCombo.getText();
		loopNode.setIndex(str == null ? "" : str);
		String expressionValue = generateExpression("updateIndex", loopNode);
		loopNode.setExpression(expressionValue);
		expressionText.setText(expressionValue);
	}
	
	private void updateListVariable() {
		String str = listVariablesCombo.getText();
		if(str != null && !str.trim().isEmpty()){
			loopNode.setListName(str);
			loopNode.setInitialValue(0);
			String expressionValue = generateExpression("updateListVariable", loopNode);
			loopNode.setExpression(expressionValue);
			expressionText.setText(expressionValue);
		}
	}
	
	private void updateFinalValue() {
		String expressionValue = generateExpression("updateFinalValue", loopNode);
		loopNode.setExpression(expressionValue);
		loopNode.setListName(null);
		expressionText.setText(expressionValue);
	}
	
	private void updateExpression() {
		loopNode.setInitialValue(0);
		String expressionValue = expressionText.getText();
		loopNode.setExpression(expressionValue.trim());
		loopNode.setListName(null);
		
		validateExpression();
	}
	
	private void updateInitialValue(){
		String initialValue = initialValueText.getText();
		if(initialValue != null && !initialValue.trim().isEmpty() && isNumber(initialValue) ){
			loopNode.setInitialValue(Integer.parseInt(initialValue));
		}
	}
	
	private void validateExpression(){
		ValidationServiceFactory serviceFactory = ValidationServiceFactory.createFactory();
		IValidationService validationService = serviceFactory.createValidationService();
		NodeElementContainer container = Utility.getNodeElementContainer(this.parentPropertySection.getPart());
		List<ValidationMessage> messages = null;
		if (container instanceof WorkflowTemplate)
			messages = validationService.validateLoopNodeExpression(loopNode,
					(WorkflowTemplate) container);
		else if (container instanceof NodeDefinition)
			messages = validationService.validateLoopNodeExpression(loopNode,
					(NodeDefinition) container);

		if (messages != null && messages.size() > 0) {

			try {
				String msg = ValidationMessages.getInstance().getMessage(messages.get(0));
				MessageDialog.openError(Display.getCurrent().getActiveShell(),
						"Error in Decision Expression", msg);
			} catch (InvalidPropertiesFormatException e) {
				logger.error("Error occured while opening the Message Dialog", e);
				throw new UIException("Error occured while opening the Message Dialog");
			} catch (IOException e) {
				logger.error("Error occured while opening the Message Dialog", e);
				throw new UIException("Error occured while opening the Message Dialog");
			}
		}
	}
	
	private String generateExpression(String operation, LoopNode loopNode){
		String expression = loopNode.getExpression() != null ? loopNode.getExpression() : "";
		if("updateIndex".equals(operation)){
			if( LoopNodeType.FOR.equals(loopNode.getLoopType()) ){
				expression = loopNode.getIndex() + " < " + finalValueText.getText() ;
			} else if( LoopNodeType.FOR_EACH.equals(loopNode.getLoopType()) ){
				expression = loopNode.getListName() + "[" + loopNode.getIndex() + "] != null";
			} 
		}
		if("updateListVariable".equals(operation)){
			expression = listVariablesCombo.getText() + "[" + loopNode.getIndex() + "] != null";
		}
		if("updateFinalValue".equals(operation)){
			expression = loopNode.getIndex() + " < " + finalValueText.getText() ;
		}
		return expression;
	}
	
	private boolean isNumber(String string){
		for (int i = 0; i < string.length(); i++) {
			if (!Character.isDigit(string.charAt(i)))
				return false;
		}
		return true;
	}

	public void focusGained(FocusEvent e) {
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == indexCombo) {
			updateIndex();
		} else if (e.widget == listVariablesCombo) {
			updateListVariable();
		} else if (e.widget == finalValueText) {
			updateFinalValue();
		} else if (e.widget == expressionText) {
			updateExpression();
		} else if (e.widget == initialValueText) {
			updateInitialValue();
		}
	}
	
}
