package com.tandbergtv.watchpoint.studio.ui.util;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.collections.CollectionUtils;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.variables.VariablesPlugin;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.editparts.AbstractEditPart;
import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.jbpm.gd.common.editor.Editor;
import org.jbpm.gd.common.model.NamedElement;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.notation.AbstractNotationElement;
import org.jbpm.gd.common.notation.NodeContainer;
import org.jbpm.gd.common.notation.NotationElement;
import org.jbpm.gd.common.part.NotationElementGraphicalEditPart;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.common.xml.XmlAdapterFactory;
import org.jbpm.gd.jpdl.model.AbstractNode;
import org.jbpm.gd.jpdl.model.Fork;
import org.jbpm.gd.jpdl.model.Join;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.Transition;
import org.jbpm.gd.jpdl.model.Variable;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.ServiceErrorCode;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceValidationException;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;
import com.tandbergtv.watchpoint.studio.ui.editor.NodeDefinitionEditor;
import com.tandbergtv.watchpoint.studio.ui.editor.WPNodeGraphicalEditPart;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.ui.editor.expression.ExpressionUtil;
import com.tandbergtv.watchpoint.studio.ui.model.AssignNode;
import com.tandbergtv.watchpoint.studio.ui.model.ManualTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPAbstractNode;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.usermgmt.UserManagementException;
import com.tandbergtv.watchpoint.studio.usermgmt.UserManager;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.rules.expression.VariableExpressionValidator;

/**
 * Utility class that contains common utility methods.
 * 
 * @author Imran Naqvi
 * 
 */
public final class Utility {

    private static final Logger logger = Logger.getLogger(Utility.class);

    public static final String PLUGIN_XML = "plugin.xml";

    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    public static final String PLUGIN_ID = "org.jbpm.gd.jpdl";

    private static final String ICONS_PATH = "icons/";

    /* ID of the watchpoint template editor */
    public static final String TEMPLATE_EDITOR_ID = "org.jbpm.gd.jpdl.editor";

    /* ID of the watchpoint node definition editor */
    public static final String NODEDEF_EDITOR_ID = "com.tandbergtv.jpdl.nodedefinitioneditor";

    /* ID of the watchpoint Resource Type Editor */
    public static final String RESOURCE_TYPE_EDITOR_ID = "com.tandbergtv.watchpoint.studio.ui.editor.ResourceTypeEditor";

    /* ID of the watchpoint node definition explorer view */
    @Deprecated
    public static final String ID_NODEDEF_EXPLORER = "com.tandbergtv.gd.NodeDefinitionExplorer";

    public static final String ID_RESOURCE_TYPE = "com.tandbergtv.gd.ResourceTypeExplorer";

    public static final String ID_RESOURCE_GROUP = "com.tandbergtv.gd.ResourceGroup";

    public static final String ID_OVERVIEW = "org.jbpm.gd.jpdl.overview";

    private static final String NODE_NAME_ATTRIBUTE_KEY = "NODE_NAME";

    private static final String ID_JPDL_DEBUG_MODEL_KEY = "ID_JPDL_DEBUG_MODEL";

    public static final String PROBLEM_MARKER_TYPE = "org.jbpm.gd.jpdl.templateProblem";

    private static IWorkbenchWindow window;

    /**
     * Private Ctor.
     */
    private Utility() {

    }

    public static final String WORKFLOW_SYSTEM_ID = "01";

    /**
     * 		Gets the system UID based on the message UID
     * @param messageUID
     */
    public static String getSystemIDByMessageUID(String messageUID) {
        String systemID = null;

        messageUID = messageUID.trim();

        if (messageUID.length() > 4) {
            /* Example: 012499
			   Where: 01 = Source SystemID
			          24 = Destination SystemID
			          99 = Message ID
             */
            systemID = messageUID.substring(0, 2);
            if (WORKFLOW_SYSTEM_ID.equals(systemID)) {
                systemID = messageUID.substring(2, 4);
            }
        }
        return systemID;
    }

    /**
     * Closes the editor whose id is passed if its open.
     * 
     * @param editorId
     *            The id of the editor to close
     * @param save
     *            <code>true</code> to save the editor contents if required (recommended), and
     *            <code>false</code> to discard any unsaved changes
     * 
     * @return <code>true</code> if the editor was successfully closed, and <code>false</code>
     *         if the editor is still open
     */
    public static boolean closeEditor(String editorId, boolean save) {
        if (window == null || window.getActivePage() == null) {
            return true;
        }
        IEditorPart editor = getFirstEditor(editorId);
        /* close editor */
        if (editor == null) {
            return true;
        }
        return window.getActivePage().closeEditor(editor, save);
    }

    /**
     * Finds the editor whose id is passed, if its open.
     * 
     * @param editorId
     *            The id of the editor
     * @return editor whose id is passed
     */
    public static IEditorPart getFirstEditor(String editorId) {
        if (window == null || window.getActivePage() == null) {
            return null;
        }

        List<IEditorPart> editorParts = getEditors(editorId);

        return editorParts.size() > 0 ? editorParts.get(0) : null;
    }

    public static IEditorPart getEditor(String editorId, IResource fileResource) {
        IResource resource = null;
        for (IEditorPart editor : getEditors(editorId)) {
            resource = (IResource) editor.getEditorInput().getAdapter(IResource.class);
            if (resource != null && resource.equals(fileResource)) {
                return editor;
            }
        }
        return null;
    }

    public static List<IEditorPart> getEditors(String editorId) {
        List<IEditorPart> editorParts = new ArrayList<IEditorPart>();

        for (IEditorReference ref : window.getActivePage().getEditorReferences()) {
            IEditorPart editor = ref.getEditor(false);
            /* Skip if current editor is not the editor we are looking for */
            if (editor == null || !editor.getEditorSite().getId().equals(editorId)) {
                continue;
            }
            /* Skip if editor does not have an active page */
            if (editor.getSite().getWorkbenchWindow().getActivePage() == null) {
                continue;
            }
            editorParts.add(editor);
        }
        return editorParts;
    }

    /**
     * Returns the node element container associated with the editor, if there is one. Only the
     * template editor and the node definition editor have node element containers associated with
     * them.
     * 
     * @param part
     *            the editor whose container is required
     * 
     * @return the node element container associated with the editor if there is one, null otherwise
     */
    public static NodeElementContainer getNodeElementContainer(IWorkbenchPart part) {
        NodeElementContainer container = null;
        if (part instanceof WatchPointTemplateEditor) {
            container = ((WatchPointTemplateEditor) part).getProcessDefinition();
        } else if (part instanceof NodeDefinitionEditor) {
            container = ((NodeDefinitionEditor) part).getNodeDefinition();
        }

        return container;
    }

    /**
     * Creates a WorkflowTemplateDTO object from the name, jpdl xml and the gpd xml given.
     * 
     * @param id
     *            The Template database Id
     * @param version
     *            The Template version
     * @param name
     *            The Template name
     * @param xml
     *            The Process Definition xml
     * @param gpdXml
     *            The GPD xml
     * 
     * @return The WorkflowTemplateDTO created.
     */
    public static WorkflowTemplateDTO createTemplateDTO(long id, int version, String name,
            String xml, String gpdXml) {
        WorkflowTemplateDTO templateDTO = new WorkflowTemplateDTO();
        templateDTO.setId(id);
        templateDTO.setVersion(version);
        templateDTO.setName(name);
        return templateDTO;
    }

    /**
     * Creates a NodeDefinitionDTO object from the name, jpdl xml and the gpd xml given.
     * 
     * @param id
     *            The Node Definition database Id
     * @param name
     *            The Node Definition name
     * @param xml
     *            The Node Definition xml
     * @param gpdXml
     *            The GPD xml
     * 
     * @return The NodeDefinitionDTO created.
     */
    public static NodeDefinitionDTO createNodeDefinitionDTO(long id, String name, String xml,
            String gpdXml) {
        NodeDefinitionDTO nodeDefinitionDTO = new NodeDefinitionDTO();
        nodeDefinitionDTO.setId(id);
        nodeDefinitionDTO.setName(name);
        nodeDefinitionDTO.setGpd(gpdXml);
        nodeDefinitionDTO.setXml(xml);
        return nodeDefinitionDTO;
    }

    /**
     * Gets the user name.
     * 
     * @return The User Name for the current user
     */
    public static String getUserName() {
        String userId = null;

        try {
            userId = UserManager.getInstance().getCurrentUserId();
        }
        catch (UserManagementException e) {
            logger.error("Failed to get the current User Id.");
        }

        return userId;
    }

    /**
     * Creates a DOM Document from a string containing xml.
     * 
     * @param xmlString
     *            the string that contains xml
     * 
     * @return The Document constructed for the xml.
     */
    public static Document getDocumentFromString(String xml) {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            StringReader reader = new StringReader(xml);

            return factory.newDocumentBuilder().parse(new InputSource(reader));
        } catch (Exception e) {
            throw new UIException("Failed to create DOM from text", e);
        }
    }

    /**
     * Get the XML string from a given DOM document
     * 
     * @param doc
     *            The DOM Document from which the XML will be created
     * 
     * @return The XML string
     */
    public static String getXMLFromDocument(Document doc) {
        StringWriter stringWriter = new StringWriter();

        try {
            TransformerFactory tFactory = TransformerFactory.newInstance();
            Transformer transformer = tFactory.newTransformer();
            DOMSource source = new DOMSource(doc.getDocumentElement());
            StreamResult result = new StreamResult(stringWriter);
            transformer.transform(source, result);
        }
        catch (TransformerException ex) {
            throw new UIException("Could not transform DOM document.", ex);
        }

        return stringWriter.getBuffer().toString().replaceAll("&#13;", "");
    }

    /**
     * Checks to see if the given UID is for outgoing messages.
     * 
     * @param uid the UID to be checked
     * @return true if the UID is for an outgoing message; false otherwise.
     */
    public static boolean isOutgoingMessage(String uid) {
        String workflowSystemId = ApplicationProperties.getInstance().get(
                ApplicationPropertyKeys.WORKFLOW_SYSTEM_ID).toString();
        return !uid.substring(2, 4).equals(workflowSystemId);
    }

    /**
     * @return the window
     */
    public static IWorkbenchWindow getWindow() {
        return window;
    }

    /**
     * @param window
     *            the window to set
     */
    public static synchronized void setWindow(IWorkbenchWindow window) {
        Utility.window = window;
    }

    /**
     * Returns the image descriptor with the given relative path.
     * 
     * @param relativePath
     *            the path to the Image
     * 
     * @return the Image Descriptor
     */
    public static ImageDescriptor getImageDescriptor(String relativePath) {
        return getImageDescriptor(PLUGIN_ID, relativePath);
    }

    /**
     * Returns the image descriptor with the given relative path.
     * 
     * @param pluginId
     *            the plugin id
     * @param relativePath
     *            the path to the Image
     * 
     * @return the Image Descriptor
     */
    public static ImageDescriptor getImageDescriptor(String pluginId, String relativePath) {
        try {
            URL installURL = Platform.getBundle(pluginId).getEntry("/");
            URL url = new URL(installURL, ICONS_PATH + relativePath);
            return ImageDescriptor.createFromURL(url);
        }
        catch (MalformedURLException e) {
            throw new UIException("Could not open image at path " + ICONS_PATH + relativePath, e);
        }
    }


    /**
     * Shows error dialog in the UI with the given message and throws a UI exception created from
     * the given throwable to be logged.
     * 
     * @param errorMessage
     *            the message to show in the UI
     * @param e
     *            the throwable to log
     */
    public static void handleException(String errorMessage, Throwable e) {
        MessageDialog.openError(Display.getCurrent().getActiveShell(), "Error", errorMessage);
    }

    /**
     * Gets a DOM Adapter representing the XML in the given input stream.
     * 
     * @param fileName
     *            the name of the file containing the XML.
     * @param xmlStream
     *            the input stream to the file containing the XML
     * @param elementFactory
     *            the semantic element factory used to create the XmlAdapterFactory
     * @return a DOM Adapter representing the XML in the given input stream
     */
    @SuppressWarnings({ "restriction", "deprecation" })
    public static synchronized XmlAdapter getAdapterFromStream(String fileName,
            InputStream xmlStream, SemanticElementFactory elementFactory) {
        org.eclipse.wst.sse.core.internal.provisional.IStructuredModel model = null;

        try {
            model = org.eclipse.wst.sse.core.internal.model.ModelManagerImpl.getInstance()
                    .getModelForRead("adapter", xmlStream, null);

            org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel domModel = null;
            domModel = (org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel) model;

            XmlAdapterFactory factory = new XmlAdapterFactory(domModel.getDocument(),
                    elementFactory);
            XmlAdapter xmlAdapter = factory.adapt(domModel.getDocument()
                    .getDocumentElement());
            return xmlAdapter;
        }
        catch (UnsupportedEncodingException e) {
            throw new UIException("Invalid encoding in xml file " + fileName + ".", e);
        }
        catch (IOException e) {
            throw new UIException("Could not read from xml file " + fileName + " to create model.",
                    e);
        }
        finally {
            model.releaseFromRead();
        }
    }

    public static String getCurrentStudioVersion() {
        String studioVersion =
                WatchPointStudioPlugin.getDefault().getBundle().getHeaders().get(org.osgi.framework.Constants.BUNDLE_VERSION);
        return studioVersion;
    }

    /**
     * Gets a DOM Adapter representing the XML in the given string.
     * 
     * @param xmlString
     *            the string containing the XML
     * @param elementFactory
     *            the semantic element factory used to create the XmlAdapterFactory
     * @return a DOM Adapter representing the XML in the given input stream
     */
    @SuppressWarnings({ "restriction", "deprecation" })
    public static synchronized XmlAdapter getAdapterFromString(String xmlString,
            SemanticElementFactory elementFactory) {
        org.eclipse.wst.sse.core.internal.provisional.IStructuredModel model = null;

        try {
            ByteArrayInputStream xmlStream = new ByteArrayInputStream(xmlString.getBytes());
            model = org.eclipse.wst.sse.core.internal.model.ModelManagerImpl.getInstance()
                    .getModelForRead("adapter", xmlStream, null);

            org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel domModel = null;
            domModel = (org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel) model;

            XmlAdapterFactory factory = new XmlAdapterFactory(domModel.getDocument(),
                    elementFactory);
            XmlAdapter xmlAdapter = factory.adapt(domModel.getDocument()
                    .getDocumentElement());
            return xmlAdapter;
        }
        catch (UnsupportedEncodingException e) {
            throw new UIException("Invalid encoding in xml.", e);
        } catch (IOException e) {
            throw new UIException("Could not read from xml to create model.", e);
        } finally {
            if (model != null) {
                model.releaseFromRead();
            }
        }
    }

    /**
     * Reports the error in UI in a message box.
     * 
     * @param headerMessage
     *            The Header to use for the message box
     * @param e
     *            The Error to report
     */
    public static void reportError(String headerMessage, Throwable e) {
        String errorMessage = headerMessage + LINE_SEPARATOR + LINE_SEPARATOR + "Reason(s):"
                + LINE_SEPARATOR + e.getClass().getName() + LINE_SEPARATOR ;
        if (e instanceof ServiceValidationException) {
            List<ValidationMessage> validationMessages = ((ServiceValidationException) e).getValidationMessages();
            if (validationMessages != null && !validationMessages.isEmpty()) {
                StringBuffer errorMessageStrBuf = new StringBuffer();
                for (ValidationMessage validationMessage : validationMessages) {
                    try {
                        errorMessageStrBuf.append(ValidationMessages.getInstance().getMessage(
                                validationMessage));
                        errorMessageStrBuf.append(LINE_SEPARATOR);
                    }
                    catch (Exception subException) {
                        String subExceptionErrorMessage = "Could not read validation message "
                                + validationMessage.getCode() + " from property file.";
                        errorMessageStrBuf.append(subExceptionErrorMessage);
                        errorMessageStrBuf.append(LINE_SEPARATOR);
                    }
                }
                errorMessage += errorMessageStrBuf.toString();
                org.jbpm.gd.jpdl.Logger.logError(errorMessage, e);
            }
            else {
                org.jbpm.gd.jpdl.Logger.logError(errorMessage, e);
                errorMessage += e.getMessage();
            }
        }
        else if (e instanceof ServiceException) {
            ServiceException se = (ServiceException) e;
            ServiceErrorCode code = se.getServiceErrorCode();
            errorMessage = ServiceErrors.getInstance().getProperty(code.getCode());
            org.jbpm.gd.jpdl.Logger.logError(errorMessage, e);
        }
        else {
            org.jbpm.gd.jpdl.Logger.logError(errorMessage, e);
            errorMessage += e.getMessage();
        }

        logger.error("Failure in UI: " + errorMessage, e);
        MessageDialog.openError(Display.getCurrent().getActiveShell(), "Error", errorMessage);
    }

    public static void reportOperationCompleted(String operation) {
        MessageDialog.openInformation(Display.getCurrent().getActiveShell(), operation, "Operation completed successfully.");
    }

    /**
     * This method retrieves the defined variables in start state node
     * 
     * @param container
     *            The Variable Container
     * 
     * @return Array of Strings
     */
    public static List<WPVariable> getVariables(NodeElementContainer container) {
        List<WPVariable> wpVariables = new ArrayList<WPVariable>();

        if (container instanceof WorkflowTemplate) {
            StartState startState = ((WorkflowTemplate) container).getStartState();
            if (startState != null) {
                wpVariables.addAll(ModelHandler.getStartStateValidVariables(startState));
            }

        } else if (container instanceof NodeDefinition) {
            NodeDefinition nodeDefinition = (NodeDefinition) container;
            /*
             * Since node definition variables just have mapped names we need to create new
             * variables with names
             */
            for (WPVariable var : nodeDefinition.getVariables()) {
                WPVariable tempVar = new WPVariable();
                tempVar.setName(var.getMappedName());
                wpVariables.add(tempVar);
            }
        }

        return wpVariables;
    }

    public static IProblem[] isScriptValid(IProject project, DataType returnType, String script, Variable[] contextVars) {
        if (!script.trim().isEmpty()) {
            VariableExpressionValidator validator = new VariableExpressionValidator();
            validator.setProject(project);
            if (returnType != null) {
                validator.setReturnType(returnType);
            }
            Collection<Variable> contextVarsStr = new ArrayList<Variable>();
            CollectionUtils.addAll(contextVarsStr, contextVars);
            validator.setContextVariables(ExpressionUtil.toJavaVariableStr(contextVarsStr));
            validator.validateExpression(script);
            return validator.getProblems();
        }
        return null;
    }

    public static Set<String> getUsedVariablesInScript(IProject project, DataType returnType, String script, Variable[] contextVars) {
        Set<String> result = new HashSet<String>();
        if (script != null && !script.trim().isEmpty()) {
            VariableExpressionValidator validator = new VariableExpressionValidator();
            validator.setProject(project);
            if (returnType != null) {
                validator.setReturnType(returnType);
            }
            Collection<Variable> contextVarsStr = new ArrayList<Variable>();
            CollectionUtils.addAll(contextVarsStr, contextVars);
            validator.setContextVariables(ExpressionUtil.toJavaVariableStr(contextVarsStr));
            return validator.validateExpression(script);
        }
        return result;
    }

    /**
     * Gets the active editor in the current active page.
     * 
     * 
     * @return editor the editor which is currently active
     */
    public static IEditorPart getActiveEditor() {
        if (window == null || window.getActivePage() == null) {
            return null;
        }

        return window.getActivePage().getActiveEditor();
    }

    public static IProject getProjectFromEditor(Editor editor) {
        IProject project = null;
        if (editor != null && editor.getEditorInput() instanceof IFileEditorInput) {
            project = ((IFileEditorInput) editor.getEditorInput()).getFile().getProject();
        }
        return project;
    }

    public static <T> List<AbstractNotationElement> getNotationElementsOfType(IEditorPart editorPart, NodeContainer container, Class<T> type) {
        if (container == null) {
            Editor editor = getEditor(editorPart);
            if (editor != null) {
                container = editor.getRootContainer();
            } else {
                return null;
            }
        }
        List<AbstractNotationElement> result = new ArrayList<AbstractNotationElement>();

        for (int i = 0; i < container.getNodes().size(); i++) {
            AbstractNotationElement node = (AbstractNotationElement) container.getNodes().get(i);
            T semanticElement = get(node.getSemanticElement(), type);
            if (semanticElement != null) {
                result.add(node);
            }
            NodeContainer newContainer = getNodeContainer(node);
            if (newContainer != null) {
                result.addAll(getNotationElementsOfType(null, newContainer, type));
            }
        }

        return result;
    }

    public static List<AbstractNotationElement> getNotationElementsOfTypeNamedElement(IEditorPart editorPart) {
        return getNotationElementsOfType(editorPart, null, NamedElement.class);
    }

    public static List<AbstractNotationElement> getNotationElementsOfTypeNodeDefinition(IEditorPart editorPart) {
        return getNotationElementsOfType(editorPart, null, NodeDefinition.class);
    }

    public static List<AbstractNotationElement> getNotationElementsOfTypeFork(WatchPointTemplateEditor editorPart) {
        return getNotationElementsOfType(editorPart, null, Fork.class);
    }

    public static AbstractNotationElement getNotationElementByName(IEditorPart editorPart, String name) {
        for (AbstractNotationElement notationElement : getNotationElementsOfTypeNamedElement(editorPart)) {
            NamedElement namedElement = getNamedElement(notationElement);
            if (namedElement.getName().equals(name)) {
                return notationElement;
            }
        }
        return null;
    }

    public static NodeDefinition getNodeDefinitionByName(IEditorPart editorPart, NodeContainer container, String name) {
        AbstractNotationElement notation = getNotationElementByName(editorPart, name);
        return getNodeDefinition(notation);
    }

    public static void refreshEditPartByNodeName(IEditorPart editorPart, String nodeName) {
        Editor editor = Utility.getEditor(editorPart);
        NotationElement notationElement = Utility.getNotationElementByName(editorPart, nodeName);
        EditPart editPart = (EditPart) editor.getGraphicalViewer().getEditPartRegistry().get(notationElement);
        WPNodeGraphicalEditPart wpEditPart = getEditPart(editPart);
        editPart.refresh();
        if (wpEditPart != null) {
            wpEditPart.repaintHeader();
        }
    }

    public static List<NodeDefinition> getNodeDefinitions(IEditorPart editorPart) {
        List<NodeDefinition> result = new ArrayList<NodeDefinition>();
        for (AbstractNotationElement notationElement : getNotationElementsOfTypeNodeDefinition(editorPart)) {
            NodeDefinition nodeDefinition = getNodeDefinition(notationElement);
            result.add(nodeDefinition);
        }
        return result;
    }

    public static AbstractEditPart getEditPart(ISelection selection) {
        Object firstElement = ((IStructuredSelection) selection).getFirstElement();
        return get(firstElement, AbstractEditPart.class);
    }

    public static NodeDefinition getNodeDefinition(ISelection selection) {
        EditPart editPart = getEditPart(selection);
        SemanticElement semanticElement = getSemanticElement(editPart);
        return get(semanticElement, NodeDefinition.class);
    }

    public static ManualTaskNode getManualTaskNode(ISelection selection) {
        EditPart editPart = getEditPart(selection);
        SemanticElement semanticElement = getSemanticElement(editPart);
        return get(semanticElement, ManualTaskNode.class);
    }

    public static boolean hasBreakpoint(NamedElement namedElement, IEditorInput editorInput) {
        return hasBreakpoint(namedElement.getName(), editorInput);
    }

    public static boolean hasBreakpoint(String nodeName, IEditorInput editorInput) {
        try {
            String idJpdlDebugModel = VariablesPlugin.getDefault().getStringVariableManager()
                    .getValueVariable(ID_JPDL_DEBUG_MODEL_KEY).getValue().toString();
            String nodeNameAttribute = VariablesPlugin.getDefault().getStringVariableManager()
                    .getValueVariable(NODE_NAME_ATTRIBUTE_KEY).getValue().toString();
            IResource resource = (IResource) editorInput.getAdapter(IResource.class);

            IBreakpoint[] breakpoints = DebugPlugin.getDefault().getBreakpointManager()
                    .getBreakpoints(idJpdlDebugModel);
            for (IBreakpoint breakpoint : breakpoints) {
                if (resource.equals(breakpoint.getMarker().getResource())) {
                    if (breakpoint.getMarker().getAttribute(nodeNameAttribute).equals(nodeName)) {
                        return true;
                    }
                }
            }

            return false;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public static List<Fork> getForkNodes(WatchPointTemplateEditor editor) {
        List<Fork> forkNodes = new ArrayList<Fork>();
        for (NotationElement notationElement : Utility.getNotationElementsOfTypeFork(editor)) {
            Fork fork = Utility.getFork(notationElement);
            if (fork != null) {
                forkNodes.add(fork);
            }
        }
        return forkNodes;
    }

    public static boolean isNodeInForkPaths(Editor editor, Fork fork, Transition[] transitions, String nodeName) {
        if (transitions == null) {
            transitions = fork.getTransitions();
        }

        AbstractNotationElement notationDestination = null;
        AbstractNode node = null;
        for (Transition transition : transitions) {
            notationDestination = getNotationElementByName(editor, transition.getTo());
            node = get(notationDestination.getSemanticElement(), AbstractNode.class);
            if (node == null || node instanceof Join) {
                continue;
            } else if (node.getName().equals(nodeName)) {
                return true;
            } else if (isNodeInForkPaths(editor, null, node.getTransitions(), nodeName)){
                return true;
            }
        }
        return false;
    }

    public static boolean isNodeInForkPath(WatchPointTemplateEditor editor, Transition transition, String nodeName) {
        AbstractNotationElement notationDestination = getNotationElementByName(editor, transition.getTo());
        AbstractNode node = get(notationDestination.getSemanticElement(), AbstractNode.class);
        if (node == null || node instanceof Join) {
            return false;
        } else if (node.getName().equals(nodeName)) {
            return true;
        } else {
            for (Transition nextTransition : node.getTransitions()) {
                return isNodeInForkPath(editor, nextTransition, nodeName);
            }
        }
        return false;
    }

    public static boolean forkPathHasBreakpoint(WatchPointTemplateEditor editor, Transition transition, IEditorInput editorInput) {
        AbstractNotationElement notationDestination = getNotationElementByName(editor, transition.getTo());
        AbstractNode node = get(notationDestination.getSemanticElement(), AbstractNode.class);
        if (node == null || node instanceof Join) {
            return false;
        } else if (hasBreakpoint(node.getName(), editorInput)) {
            return true;
        } else {
            for (Transition nextTransition : node.getTransitions()) {
                return forkPathHasBreakpoint(editor, nextTransition, editorInput);
            }
        }
        return false;
    }

    public static <T> T get(Object target, Class<T> clazz) {
        if (target != null && clazz.isAssignableFrom(target.getClass())) {
            return (T) target;
        }
        return null;
    }

    public static NodeDefinition getNodeDefinition(AbstractNotationElement node) {
        return get(node.getSemanticElement(), NodeDefinition.class);
    }

    public static Editor getEditor(IEditorPart editorPart) {
        return get(editorPart, Editor.class);
    }

    public static Fork getFork(NotationElement notationElement) {
        return get(notationElement.getSemanticElement(), Fork.class);
    }

    public static WatchPointTemplateEditor getEditor(IWorkbenchPart part) {
        return get(part, WatchPointTemplateEditor.class);
    }

    public static WPNodeGraphicalEditPart getEditPart(EditPart editPart) {
        return get(editPart, WPNodeGraphicalEditPart.class);
    }

    public static NodeContainer getNodeContainer(AbstractNotationElement node) {
        return get(node, NodeContainer.class);
    }

    public static NamedElement getNamedElement(AbstractNotationElement notationElement) {
        return get(notationElement.getSemanticElement(), NamedElement.class);
    }

    public static NamedElement getNamedElement(ISelection selection) {
        SemanticElement semanticElement = getSemanticElement(selection);
        return getNamedElement(semanticElement);
    }

    public static NamedElement getNamedElement(SemanticElement semanticElement) {
        return get(semanticElement, NamedElement.class);
    }

    public static SemanticElement getSemanticElement(ISelection selection) {
        EditPart editPart = getEditPart(selection);
        return getSemanticElement(editPart);
    }

    public static SemanticElement getSemanticElement(EditPart editPart) {
        NotationElementGraphicalEditPart notationElementEditPart = get(editPart, NotationElementGraphicalEditPart.class);
        if (notationElementEditPart != null) {
            return notationElementEditPart.getNotationElement().getSemanticElement();
        }
        if (editPart != null) {
            return get(editPart.getModel(), SemanticElement.class);
        }
        return null;
    }

    public static AssignNode getAssignNode(ISelection selection) {
        SemanticElement semanticElement = getSemanticElement(selection);
        return getAssignNode(semanticElement);
    }

    public static AssignNode getAssignNode(SemanticElement semanticElement) {
        return get(semanticElement, AssignNode.class);
    }

    public static StartState getStartState(SemanticElement semanticElement) {
        return get(semanticElement, StartState.class);
    }

    public static List<WPVariable> getVariables(IWorkbenchPart part) {
        NodeElementContainer nodeContainer = getNodeElementContainer(part);
        return getVariables(nodeContainer);
    }

    public static String getTemplateLocation(WPAbstractNode target) {
        return getTemplate(target).getLocation();
    }

    public static WorkflowTemplate getTemplate(WPAbstractNode target) {
        WorkflowTemplate template = null;
        while (template == null && target != null && target.getParent() != null) {
            template = get(target.getParent(), WorkflowTemplate.class);
            if (template != null) {
                return template;
            }
            return getTemplate((WPAbstractNode) target.getParent());
        }

        return null;
    }

    public static NodeGroup getNodeGroup(WPAbstractNode target) {
        NodeGroup nodeGroup = null;
        while (nodeGroup == null && target != null && target.getParent() != null) {
            nodeGroup = get(target.getParent(), NodeGroup.class);
            if (nodeGroup != null) {
                return nodeGroup;
            }
            if (target.getParent() instanceof WorkflowTemplate) {
                return null;
            }
            return getNodeGroup((WPAbstractNode) target.getParent());
        }

        return null;
    }

    public static NodeDefinition getNodeDefinition(WPAbstractNode target) {
        NodeDefinition nodeDefinition = null;
        while (nodeDefinition == null && target != null && target.getParent() != null) {
            nodeDefinition = get(target.getParent(), NodeDefinition.class);
            if (nodeDefinition != null) {
                return nodeDefinition;
            }
            if (target.getParent() instanceof WorkflowTemplate) {
                return null;
            }
            return getNodeDefinition((WPAbstractNode) target.getParent());
        }

        return null;
    }

    public static NodeElementContainer getTemplateOrSuperState(WPAbstractNode target) {
        NodeDefinition nodeDefinition = getNodeDefinition(target);
        if (nodeDefinition != null) {
            return nodeDefinition;
        }
        WorkflowTemplate template = getTemplate(target);
        if (template != null) {
            return template;
        }

        return null;
    }

    /**
     * Sort the {@link WPVariable} inside the input {@link IWorkbenchPart} base on the alphabetical order or the
     * variable names, <b>Case-insentive!</b>
     * 
     * @param part
     * @return
     */
    public static List<WPVariable> getSortedVariables(IWorkbenchPart part) {
        List<WPVariable> variables = getVariables(part);
        Collections.sort(variables, new Comparator<WPVariable>() {

            @Override
            public int compare(WPVariable o1, WPVariable o2) {
                if (o1 == null) {
                    return 1;
                }
                if (o1.getName() == null) {
                    return 1;
                }
                if (o2 == null) {
                    return -1;
                }
                if (o2.getName() == null) {
                    return -1;
                }
                return o1.getName().compareToIgnoreCase(o2.getName());
            }
        });
        return variables;
    }

    public static String[] toNameArray(List<WPVariable> variables) {
        return toVariableArray(variables, "name", null);
    }

    public static String[] toMappedNameArray(List<WPVariable> variables) {
        return toVariableArray(variables, "mappedName", null);
    }

    public static String[] toNameArray(List<WPVariable> variables, DataType type) {
        return toVariableArray(variables, "name", type);
    }

    public static String[] toMappedNameArray(List<WPVariable> variables, DataType type) {
        return toVariableArray(variables, "mappedName", type);
    }

    public static String[] toVariableArray(List<WPVariable> variables, String property, DataType type) {
        if (variables == null) {
            return new String[0];
        }
        String[] result = new String[variables.size()];
        for (int i = 0; i < variables.size(); i++) {
            if (type != null && !type.toString().equals(variables.get(i).getType())) {
                continue;
            }
            Object propertyValue = getProperty(variables.get(i), property);
            result[i] = (String) propertyValue;
        }

        Arrays.sort(result, new Comparator<String>() {
            @Override
            public int compare(String strA, String strB) {
                return strA.compareToIgnoreCase(strB);
            }
        });

        return result;
    }

    private static Object getProperty(Object target, String property) {
        try {
            return new PropertyDescriptor(property, Variable.class).getReadMethod().invoke(target);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        } catch (IntrospectionException e) {
            throw new RuntimeException(e);
        }
    }

    public static List<WPVariable> getVariables(WPAbstractNode target) {
        WorkflowTemplate template = getTemplate(target);
        return Utility.getVariables(template);
    }

    public static StartState extractStartState(String templatePath) {
        try {
            Document doc = XMLDocumentUtility.loadFileFromWorkspace(templatePath);

            XPath xpath = XPathFactory.newInstance().newXPath();

            NodeList variables = (NodeList) xpath.evaluate("//start-state/task/controller/variable",
                    doc.getDocumentElement(), XPathConstants.NODESET);

            SemanticElementFactory factory = new SemanticElementFactory(Utility.TEMPLATE_EDITOR_ID);
            List<WPVariable> variableList = new ArrayList<WPVariable>();
            WPVariable variable = null;
            for (int i = 0; i < variables.getLength(); i++) {
                Element e = (Element) variables.item(i);
                String access = e.getAttribute("access");
                String mappedName = e.getAttribute("mapped-name");
                String type = e.getAttribute("type");
                String name = e.getAttribute("name");
                String value = e.getAttribute("value");

                variable = (WPVariable) factory.createById(SemanticElementConstants.VARIABLE_SEID);
                variable.setName(name);
                variable.setMappedName(mappedName);
                variable.setType(type);
                variable.setAccess(access);
                variable.setValue(value);

                variableList.add(variable);
            }

            return ModelHandler.createStartStateWithVariables(variableList);
        } catch (Exception e1) {
            return null;
        }
    }

    /**
     * Check if the file with the given URL exist.
     * 
     * @param url
     * @return
     */
    public static boolean isFileExist(String url) {
        return ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(url)).exists();
    }

    public static String readFile(String url) {
        try {
            InputStream is = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(url)).getContents();
            java.util.Scanner s = new java.util.Scanner(is).useDelimiter("\\A");
            return s.hasNext() ? s.next() : "";
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public static String getNextAvailableVariableName(Table table) {
        int runner = 1;
        String suggestion = "variable" + runner;

        boolean nameIsInUse;
        do {
            nameIsInUse = false;
            for (TableItem tableItem : table.getItems()) {
                WPVariable variable = (WPVariable) tableItem.getData();
                if (suggestion.equals(variable.getName())) {
                    runner++;
                    suggestion = "variable" + runner;
                    nameIsInUse = true;
                    break;
                }
            }
        } while (nameIsInUse);

        return suggestion;
    }

    public static List<DataType> getWPDataTypes() {
        List<DataType> result = new ArrayList<DataType>();
        for (DataType dataType : DataType.values()) {
            if (dataType.isListable()) {
                result.add(dataType);
            }
        }

        return result;
    }

    public static String[] toVariableTypeName(List<DataType> wpDataTypes) {
        if (wpDataTypes == null) {
            return new String[0];
        }
        String[] result = new String[wpDataTypes.size()];
        for (int i = 0; i < wpDataTypes.size(); i++) {
            result[i] = wpDataTypes.get(i).toString();
        }

        Arrays.sort(result, new Comparator<String>() {
            @Override
            public int compare(String strA, String strB) {
                return strA.compareToIgnoreCase(strB);
            }
        });

        return result;
    }

    public static StartState getStartState(ISelection selection) {
        SemanticElement semanticElement = getSemanticElement(selection);
        return getStartState(semanticElement);
    }

    public static String readResourceTypeName(String fullPath) {
        try {
            Document doc = XMLDocumentUtility.loadFileFromWorkspace(fullPath);

            XPath xpath = XPathFactory.newInstance().newXPath();
            NodeList parameters  = (NodeList) xpath.evaluate(
                    "/plugin/extension[@point-id='resourcetype']/parameter[@id='name']", doc.getDocumentElement(),
                    XPathConstants.NODESET);
            for (int i = 0; i < parameters.getLength();) {
                Node node = parameters.item(i);
                return node.getAttributes().getNamedItem("value").getNodeValue();
            }
            return null;

        } catch (XPathExpressionException e) {
            return null;

        } catch (Exception e) {
            return null;

        }
    }

    public static String readMessageNodeUid(String fullPath) {
        try {
            Document doc = XMLDocumentUtility.loadFileFromWorkspace(fullPath);

            XPath xpath = XPathFactory.newInstance().newXPath();
            NodeList sends  = (NodeList) xpath.evaluate(
                    "/nodeDefinition/auto-task/action[@class='com.tandbergtv.workflow.exe.message.MessageEmitter']/send", doc.getDocumentElement(),
                    XPathConstants.NODESET);
            for (int i = 0; i < sends.getLength();) {
                Node node = sends.item(0);
                return node.getFirstChild().getNodeValue().trim();
            }

            NodeList receives  = (NodeList) xpath.evaluate(
                    "/nodeDefinition/auto-task/action[@class='com.tandbergtv.workflow.exe.message.MessageReceiver']/receive", doc.getDocumentElement(),
                    XPathConstants.NODESET);

            for (int i = 0; i < receives.getLength();) {
                Node node = receives.item(i);
                return node.getFirstChild().getNodeValue().trim();
            }

            return null;

        } catch (XPathExpressionException e) {
            return null;

        } catch (Exception e) {
            return null;

        }
    }

    public static String readSuperstateName(String fullPath) {
        try {
            Document doc = XMLDocumentUtility.loadFileFromWorkspace(fullPath);

            XPath xpath = XPathFactory.newInstance().newXPath();
            NodeList superstates  = (NodeList) xpath.evaluate(
                    "/nodeDefinition/super-state", doc.getDocumentElement(),
                    XPathConstants.NODESET);
            for (int i = 0; i < superstates.getLength();) {
                Node node = superstates.item(i);
                return node.getAttributes().getNamedItem("name").getNodeValue();
            }
            return null;

        } catch (XPathExpressionException e) {
            return null;

        } catch (Exception e) {
            return null;

        }
    }

    public static String getDescriptionFromPluginXml(String projectName, String uid) {
        try {
            Document doc = XMLDocumentUtility.loadFileFromWorkspace(projectName + "/plugin.xml");

            XPath xpath = XPathFactory.newInstance().newXPath();
            NodeList uidMessages = (NodeList) xpath
                    .evaluate(
                            new StringBuilder(
                                    "/plugin/extension[@point-id='resourcetype']/parameter[@id='communication']/parameter[@id='messages']/parameter[@id='message']/parameter[@id='uid' and @value=")
                            .append(uid).append("]").toString(), doc.getDocumentElement(),
                            XPathConstants.NODESET);
            for (int i = 0; i < uidMessages.getLength(); i++) {
                Node node = uidMessages.item(i);
                NodeList messages = node.getParentNode().getChildNodes();
                for (int j = 0; j < messages.getLength(); j++) {
                    Node message = messages.item(j);
                    if (message.getAttributes() != null && message.getAttributes().getNamedItem("id").getNodeValue().equals("description")) {
                        return message.getAttributes().getNamedItem("value").getNodeValue();
                    }
                }
            }
            return null;

        } catch (XPathExpressionException e) {
            return null;

        } catch (Exception e) {
            return null;

        }
    }
}