package com.tandbergtv.watchpoint.studio.ui.wizard;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.ui.util.UIValidator;

public class MessageWizardPage extends WizardPage {
	private static final String DEFAULT_TIME_OUT = "00:05:00";
	
	private Text nameText, uidManualEntryText, timeoutText, descriptionText;
	private Group uidGroup, typeGroup;
	private Button uidAutoGenerateButton, uidManualEntryButton, typeIncomingButton, typeOutgoingButton;
	private Combo reqElementCombo;
	private Label reqElementLabel, uidNumber;
	private ModifyListener validationListener;
	private ArrayList<String> comboElements;
	
	private String rtSystemId;
	private String uidPart;
	
	public void setRtSystemId(String systemId){
		this.rtSystemId = systemId;
	}
	
	
	public MessageWizardPage() {
		super("Message");
		setTitle("Create Message");
		setDescription("Create a new Message");
		validationListener = new ModifyListener() {
			public void modifyText(ModifyEvent arg0) {
				validate();
			}
		};
	}
	
	/**
	 * 
	 * @return Message Name (null if the message name is empty string) 
	 */
	public String getMessageName() {
		return convertEmptyToNull(nameText.getText());
	}
	
	public Message getMessageObject() {
		Message message = new Message();
		message.setName(convertEmptyToNull(nameText.getText()));
		message.setUIDGenerationRequired(uidAutoGenerateButton.getSelection());
		message.setUid(convertEmptyToNull(uidPart + uidManualEntryText.getText()));
		message.setIncoming(typeIncomingButton.getSelection());
		message.setProtocol(null);
		message.setDescription(convertEmptyToNull(descriptionText.getText()));		
		return message;
	}
	
	public void setComboElements(String[] elements){
		this.comboElements = new ArrayList<String>();
		for (int i = 0; i < elements.length; i++) {
			this.comboElements.add(elements[i]);
		}
	}
	
	public String getSelectedRequestElement(){
		String selectedRequestElement = null;
		if(comboElements != null && comboElements.size() > 0){
			int index = this.reqElementCombo.getSelectionIndex();
			// This is necessary because there's an empty option in the combo.
			// So the indexes are different in the comboElements array. 
			int correctIndex = index - 1;
			if(correctIndex < 0){
				selectedRequestElement = null;
			}else{
				selectedRequestElement = this.comboElements.get(correctIndex);
			}
		}
		return selectedRequestElement;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.WizardPage#isPageComplete()
	 */
	public boolean isPageComplete() {
		/*
		 * Checking if the name is entered is to make sure the finish button
		 * is disabled when the wizard starts up
		 */
		return (getErrorMessage() == null && nameText.getText().length() > 0);
	}

	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = createClientArea(parent);
		createCustomControls(composite);
		setValuesAndState();
		setControl(composite);
		Dialog.applyDialogFont(composite);
		parent.getShell().pack();
	}
	
	/*
	 * Sets the layout of the client area.
	 * 
	 * @param parent
	 *            the parent composite of the client area
	 * @return the composite representing the client area
	 */
	private Composite createClientArea(Composite parent) {
		parent.getShell().setMinimumSize(400, 550);
		Composite composite = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		composite.setLayout(layout);
		return composite;
	}
	
	private void setUidPart(String uidPart){
		uidNumber.setText(uidPart);
		this.uidPart = uidPart;
	}
	
	private void createCustomControls(Composite parent) {
		//create controls
		
		//name label
		Label nameLabel = new Label(parent, SWT.NONE);
		nameLabel.setText("Name : ");
		
		//name text
		nameText = new Text(parent, SWT.BORDER);
		nameText.setTextLimit(50);
		nameText.addModifyListener(validationListener);
		
		//uid label
		Label uidLabel = new Label(parent, SWT.NONE);
		uidLabel.setText("UID : ");
		
		//uid group
		uidGroup = new Group(parent, SWT.NULL);
		uidAutoGenerateButton = new Button(uidGroup, SWT.RADIO);
		uidAutoGenerateButton.setText("Auto Generate");
		SelectionListener uidSelectionListener = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				uidManualEntryText.setEnabled(uidManualEntryButton.getSelection());
				if(!uidManualEntryButton.getSelection())
					uidManualEntryText.setText("");
				validate();
			}
		};
		uidAutoGenerateButton.addSelectionListener(uidSelectionListener);
		uidManualEntryButton = new Button(uidGroup, SWT.RADIO);
		uidManualEntryButton.setText("Manual Entry");
		uidManualEntryButton.addSelectionListener(uidSelectionListener);
		uidManualEntryText = new Text(uidGroup, SWT.BORDER);
		uidManualEntryText.setTextLimit(2);
		uidManualEntryText.addModifyListener(validationListener);
		uidNumber = new Label(uidGroup, SWT.NONE);
		setUidPart("01" + rtSystemId);
		
		//type label
		Label typeLabel = new Label(parent, SWT.NONE);
		typeLabel.setText("Type : ");

		//type group
		typeGroup = new Group(parent, SWT.NONE);
		typeIncomingButton = new Button(typeGroup, SWT.RADIO);
		typeIncomingButton.setText("Incoming");
		SelectionListener typeSelectionListener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleTypeSelection();
			}
		};
		typeIncomingButton.addSelectionListener(typeSelectionListener);
		typeOutgoingButton = new Button(typeGroup, SWT.RADIO);
		typeOutgoingButton.setText("Outgoing");
		typeOutgoingButton.addSelectionListener(typeSelectionListener);
		
		//timeout label
		Label timeoutLabel = new Label(parent, SWT.NONE);
		timeoutLabel.setText("Time Out : ");	
		
		//timeout text
		timeoutText = new Text(parent, SWT.BORDER);
		timeoutText.setTextLimit(20);
		timeoutText.setText(DEFAULT_TIME_OUT);
		timeoutText.addModifyListener(validationListener);
		
		//description label
		Label descriptionLabel = new Label(parent, SWT.NONE);
		descriptionLabel.setText("Description : ");
		
		//description text
		descriptionText = new Text(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		descriptionText.addModifyListener(validationListener);
		
		//request element label
		reqElementLabel = new Label(parent, SWT.NONE);
		reqElementLabel.setText("Request Element : ");
		
		//request element combo
		reqElementCombo = new Combo(parent, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL | SWT.READ_ONLY);
		reqElementCombo.addModifyListener(validationListener);
		
		//layout
		
		//name text
		FormData nameTextFD = new FormData();
		nameTextFD.left = new FormAttachment(30, 2);
		nameTextFD.right = new FormAttachment(100, -10);
		nameText.setLayoutData(nameTextFD);

		//uid label
		FormData uidLabelFD = new FormData();
		uidLabelFD.top = new FormAttachment(nameText, 10);
		uidLabel.setLayoutData(uidLabelFD);

		// uidGroup
		FormData uidGroupFD = new FormData();
		uidGroupFD.left = new FormAttachment(30, 2);
		uidGroupFD.right = new FormAttachment(100, -10);
		uidGroupFD.top = new FormAttachment(nameText, 10);
		uidGroup.setLayoutData(uidGroupFD);

		uidGroup.setLayout(new FormLayout());
		
		// uidManualEntryButton
		FormData uidManualEntryButtonFD = new FormData();
		uidManualEntryButtonFD.top = new FormAttachment(uidAutoGenerateButton, 10);
		uidManualEntryButton.setLayoutData(uidManualEntryButtonFD);
		
		//uidNumber
		FormData uidNumberFD = new FormData();
		uidNumberFD.top = new FormAttachment(uidAutoGenerateButton, 12);
		uidNumberFD.right = new FormAttachment(uidManualEntryText, -5);
		uidNumber.setLayoutData(uidNumberFD);

		// uidManualEntryText
		FormData uidManualEntryTextFD = new FormData();
		uidManualEntryTextFD.top = new FormAttachment(uidAutoGenerateButton, 10);
		uidManualEntryTextFD.left = new FormAttachment(50, 10);
		uidManualEntryTextFD.right = new FormAttachment(100, -5);
		uidManualEntryText.setLayoutData(uidManualEntryTextFD);
		
		// typeLabel
		FormData typeLabelFD = new FormData();
		typeLabelFD.top = new FormAttachment(uidGroup, 10);
		typeLabel.setLayoutData(typeLabelFD);
		
		// typeGroup
		FormData typeGroupFD = new FormData();
		typeGroupFD.left = new FormAttachment(30, 2);
		typeGroupFD.right = new FormAttachment(100, -10);
		typeGroupFD.top = new FormAttachment(uidGroup, 10);
		typeGroup.setLayoutData(typeGroupFD);

		typeGroup.setLayout(new FormLayout());
		
		// typeOutgoingButton
		FormData typeOutgoingButtonFD = new FormData();
		//typeOutgoingButtonFD.right = new FormAttachment(100, -5);
		typeOutgoingButtonFD.left = new FormAttachment(typeIncomingButton, 10);
		typeOutgoingButton.setLayoutData(typeOutgoingButtonFD);
		
		FormData timeoutLabelFD = new FormData();
		//timeoutLabelFD.height = 70;
		timeoutLabelFD.top = new FormAttachment(typeGroup, 10);
		timeoutLabel.setLayoutData(timeoutLabelFD);
	
		FormData timeoutTextFD = new FormData();
		timeoutTextFD.left = new FormAttachment(30, 2);
		timeoutTextFD.right = new FormAttachment(100, -10);
		timeoutTextFD.top = new FormAttachment(typeGroup, 10);
		timeoutText.setLayoutData(timeoutTextFD);

		// descriptionLabel
		FormData descriptionLabelFD = new FormData();
		descriptionLabelFD.height = 70;
		descriptionLabelFD.top = new FormAttachment(timeoutText, 10);
		descriptionLabelFD.bottom = new FormAttachment(reqElementCombo, -10);
		descriptionLabel.setLayoutData(descriptionLabelFD);

		// descriptionText
		FormData descriptionTextFD = new FormData();
		descriptionTextFD.top = new FormAttachment(timeoutText, 10);
		descriptionTextFD.left = new FormAttachment(30, 2);
		descriptionTextFD.right = new FormAttachment(100, -10);
		descriptionTextFD.bottom = new FormAttachment(reqElementCombo, -10);
		descriptionText.setLayoutData(descriptionTextFD);
		
		// request element label
		FormData reqElementLabelFD = new FormData();
		reqElementLabelFD.bottom = new FormAttachment(100, -10);
		reqElementLabel.setLayoutData(reqElementLabelFD);
		
		// request element combo
		FormData reqElementComboFD = new FormData();
		reqElementComboFD.left = new FormAttachment(30, 2);
		reqElementComboFD.right = new FormAttachment(100, -10);
		reqElementComboFD.bottom = new FormAttachment(100, -10);
		reqElementCombo.setLayoutData(reqElementComboFD);
	}
	
	private void handleTypeSelection(){
		if(typeOutgoingButton.getSelection()){
			setUidPart("01" + rtSystemId);
			reqElementLabel.setText("Request Element : ");
		}
		if(typeIncomingButton.getSelection()){
			reqElementLabel.setText("Response Element : ");
			setUidPart(rtSystemId + "01");
		}
		validate();
	}
	
	private void setValuesAndState() {
		uidAutoGenerateButton.setSelection(true);
		uidManualEntryText.setEnabled(false);
		typeOutgoingButton.setSelection(true);
		
		if(comboElements != null && comboElements.size() > 0){
			reqElementCombo.add("");
			
			for (int i = 0; i < comboElements.size(); i++) {
				String elementName = (String) comboElements.get(i);
				reqElementCombo.add(elementName);
			}
			reqElementCombo.select(0);
		}else{
			// Do not show the request combo if there's no elements (no schema)
			reqElementLabel.setVisible(false);
			reqElementCombo.setVisible(false);
		}
	}
	
	private void validate() {
		String errorMessage = null;
		
		errorMessage = UIValidator.validateValue("Name", nameText.getText(), UIValidator.Validation.NOT_BLANK, UIValidator.Validation.TRIMMED, UIValidator.Validation.SLASH_BACK_SLASH);
		if(errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}
		
		errorMessage = UIValidator.validateValue("Timeout", timeoutText.getText(), UIValidator.Validation.NOT_BLANK, UIValidator.Validation.TRIMMED);
		if(errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}
		
		if(uidManualEntryButton.getSelection()) {
            String entryUID = uidManualEntryText.getText();
            errorMessage = UIValidator.validateValue("UID", entryUID, UIValidator.Validation.NOT_BLANK, UIValidator.Validation.TRIMMED);
            if(errorMessage != null) {
                updateErrorMessage(errorMessage);
                return;
            }
            
			if (entryUID != null && !entryUID.isEmpty()) {
				//No I don't know regex
				for (char uidChar : entryUID.toCharArray()) {
					if (!Character.isDigit(uidChar)) {
						updateErrorMessage("The UID field must only contain numbers");
						return;
					}
				}
				
				if (entryUID.length() != 2) {
					updateErrorMessage("The UID field must be exactly two digits");
					return;
				}
			}
        }
		
		errorMessage = UIValidator.validateValue("Description", descriptionText.getText(), UIValidator.Validation.TRIMMED);
		if(errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}				

		updateErrorMessage(null);
	}
	
	public void updateErrorMessage(String errorMessage) {
		setErrorMessage(errorMessage);
		getContainer().updateButtons();
	}
	
	private String convertEmptyToNull(String str) {
		return ((str.length() == 0) ? null : str);
	}
	
	public String getTimeout() {
		return timeoutText.getText();
	}
}
