/*
 * Created on Jul 10, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.jbpm.gd.common.model.NamedElement;

import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;

/**
 * A message resulting from validation of an object.
 * 
 * @author Vijay Silva
 */
public class ValidationMessage {
    private ValidationMessageType type;
	private ValidationMessageCode messageCode;
	private List<String> parameterValues = new ArrayList<String>();
	private Map<String, Object> attributes = new HashMap<String, Object>();

	/**
	 * Constructor
	 * 
	 * @param element
	 *            The element being validated causing this message
	 * @param messageType
	 *            The type of the validation message
	 * @param code
	 *            The message code
	 */
    public ValidationMessage(Object element, ValidationMessageType messageType, ValidationMessageCode code) {
        this(element, messageType, code, null);
    }

	/**
	 * Constructor
	 * 
	 * @param element
	 *            The element being validated causing this message
	 * @param type
	 *            The message type
	 * @param messageCode
	 *            The message code
	 * @param parameterValues
	 *            The parameters associated with the message
	 */
    public ValidationMessage(Object element, ValidationMessageType type, ValidationMessageCode messageCode,
            List<String> parameterValues) {
        setElement(element);
        this.type = type;
        this.messageCode = messageCode;
        this.parameterValues = parameterValues;
    }

    public Object getElement() {
        return attributes.get("element");
    }

    public void setElement(Object element) {
        this.attributes.put("element", element);
    }

    public Object get(String key) {
        return attributes.get(key);
    }

    public void set(String key, Object value) {
        attributes.put(key, value);
    }

    public ValidationMessageType getType() {
        return type;
    }

    public void setType(ValidationMessageType validationMessageType) {
        this.type = validationMessageType;
    }

    public ValidationMessageCode getMessageCode() {
        return messageCode;
    }

    public void setMessageCode(ValidationMessageCode code) {
        this.messageCode = code;
    }

	/**
	 * Get the Code value for the Validation Message Code
	 * 
	 * @return The code value
	 */
    public String getCode() {
        return (messageCode != null) ? messageCode.getCode() : null;
    }
	
    public List<String> getParameterValues() {
        return parameterValues;
    }

    public void setParameterValues(List<String> parameterValues) {
		parameterValues.clear();

		if (parameterValues != null) {
			parameterValues.addAll(parameterValues);
		}
	}

    /**
     * Passes all attributes from the map inside this ValidationMessage to the IMarker. Since IMarker cannot store other than Integer, String or Boolean, 
     * this method is useful when one wants to serialize a collection, such as the ContainerVariable list, into the IMarker using other representation. 
     */
    public void copyAttributesTo(IMarker marker) {
        try {
            for (String key : attributes.keySet()) {
                if (attributes.get(key) instanceof List) {
                    List list = (List) attributes.get(key);
                    if (!list.isEmpty() && (list.get(0) instanceof WPVariable)) {
                        serializeContainerVariableList(marker, key, list);
                    }
                } else if (attributes.get(key) instanceof NamedElement) {
                    marker.setAttribute(key.substring(0), ((NamedElement)attributes.get(key)).getName());
                } else if ((attributes.get(key) instanceof String) || (attributes.get(key) instanceof Integer)
                        || (attributes.get(key) instanceof Boolean)) {
                    marker.setAttribute(key.substring(0), attributes.get(key));
                }
            }
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    private void serializeContainerVariableList(IMarker marker, String key, List list) throws CoreException {
        WPVariable containerVariable = null;
        StringBuilder arrayStr = new StringBuilder();
        for (int i = 0; i< list.size(); i++) {
            containerVariable = (WPVariable) list.get(i);
            if (!arrayStr.toString().isEmpty()) {
                arrayStr.append("\n\n");
            }
            arrayStr.append(i).append("_name").append("=").append(containerVariable.getName()).append("\n");
            arrayStr.append(i).append("_type").append("=").append(containerVariable.getType()).append("\n");
            arrayStr.append(i).append("_access").append("=").append(containerVariable.getAccess());
        }
        marker.setAttribute(key.substring(0), arrayStr.toString());
    }

    /**
     * Method useful when one wants to deserialize the serialized representations inside the IMarker.
     */
    public static Map<String, Object> copyAttributesFrom(IMarker marker) {
        try {
            Map<String, Object> dest = new HashMap<String, Object>();
            Map<String, Object> src = marker.getAttributes();
            for (String key : src.keySet()) {
                if ((src.get(key) instanceof String)
                        && ((String) src.get(key)).matches("(?s).*_name=.*\n[0-9]+_type=.*\n[0-9]+_access=.*")) {
                    dest.put(key, deSerializeContainerVariableList((String) src.get(key)));

                } else {
                    dest.put(key, src.get(key));
                }
            }

            return dest;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static List<WPVariable> deSerializeContainerVariableList(String source) throws IOException {
        WPVariable containerVariable = null;
        Properties properties = new Properties();
        properties.load(new StringReader(source));
        int i = 0;
        List<WPVariable> result = new ArrayList<WPVariable>();
        while (properties.get(i + "_name") != null) {
            containerVariable = new WPVariable();
            containerVariable.setName((String) properties.get(i + "_name"));
            containerVariable.setType((String) properties.get(i + "_type"));
            containerVariable.setAccess((String) properties.get(i + "_access"));
            result.add(containerVariable);
            i++;
        }
        return result;
    }
}