/**
 * 
 */
package com.tandbergtv.watchpoint.studio.validation.rules.nodedefinition.graph;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.EndState;
import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.graph.NodeDefinitionGraph;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;
import com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule;

import edu.uci.ics.jung.graph.Vertex;

/**
 * Validation rule that ensures that every node in the Node Definition is 'reachable'. A node is
 * reachable if and only if a path exists from the root node to that node, and a path exists from
 * that node to an 'End' node.
 * 
 * @author Imran Naqvi
 * 
 */
public class NodeDefinitionNodeReachableRule extends NodeReachableRule<NodeDefinitionGraph> {

	/**
	 * Get the Start Vertex.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#getRootVertices(com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph)
	 */
	@Override
	protected Set<Vertex> getRootVertices(NodeDefinitionGraph graph) {
		Set<Vertex> startVertices = new HashSet<Vertex>();

		/* Find the start node vertices */
		for (Object vertexObject : graph.getVertices()) {
			Vertex vertex = (Vertex) vertexObject;
			/* If vertex has an incoming edge it cannot be the root vertex */
			if (vertex.inDegree() != 0)
				continue;
			
			startVertices.add(vertex);
		}

		return startVertices;
	}

	/**
	 * Get the End Vertex.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#getSinkVertices(com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph)
	 */
	@Override
	protected Set<Vertex> getSinkVertices(NodeDefinitionGraph graph) {
		Set<Vertex> endVertices = new HashSet<Vertex>();

		/* Find the end node vertices */
		for (Object vertexObject : graph.getVertices()) {
			Vertex vertex = (Vertex) vertexObject;
			NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);
			if (node instanceof EndState) {
				endVertices.add(vertex);
			}
		}

		return endVertices;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#validateNodesUnreachableFromRoot(java.util.Set,
	 *      java.util.List)
	 */
	@Override
	protected void validateNodesUnreachableFromRoot(Set<Vertex> vertices,
			final List<ValidationMessage> messages) {
		this.validateUnreachableNode(vertices, messages, true);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#validateNodesUnreachableToSink(java.util.Set,
	 *      java.util.List)
	 */
	@Override
	protected void validateNodesUnreachableToSink(Set<Vertex> vertices,
			final List<ValidationMessage> messages) {
		this.validateUnreachableNode(vertices, messages, false);
	}

	/*
	 * Create the validation messages for all nodes not reachable from the Start Node
	 */
	private void validateUnreachableNode(Set<Vertex> vertices, List<ValidationMessage> messages,
			boolean fromStart) {
		if (vertices == null)
			return;

		for (Vertex vertex : vertices) {
			NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);
			ValidationMessageCode code = (fromStart)
					? ValidationMessageCode.NODE_UNREACHABLE_FROM_START
					: ValidationMessageCode.NODE_UNREACHABLE_TO_END;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code);
		}
	}
}
