/*
 * Created on Jul 24, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Validation Rule to ensure that all Nodes in a Node Element Container have unique names.
 * 
 * @param <NEC>
 *            The type of the Node Element Container to validate
 * 
 * @author Vijay Silva
 */
public abstract class NodeNamesUniqueRule<NEC extends NodeElementContainer> extends ValidationRule<NEC>
{
	/**
	 * Ensures that all Node Names are unique.
	 * 
	 * @param target
	 *            The Object to validate
	 * 
	 * @return The list of Validation Messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(NEC target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		Map<String, Set<NodeElement>> nameMap = new HashMap<String, Set<NodeElement>>();

		// Build a Map of all the names against the Node Elements
		for (NodeElement element : target.getNodeElements())
		{
			String name = element.getName();
			if (name == null || name.trim().length() == 0)
			{ // Don't consider blank names for duplicate checking
				continue;
			}

			Set<NodeElement> elements = nameMap.get(name);
			if (elements == null)
			{
				elements = new HashSet<NodeElement>();
				nameMap.put(name, elements);
			}

			elements.add(element);
		}

		// Validate the list of duplicate node names
		this.validateDuplicateNodeNames(target, nameMap, messages);

		return messages;
	}

	/**
	 * Validate the map of duplicate node names, generating the validation messages if required.
	 * 
	 * @param target
	 *            The Node Element Container being validated
	 * @param duplicates
	 *            A Map containing all the duplicate node names and for each duplicate name, the set
	 *            of Node Elements with the same name.
	 * @param messages
	 *            The list of validation messages to which new validation messages must be added.
	 */
	protected void validateDuplicateNodeNames(NEC target, Map<String, Set<NodeElement>> duplicates,
			final List<ValidationMessage> messages)
	{
		if (duplicates == null)
			return;

		for (String name : duplicates.keySet())
		{
			Set<NodeElement> elements = duplicates.get(name);
			if (elements != null && elements.size() > 1)
			{
				ValidationMessageCode code = this.getDuplicateNodeNameValidationCode();
				List<String> parameters = new ArrayList<String>();
				parameters.add(name);
				ValidationMessageAdder.getInstance().addValidationMessage(messages, target, code, parameters);
			}
		}
	}

	/**
	 * Get the ValidationMessageCode to use for reporting duplicate Node Names.
	 * 
	 * @return The ValidationMessageCode
	 */
	protected abstract ValidationMessageCode getDuplicateNodeNameValidationCode();
}
