/*
 * Created on Aug 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.resourcetype;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import com.tandbergtv.watchpoint.studio.dto.ConnectionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Validation of the Resource Type Basic Properties (Name, System Id, Connection Type)
 * 
 * @author Vijay Silva
 */
public class BasicPropertiesRule extends ValidationRule<ResourceType>
{
	private static final String SYSTEM_ID_REGEX = "^(\\p{Alnum}{2})$";

	/**
	 * Validate all the basic required properties for the Resource Type
	 * 
	 * @param target
	 *            The Resource Type to validate
	 * 
	 * @return The list of validation errors.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ResourceType target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		this.validateName(target, messages);
		this.validateSystemId(target, messages);
		this.validateConnectionType(target, messages);

		return messages;
	}

	/*
	 * Validate the Name property
	 */
	private void validateName(ResourceType resourceType, List<ValidationMessage> messages)
	{
		String name = resourceType.getName();

		if (name == null || name.trim().length() == 0)
		{ // Check for blanks
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_NAME_BLANK);
		}
		else if (name.length() > 50)
		{ // The ResourceType Name cannot exceed 50 characters
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_NAME_TOO_LONG);
		}
	}

	/*
	 * Validate the System Id property
	 */
	private void validateSystemId(ResourceType resourceType, List<ValidationMessage> messages)
	{
		String systemId = resourceType.getSystemId();

		if (systemId == null || systemId.trim().length() == 0)
		{ // Check for blanks
		    ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_SYSTEM_ID_BLANK);
		}
		else if (!Pattern.matches(SYSTEM_ID_REGEX, systemId))
		{ // Check the format of the System Id
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_SYSTEM_ID_INVALID);
		}
	}

	/*
	 * Validate the Resource Type's connection type.
	 */
	private void validateConnectionType(ResourceType resourceType, List<ValidationMessage> messages)
	{
		ConnectionType connectionType = resourceType.getConnectionType();

		if (connectionType == null)
		{ // Must be set
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_CONNECTION_TYPE_BLANK);
		}
	}

}
