package com.tandbergtv.watchpoint.studio.builder;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.jbpm.gd.common.model.NamedElement;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.fs.dao.ResourceGroupParser;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.view.DTONameDecorator;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

/**
 * 		Visitor for the Resource Group. Parses the file, validates and publishes on the database.
 * 
 * @author <a href="francisco.neto@venturus.org.br">vntfrne Francisco Bento da Silva Neto</a>
 *
 */
public class ResourceGroupVisitor implements IResourceDeltaVisitor, IResourceVisitor {
	// The Logger
	private static final Logger logger = Logger.getLogger(ResourceGroupVisitor.class);

	private static final String MARKER_TYPE = "org.jbpm.gd.jpdl.resourceGroupProblem";
	
	/**
	 * This method is invoked when a resource changes.
	 * 
	 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
	 */
	public boolean visit(IResourceDelta delta) throws CoreException {
		boolean visitChilden = true;
		
		IResource resource = delta.getResource();
		
		// Verify if its not the "classes" or "bin" folder where the java compiler creates the compiled files.
		if (!isClassesFolder(resource)) {
			
			boolean isPlugin = checkPluginXML(resource);
			
			if (isPlugin) {
				switch (delta.getKind()) {
				case IResourceDelta.ADDED:
				case IResourceDelta.CHANGED:
					// handle added or changed resource
					saveOrUpdate(resource);
					break;
				case IResourceDelta.REMOVED:
					// handle removed resource
					deleteResourceGroup(resource);
					break;
				}
			}
		} else {
			// Its the "bin" or "classes" folder, skip visiting
			visitChilden = false;
		}

		return visitChilden;
	}

	/**
	 * Invoked when a FULL BUILD is executed.
	 * 
	 * @see org.eclipse.core.resources.IResourceVisitor#visit(org.eclipse.core.resources.IResource)
	 */
	public boolean visit(IResource resource) {
		boolean visitChilden = true;
		
		// Verify if its not the "classes" or "bin" folder where the java compiler put the compiled files.
		if (!isClassesFolder(resource)) {
			boolean isPlugin = checkPluginXML(resource);
			
			if (isPlugin) {
				saveOrUpdate(resource);
			}
		} else {
			// Its the "bin" or "classes" folder, skip visiting
			visitChilden = false;
		}
		
		//return true to continue visiting children.
		return visitChilden;
	}
	
	/**
	 * 		Validates the Resource Group, and apply the markers when needed.
	 * @param file
	 * @param template
	 */
	void validateResourceGroup(IFile file, ResourceGroup resourceGroup) {
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		
		List<ValidationMessage> messages = service.validateResourceGroup(resourceGroup);
		try {
			file.deleteMarkers(MARKER_TYPE, false, IResource.DEPTH_ZERO);
		} catch (CoreException e) {
			logger.error("Error deleting markers", e);
		}
		boolean hasErrors = false;
		boolean hasWarnings = false;
		if (messages != null && !messages.isEmpty()) {
			for (ValidationMessage message : messages) {
				try {
					IMarker marker = file.createMarker(MARKER_TYPE);
					String msg = message.getCode();
					try {
						msg = com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages.getInstance().getMessage(message);
						String elementName = "";
						if (message.getElement() instanceof NamedElement) {
							elementName = ((NamedElement) message.getElement()).getName();
						} else if (message.getElement() instanceof IWatchPointDTO) {
							elementName = new DTONameDecorator((IWatchPointDTO)message.getElement()).getName();
						}
						
						if (elementName != null && !elementName.isEmpty()) {
							msg = elementName + " : " + msg;
						}
					} catch (Exception e) {
						logger.error("Error reading properties", e);
					} 
					marker.setAttribute(IMarker.MESSAGE, msg);
					
					// defaul severity
					int severity = IMarker.SEVERITY_ERROR;
					switch (message.getType()) {
						case Error:
							hasErrors = true;
							break;
						
						case Warning:
							hasWarnings = true;
							severity = IMarker.SEVERITY_WARNING;
							break;
	
						case Notification:
							severity = IMarker.SEVERITY_INFO;
							break;
					}
					marker.setAttribute(IMarker.SEVERITY, severity);
					marker.setAttribute(IMarker.LINE_NUMBER, 1);
                    message.copyAttributesTo(marker);
				} catch (CoreException e) {
					logger.error("Error adding marker", e);
				}
			}
		} 
		resourceGroup.setHasErrors(hasErrors);
		resourceGroup.setHasWarnings(hasWarnings);
		// Update the ResourceGroup hasErrors and hasWarnings properties.
		service.saveResourceGroup(resourceGroup);
	}
	
	
	/**
	 * 		Saves or update the Resource Group represented by the resource.
	 * 
	 * @param resource
	 */
	void saveOrUpdate(IResource resource) {
		IFile file = (IFile) resource;
		ResourceGroup resourceGroup = ResourceGroupParser.parse(file);
		
		if (resourceGroup != null) {
			resourceGroup.setProjectName(file.getProject().getName());
			resourceGroup.setPath(resource.getFullPath().toPortableString());
			
			IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();

			ResourceGroup previousResourceGroup = service.getResourceGroupByPath(resourceGroup.getPath());
			String previousResourceGroupName = null; 
			if (previousResourceGroup != null) {
				previousResourceGroupName = previousResourceGroup.getName();
			}
			
			service.saveResourceGroup(resourceGroup);
			validateResourceGroup(file, resourceGroup);
			
			validateNameChanges(previousResourceGroupName, resourceGroup);
			validateDependencies(resourceGroup);
			
		}
			
	}
	
	/*
	 * 		Re-Validate all ResourceGroup dependent entities.
	 * 
	 * @param resourceType
	 */
	private void validateDependencies(ResourceGroup resourceGroup) {
		logger.debug("Validating deps for " + resourceGroup.getName());
		/* Validate the Templates. */
		IWorkflowTemplateService templateService = ServiceFactory.createFactory().createWorkflowTemplateService();
		WatchpointTemplateVisitor templateVisitor = new WatchpointTemplateVisitor();
		List<WorkflowTemplateDTO> referencingTemplates = templateService.getTemplatesByResourceGroupUsage(resourceGroup.getName());
		Set<WorkflowTemplateDTO> set = new HashSet<WorkflowTemplateDTO>(referencingTemplates);
		
		logger.debug("Validating " + set.size() + " templates");
		
		for (WorkflowTemplateDTO referencingTemplate : set) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(referencingTemplate.getPath()));
			templateVisitor.validateTemplate(file, referencingTemplate);
		}
	}
	
	/**
	 * 		Validate name changes. 
	 * 		1. If the ResourceGroup name has changed, have to check if its matching
	 * another ResourceGroup name and then add validation errors (markers) to these ResourceGroups.
	 * 		2. If the ResourceGroup name has changed and its not matching another ResourceGroup name,
	 * have to clear the possibly previous errors (markers).
	 *  
	 * @param previousResourceGroupName
	 * @param resourceGroup
	 */
	void validateNameChanges(String previousResourceGroupName, ResourceGroup resourceGroup) {
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		if (previousResourceGroupName != null) {
			// If name has changed, call builder for the ResourceGroups with the old name
			if (!previousResourceGroupName.equals(resourceGroup.getName())) {
				List<ResourceGroup> previousNameResourceGroups = service.getResourceGroupByName(previousResourceGroupName);
				for (ResourceGroup previousNameResouceGroup : previousNameResourceGroups) {
					if (!previousNameResouceGroup.getPath().equals(resourceGroup.getPath())) {
						IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(previousNameResouceGroup.getPath()));
						validateResourceGroup(file, previousNameResouceGroup);
					}
				}
	 		}
		} 
		
		// If name has changed, call builder for the ResourceGroups with the new name
		List<ResourceGroup> newNameResourceGroups = service.getResourceGroupByName(resourceGroup.getName());
		for (ResourceGroup newNameResourceGroup : newNameResourceGroups) {
			if (!newNameResourceGroup.getPath().equals(resourceGroup.getPath())) {
				IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(newNameResourceGroup.getPath()));
				validateResourceGroup(file, newNameResourceGroup);
			}
		}
	}
	
	/**
	 * 		Deletes the Resource Group.
	 * 
	 * @param resource
	 */
	public void deleteResourceGroup(IResource resource) {
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		ResourceGroup group = service.getResourceGroupByPath(resource.getFullPath().toPortableString());
		if (group != null) {
			deleteResourceGroup(group);
		}
	}
	
	private void deleteResourceGroup(ResourceGroup group) {
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		if (group != null) {
			service.deleteResourceGroup(group.getPath());
			/* Re-Validate resource groups with the same name */ 
			List<ResourceGroup> homonymous = service.getResourceGroupByName(group.getName());
			for (ResourceGroup homonym : homonymous) {
				if (!group.getPath().equals(homonym.getPath())) {
					IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(homonym.getPath()));
					validateResourceGroup(file, homonym);
				}
			}
			validateDependencies(group);
		}
	}
	
	/**
	 * 		Deletes all Resource Groups from a specified project.
	 * 
	 * @param resource
	 */
	public void delete(IProject resource) {
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		
		List<ResourceGroup> groups = service.getResourceGroupsByProject(resource.getName());
		for (ResourceGroup group : groups) {
			deleteResourceGroup(group);
		}
	}
	
	/**
	 * 		Check if its a valid ResourceGroup file.
	 * @param resource
	 * @return
	 */
	boolean checkPluginXML(IResource resource) {
		boolean isPlugin = false;
		
		if (resource instanceof IFile && resource.getName().equals(ResourceGroup.RESOURCE_GROUP_FILENAME)) {
			isPlugin = true;
		}
		return isPlugin;
	}

	/**
	 * 		Check if its the java compiled classes folder - usually project/bin or project/classes . 
	 * 
	 * @param resource
	 * @return
	 */
	private boolean isClassesFolder(IResource resource) {
		boolean isClassesFolder = false;
		
		IJavaProject project = JavaCore.create(resource.getProject());
		try {
			IPath outputFolder = project.getOutputLocation();
			if (outputFolder.isPrefixOf(resource.getFullPath())) {
				isClassesFolder = true;
			}
		} catch (JavaModelException e) {
			// ignore as its just for optimization
		}
		return isClassesFolder;
	}
	
}
