/*
 * Created on Jun 12, 2006
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess.jpa;

import java.io.Serializable;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dto.IPersistable;

/**
 * The Abstract JPA Data Access Object that is used to save data to the persistence layer
 * 
 * @author Vijay Silva
 * 
 * @param <EC>
 *            The Entity
 * @param <EKC>
 *            The Entity Key
 */
public abstract class PersistenceDAO<EC extends IPersistable, EKC extends Serializable> implements
		DataAccessInterface<EC, EKC>
{
	private Class<EC> entityClass;

	private JPAPersistenceContext context;

	// ========================================================================
	// ================= CONSTRUCTORS
	// ========================================================================

	/**
	 * Constructor
	 * 
	 * @param entityClass
	 *            The Data Object Class
	 * @param persistenceContext
	 *            The Persistence Context used to perform persistence operations
	 */
	protected PersistenceDAO(Class<EC> entityClass, JPAPersistenceContext persistenceContext)
	{
		this.entityClass = entityClass;
		this.context = persistenceContext;
	}

	// ========================================================================
	// ================= DATA ACCESS OPERATIONS
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface#create(IPersistable)
	 */
	public EC create(EC entity)
	{
		this.getEntityManager().persist(entity);
		return entity;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface#update(IPersistable)
	 */
	public EC update(EC entity)
	{
		return this.getEntityManager().merge(entity);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface#delete(java.io.Serializable)
	 */
	public void delete(EKC key)
	{
		EC entity = this.find(key);
		if (entity != null)
			this.getEntityManager().remove(entity);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface#find(Serializable)
	 */
	public EC find(EKC key)
	{
		return this.getEntityManager().find(this.getEntityClass(), key);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface#findAll()
	 */
	@SuppressWarnings("unchecked")
	public List<EC> findAll()
	{
		String queryString = "SELECT entity " + this.getFromClause() + " entity";
		Query query = this.getEntityManager().createQuery(queryString);
		return query.getResultList();
	}

	/**
	 * Refreshes the state of the entity with the state in the DB.
	 * 
	 * @param entity
	 *            The entity to refresh
	 */
	protected void refresh(EC entity)
	{
		this.getEntityManager().refresh(entity);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface#getPersistenceContext()
	 */
	public JPAPersistenceContext getPersistenceContext()
	{
		return this.context;
	}

	// ========================================================================
	// ================= HELPER ACCESSOR METHODS
	// ========================================================================

	/**
	 * Gets the Single Result for the Query, or null if no result exists.
	 */
	protected Object getSingleResultOrNull(Query query)
	{
		try
		{
			return query.getSingleResult();
		}
		catch (NoResultException nre)
		{
			return null;
		}
	}

	/**
	 * Method to get the EntityManager to use for data operations.
	 * 
	 * @return The Entity Manager
	 * 
	 * @throws IllegalStateException
	 *             The Persistence Context is not created or initalized.
	 */
	protected EntityManager getEntityManager()
	{
		if (this.context == null || !this.context.isInitialized())
		{
			String msg = "The Persistence Context has not been initialized, cannot perform data operation.";
			throw new IllegalStateException(msg);
		}

		return this.context.getEntityManager();
	}

	/**
	 * Get the Class associated with this Data Access Object
	 * 
	 * @return The Entity Type
	 */
	protected Class<EC> getEntityClass()
	{
		return this.entityClass;
	}

	/**
	 * Generates a simple FROM clause with format: FROM 'entity class name'
	 * 
	 * @return The From clause
	 */
	protected String getFromClause()
	{
		return "FROM " + this.getEntityClass().getName();
	}
}
