/**
 * UniqueNodeNameGenerator.java
 * Created Feb 29, 2012
 */
package com.tandbergtv.watchpoint.studio.external.wpexport.impl.template;

import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.SuperState;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;

/**
 * Ensures that all nodes within a super state have unique names that don't clash with the
 * template's top level nodes
 */
public class UniqueNodeNameGenerator {

	protected Map<String,List<NodeElement>> nodeMap = new HashMap<String,List<NodeElement>>();
	protected Map<String,Integer> nodeNameRunnerMap = new HashMap<String,Integer>();
	
	protected NodeElementContainer rootContainer;
	
	protected boolean considerSuperstates;
	
	/**
	 * 		
	 * @param considerSuperstates Boolean param indicating if will consider
	 * inner SuperState nodes when generating unique names 
	 */
	public UniqueNodeNameGenerator(boolean considerSuperstates) {
		this.considerSuperstates = considerSuperstates;
	}
	
	public void setRootContainer(NodeElementContainer rootContainer) {
		this.rootContainer = rootContainer;
		buildNodesMap(rootContainer);
	}

	/**
	 * 		Builds the node map. 
	 * By default consider only rootContainer and LoopNode namespace.
	 * 
	 * @param container
	 */
	protected void buildNodesMap(NodeElementContainer container) {
		for (NodeElement node : container.getNodeElements()) {
			putNode(node);
			
			boolean isSuperstate = false;
			if (node instanceof NodeDefinition) {
				NodeDefinition ss = (NodeDefinition) node;
				isSuperstate = ss.getNodeType().equals(SuperState);
				if (isSuperstate) {
					node = ss.getNode();
				}
			}
			
			if (node instanceof LoopNode || (isSuperstate && considerSuperstates)) {
				buildNodesMap((NodeElementContainer) node);
			}
			
		}
	}
	
	/**
	 * 		Sets a unique (within the container) node name for a node.
	 * 
	 * @param node
	 */
	public String setUniqueNodeNameFor(NodeElement node) {
		String newName = findNewName(node);
		renameNode(newName, node);
		return newName;
	}
	
	/**
	 * 		Finds a sequential new name for a node based on the name prefix.
	 * 
	 * @param prefix
	 * @return
	 */
	protected String findNewName(NodeElement nodeElement) {
		String prefix = nodeElement.getNamePrefix();
		Integer runner = nodeNameRunnerMap.get(prefix);
		if (runner == null) {
			runner = 1;
		} 
		String newName = null;
		do {
			newName = prefix + runner;
			runner++;
		} while (nodeMap.containsKey(newName));
		nodeNameRunnerMap.put(prefix, runner);
		return newName;
	}
	

	/**
	 * 		Verifies if a node name already exists
	 * 
	 * @param node
	 */
	public boolean nameExists(String name) {
		return nodeMap.containsKey(name);
	}
	
	/**
	 * 		Checks if there's another node with the same name
	 * in the container (include Loop's, other SuperStates...)
	 * 
	 * @param name
	 * @param node
	 * @return
	 */
	public boolean nameClash(String name, NodeElement node) {
		List<NodeElement> nodes = nodeMap.get(name);
		boolean clash = false;
		if (nodes != null) {
			if (nodes.size() > 1) {
				clash = true;
			} else {
				// check if its the node itself
				NodeElement storedNode = nodes.get(0);
				clash = storedNode != node;
			}
		}
		return clash;
	}
	
	/**
	 * 		Puts a new node in the cache
	 * @param node
	 */
	protected void putNode(NodeElement node) {
		List<NodeElement> nodes = nodeMap.get(node.getName());
		if (nodes == null) {
			nodes = new ArrayList<NodeElement>();
		}
		nodes.add(node);
		nodeMap.put(node.getName(), nodes);
	}
	
	/**
	 * 		Pops a node from the cache
	 * @param node
	 */
	protected void popNode(NodeElement node) {
		List<NodeElement> nodes = nodeMap.get(node.getName());
		if (nodes != null) {
			for (Iterator<NodeElement> i = nodes.iterator(); i.hasNext();) {
				if (i.next() == node) {
					i.remove();
					break;
				}
			}
			if (nodes.isEmpty()) {
				nodeMap.remove(node.getName());
			}
		}
	}
	
	/**
	 * 		Renames a node.
	 * 
	 * @param newName
	 * @param node
	 */
	protected void renameNode(String newName, NodeElement node) {
		if (node.getName() != null && !"".equals(node.getName())) {
			// pops the node just in case previous name isn't empty
			popNode(node);
		}
		node.setName(newName);
		putNode(node);
	}
	
	/**
	 * Changes names of superstate children if necessary
	 * 
	 * @param template
	 */
	public void fixSuperStateNodeNames() {
		fixSuperStateNodeNames(rootContainer);
	}
	
	/**
	 * 		Fix the names for the SuperState children
	 *  
	 * @param container
	 */
	protected void fixSuperStateNodeNames(NodeElementContainer container) {
		for (NodeElement node : container.getNodeElements()) {
			if ((node instanceof NodeDefinition) && ((NodeDefinition) node).getNodeType() == SuperState) {
				NodeGroup superState = (NodeGroup) ((NodeDefinition) node).getNode();
				
				recursiveFixNodeContainerNodeNames(superState);
			} else if (node instanceof LoopNode) {
				fixSuperStateNodeNames((LoopNode) node);
			}
		}
	}
	
	/**
	 * 		Recursively fix the nodes for a container.
	 * 
	 * @param superState
	 */
	protected void recursiveFixNodeContainerNodeNames(NodeElementContainer superState) {
		for (NodeElement nodeElement : superState.getNodeElements()) {
			String oldName = nodeElement.getName();
			boolean nameClash = nameClash(oldName, nodeElement);
			if (nameClash) {
				String newName = setUniqueNodeNameFor(nodeElement);
				
				/* Fix transitions to this node in the super-state if any */
				for (NodeElement nodeElement2 : superState.getNodeElements()) {
					for (Transition transition : nodeElement2.getTransitions()) {
						if (transition.getTo().equals(oldName)) {
							transition.setTo(newName);
						}
					}
				}
			}
			// its a loop inside a SuperState node, loop through its child nodes.
			if (nodeElement instanceof LoopNode) {
				recursiveFixNodeContainerNodeNames((NodeElementContainer) nodeElement);
			}
		}
	}
}
