/*
 * Created on Jul 5, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service.impl;

import static com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys.HUMAN_SYSTEM_ID;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.ProtectionKeyDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI;
import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter;
import com.tandbergtv.watchpoint.studio.external.wpexport.WatchPointDTOExporterFactory;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * The IResourceGroupService implementation.
 * 
 * @author Vijay Silva
 */
public class ResourceGroupService extends ServiceImpl implements IResourceGroupService
{
	/**
	 * Constructor
	 */
	public ResourceGroupService()
	{
		super();
	}

	// ========================================================================
	// ====== IRESOURCEGROUPSERVICE METHODS
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getAllResourceGroups()
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceGroup> getAllResourceGroups()
	{
		Class<?>[] types = {};
		return (List<ResourceGroup>) super.performOperation(types);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getResourceGroupsByResourceType(String)
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceGroup> getResourceGroupsByResourceType(String resourceTypeSystemId)
	{
		Class<?>[] types = { String.class };
		return (List<ResourceGroup>) super.performOperation(types, resourceTypeSystemId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getResourceGroupsByHumanResourceType()
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceGroup> getResourceGroupsByHumanResourceType()
	{
		Class<?>[] types = {};
		return (List<ResourceGroup>) super.performOperation(types);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getResourceGroup(long)
	 */
	public ResourceGroup getResourceGroup(long resourceGroupId)
	{
		Class<?>[] types = { long.class };
		return (ResourceGroup) super.performOperation(types, resourceGroupId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getResourceGroupByName(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<ResourceGroup> getResourceGroupByName(String name)
	{
		Class<?>[] types = { String.class };
		return (List<ResourceGroup>) super.performOperation(types, name);
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getResourceGroupsByProject(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<ResourceGroup> getResourceGroupsByProject(String project)
	{
		Class<?>[] types = { String.class };
		return (List<ResourceGroup>) super.performOperation(types, project);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getProtectionKeyByResourceGroupName(java.lang.String)
	 */
	public ProtectionKey getProtectionKeyByResourceGroupName(String name)
	{
		Class<?>[] types = { String.class };
		return (ProtectionKey) super.performOperation(types, name);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getAllFunctionalTypesForResourceType(String)
	 */
	@SuppressWarnings("unchecked")
	public List<String> getAllFunctionalTypesForResourceType(String resourceTypeSystemId)
	{
		Class<?>[] types = { String.class };
		return (List<String>) super.performOperation(types, resourceTypeSystemId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#isNameExists(java.lang.String)
	 */
	public boolean isNameExists(String name)
	{
		Class<?>[] types = { String.class };
		return (Boolean) super.performOperation(types, name);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#createResourceGroup(com.tandbergtv.watchpoint.studio.dto.ResourceGroup)
	 */
	public ResourceGroup createResourceGroup(ResourceGroup group)
	{
		Class<?>[] types = { ResourceGroup.class };
		return (ResourceGroup) super.performOperation(types, group);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#updateResourceGroup(com.tandbergtv.watchpoint.studio.dto.ResourceGroup)
	 */
	public ResourceGroup updateResourceGroup(ResourceGroup group)
	{
		Class<?>[] types = { ResourceGroup.class };
		return (ResourceGroup) super.performOperation(types, group);
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#saveResourceGroup(com.tandbergtv.watchpoint.studio.dto.ResourceGroup)
	 */
	@Override
	public ResourceGroup saveResourceGroup(ResourceGroup group) {
		Class<?>[] types = { ResourceGroup.class };
		return (ResourceGroup) super.performOperation(types, group);
	}


	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#deleteResourceGroup(String)
	 */
	public void deleteResourceGroup(String path)
	{
		Class<?>[] types = { String.class };
		super.performOperation(types, path);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#exportResourceGroup(com.tandbergtv.watchpoint.studio.dto.ResourceGroup,
	 *      java.util.Map)
	 */
	public ResourceGroup exportResourceGroup(ResourceGroup group, Map<String, Object> exportInputs)
			throws ExportFailureException
	{
		// update dirty flag for the resource group in database
		Class<?>[] types = { ResourceGroup.class };
		ResourceGroup rgInDB = (ResourceGroup) super.performOperation(types, group);

		// export
		WatchPointDTOExporterFactory exporterFactory = WatchPointDTOExporterFactory.createFactory();
		IWatchPointDTOExporter<ResourceGroup> exporter = exporterFactory.getExporter(ResourceGroup.class);
		exporter.export(rgInDB, exportInputs);

		return rgInDB;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getAllProtectionKeys()
	 */
	@SuppressWarnings("unchecked")
	public List<ProtectionKey> getAllProtectionKeys()
	{
		Class<?>[] types = {};
		return (List<ProtectionKey>) super.performOperation(types);
	}

	// ========================================================================
	// ====== BUSINESS LOGIC IMPLEMENTATION (WITHOUT CONTEXT MANAGEMENT)
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getAllResourceGroups()
	 */
	protected List<ResourceGroup> getAllResourceGroups(IPersistenceContext context)
	{
		return createResourceGroupDAO(context).findAll();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getResourceGroupsByResourceType(long)
	 */
	protected List<ResourceGroup> getResourceGroupsByResourceType(String resourceTypeSystemId,
			IPersistenceContext context)
	{
		return createResourceGroupDAO(context).findByResourceTypeSystemId(resourceTypeSystemId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getResourceGroupsByHumanResourceType()
	 */
	protected List<ResourceGroup> getResourceGroupsByHumanResourceType(IPersistenceContext context)
	{
		ApplicationProperties properties = ApplicationProperties.getInstance();
		String humanSystemId = properties.getProperty(HUMAN_SYSTEM_ID);
		return createResourceGroupDAO(context).findByResourceTypeSystemId(humanSystemId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getResourceGroup(long)
	 */
	protected ResourceGroup getResourceGroup(long resourceGroupId, IPersistenceContext context)
	{
		return createResourceGroupDAO(context).find(resourceGroupId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getResourceGroupByName(java.lang.String)
	 */
	protected List<ResourceGroup> getResourceGroupByName(String name, IPersistenceContext context)
	{
		return createResourceGroupDAO(context).findByName(name);
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getResourceGroupsByTemplateUsage(java.lang.String)
	 */
	protected List<ResourceGroup> getResourceGroupsByTemplateUsage(String templatePath, IPersistenceContext context) {
		return createResourceGroupDAO(context).findByTemplateUsage(templatePath);
	}
	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getResourceGroupsByProject(java.lang.String)
	 */
	protected List<ResourceGroup> getResourceGroupsByProject(String project, IPersistenceContext context)
	{
		return createResourceGroupDAO(context).findByProject(project);
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getResourceGroupByPath(java.lang.String)
	 */
	protected ResourceGroup getResourceGroupByPath(String path, IPersistenceContext context)
	{
		return createResourceGroupDAO(context).findByPath(path);
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getDefaultResourceGroupForResourceType(com.tandbergtv.watchpoint.studio.dto.ResourceType)
	 */
	public ResourceGroup getDefaultResourceGroupForResourceType(ResourceType resourceType, IPersistenceContext context) {
		ResourceGroup defaultGroup = null; 
		
		List<ResourceGroup> groups = createResourceGroupDAO(context).findByResourceTypeSystemId(resourceType.getSystemId());
		if (groups.size() == 1) {
			defaultGroup = groups.get(0);
		}
		return defaultGroup;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getProtectionKeyByResourceGroupName(java.lang.String)
	 */
	protected ProtectionKey getProtectionKeyByResourceGroupName(String name,
			IPersistenceContext context)
	{
		return createResourceGroupDAO(context).findProtectionKeyByName(name);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getAllFunctionalTypesForResourceType(String)
	 */
	protected List<String> getAllFunctionalTypesForResourceType(String resourceTypeSystemId,
			IPersistenceContext context)
	{
		return createResourceGroupDAO(context).getAllFunctionalTypesForResourceType(resourceTypeSystemId);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#isNameExists(java.lang.String)
	 */
	protected Boolean isNameExists(String name, IPersistenceContext context)
	{
		return createResourceGroupDAO(context).isNameExists(name);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#saveResourceGroup(com.tandbergtv.watchpoint.studio.dto.ResourceGroup)
	 */
	protected ResourceGroup saveResourceGroup(ResourceGroup group, IPersistenceContext context) {
		ResourceGroupDAI rgDAO = createResourceGroupDAO(context);
		ResourceGroup resourceGroupInDB = rgDAO.findByPath(group.getPath());
		
		// create protection key in db and replace in the group object
		if (group.getProtectionKey() != null && group.getProtectionKey().getName() != null
		    && !group.getProtectionKey().getName().isEmpty()) {
			ProtectionKeyDAI protectionKeyDAO = createProtectionKeyDAO(context);
			ProtectionKey persistedPK = protectionKeyDAO.findByName(group.getProtectionKey().getName());
			if (persistedPK == null) {
				persistedPK = protectionKeyDAO.create(group.getProtectionKey());
			}
			group.setProtectionKey(persistedPK);
		}
		
		if (resourceGroupInDB != null) {
			group.setId(resourceGroupInDB.getId());
			group = rgDAO.update(group);
		} else {
			group = rgDAO.create(group);
		}
		
		return group;
	}


	/**
	 * Deletes the given resource group if the group is never exported and not used by any template.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#deleteResourceGroup(String)
	 */
	protected void deleteResourceGroup(String path, IPersistenceContext context)
	{
		ResourceGroupDAI rgDAO = createResourceGroupDAO(context);
		ResourceGroup resourceGroupInDB = rgDAO.findByPath(path);

		rgDAO.delete(resourceGroupInDB.getKey());
	}

	/**
	 * Sets the dirty flag to false for the given resource group in database.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#exportResourceGroup(ResourceGroup,
	 *      Map)
	 */
	protected ResourceGroup exportResourceGroup(ResourceGroup group, IPersistenceContext context) {
		// get resource group from database based on id or name
		ResourceGroupDAI rgDAO = createResourceGroupDAO(context);
		ResourceGroup rgInDB = rgDAO.find(group.getKey());

		// set dirty flag to false & update in DB (it is not dirty once it is exported)
		rgDAO.update(rgInDB);

		return rgInDB;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.impl.ResourceGroupService#getAllProtectionKeys()
	 */
	protected List<ProtectionKey> getAllProtectionKeys(IPersistenceContext context)
	{
		return createProtectionKeyDAO(context).findAll();
	}

	// ========================================================================
	// ==================== HELPER METHODS
	// ========================================================================

	/*
	 * Create the data access object for the ResourceGroup
	 */
	private ResourceGroupDAI createResourceGroupDAO(IPersistenceContext context)
	{
		Class<ResourceGroup> entityClass = ResourceGroup.class;
		DataAccessInterface<?, ?> dao = this.daFactory.createDataAccessObject(entityClass, context);
		return (ResourceGroupDAI) dao;
	}

	/*
	 * Create the data access object for the ProtectionKey
	 */
	private ProtectionKeyDAI createProtectionKeyDAO(IPersistenceContext context)
	{
		Class<ProtectionKey> entityClass = ProtectionKey.class;
		DataAccessInterface<?, ?> dao = this.daFactory.createDataAccessObject(entityClass, context);
		return (ProtectionKeyDAI) dao;
	}

	// ========================================================================
	// ==================== VALIDATION METHODS
	// ========================================================================

	/**
	 * Performs validation for update of the given ResourceGroup.
	 */
	protected List<ValidationMessage> validateForUpdate(ResourceGroup group,
			ResourceGroup resourceGroupInDB, IPersistenceContext context)
	{
		// basic ui validation
		List<ValidationMessage> uiValidationMessages = validateResourceGroup(group);

		// if ui validation fails, return error messages
		if (uiValidationMessages != null && !uiValidationMessages.isEmpty())
			return uiValidationMessages;

		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		// compare to check if the name is changed
        if (!group.getName().equals(resourceGroupInDB.getName())) {
            ValidationMessageAdder.getInstance().addValidationMessage(messages, group, ValidationMessageCode.RESOURCEGROUP_NAME_CHANGED);
        }

		// compare to check if the resourcetype is changed
		if (!resourceGroupInDB.getSystemId().equals(group.getSystemId())) {
            ValidationMessageAdder.getInstance().addValidationMessage(messages, group, ValidationMessageCode.RESOURCEGROUP_RESOURCETYPE_CHANGED);
		}

		return messages;
	}

	/**
	 * Performs basic UI validation using the Validation Framework.
	 * 
	 * @param group
	 *            The Resource Group object
	 * 
	 * @return The List of Validation Messages
	 */
	public List<ValidationMessage> validateResourceGroup(ResourceGroup group)
	{
		ValidationServiceFactory factory = ValidationServiceFactory.createFactory();
		IValidationService validationService = factory.createValidationService();
		return validationService.validateResourceGroup(group);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getResourceGroupByPath(java.lang.String)
	 */
	@Override
	public ResourceGroup getResourceGroupByPath(String path)
	{
		Class<?>[] types = { String.class };
		return (ResourceGroup) super.performOperation(types, path);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getDefaultResourceGroupForResourceType(com.tandbergtv.watchpoint.studio.dto.ResourceType)
	 */
	@Override
	public ResourceGroup getDefaultResourceGroupForResourceType(ResourceType resourceType) {
		Class<?>[] types = { ResourceType.class };
		return (ResourceGroup) super.performOperation(types, resourceType);
	}

	@Override
	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IResourceGroupService#getResourceGroupsByTemplateUsage(String)
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceGroup> getResourceGroupsByTemplateUsage(String templatePath) {
		Class<?>[] types = { String.class };
		return (List<ResourceGroup>) super.performOperation(types, templatePath);
	}
}
