/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.editor;

import java.util.EventObject;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.gef.DefaultEditDomain;
import org.eclipse.gef.EditDomain;
import org.eclipse.gef.commands.CommandStack;
import org.eclipse.gef.commands.CommandStackListener;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertySheetPageContributor;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.wst.sse.ui.StructuredTextEditor;
import org.jbpm.gd.common.editor.AbsPageEditorPart;
import org.jbpm.gd.common.editor.ContentProvider;
import org.jbpm.gd.common.editor.OutlineViewer;
import org.jbpm.gd.common.editor.SelectionSynchronizer;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.notation.NotationElementFactory;
import org.jbpm.gd.common.notation.RootContainer;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.common.xml.XmlAdapterFactory;
import org.jbpm.gd.jpdl.Logger;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.Transition;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * The editor for node definitions. Defines a graph page to drag and drop objects and set their
 * properties. Also defines a source page where the xml representing the graph is shown.
 * 
 * @author Imran Naqvi
 * 
 */
@SuppressWarnings("restriction")
public class NodeDefinitionEditor extends AbsPageEditorPart implements
		ITabbedPropertySheetPageContributor {

	protected static final String LINE_SEPRATOR = "line.separator";
	
	private static final String NODEDEFINITIONIMAGE = "nodedefinitionimage.jpg";

	private NodeDefinitionActionRegistry actionRegistry;

	private EditDomain editDomain;

	private CommandStackListener commandStackListener;

	private boolean isDirty = false;

	private ISelectionListener selectionListener;

	private SelectionSynchronizer selectionSynchronizer;

	private SemanticElementFactory semanticElementFactory;

	private NotationElementFactory notationElementFactory;

	private RootContainer rootContainer;

	private NodeDefinitionGraphPage graphPage;

	private StructuredTextEditor sourcePage;

	private MenuManager sourceContextMenuManager;

	private ContentProvider contentProvider;

	private NodeDefinitionDTO nodeDefinitionDTO;

	/**
	 * @see org.eclipse.wst.xml.ui.internal.tabletree.XMLMultiPageEditorPart#init(org.eclipse.ui.IEditorSite,
	 *      org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		FileEditorInput fileInput = (FileEditorInput) input;
		NodeDefinitionDTO tempDTO = (NodeDefinitionDTO) ((WPFileEditorInput) fileInput).getWatchPointDTO();
		this.nodeDefinitionDTO = tempDTO;
		super.init(site, fileInput);
		initEditDomain();
		initCommandStackListener();
		initSelectionListener();
		initSelectionSynchronizer();
		/* register this editor as a listener to editor lifecycle events */
		Utility.setWindow(site.getWorkbenchWindow());
		checkTemplateUsage();
	}
	
	/**
	 * 		Checks if the SuperState is being used in any templates and shows a warning.
	 */
	protected void checkTemplateUsage() {
		Display display = getSite().getShell().getDisplay();
		display.asyncExec(new Runnable() {
			public void run() {
				IWorkflowTemplateService templateService = ServiceFactory.createFactory().createWorkflowTemplateService();
				List<WorkflowTemplateDTO> templateList = templateService.getTemplatesBySuperStateUsage(nodeDefinitionDTO.getName(), nodeDefinitionDTO.getResourceType().getSystemId());
				
				if (templateList.size() > 0) {
					Set<String> templateSet = new HashSet<String>();
					for (WorkflowTemplateDTO dto : templateList) {
						templateSet.add(dto.getName());
					}
					
				    String message = "SuperState is being used by the following templates: "
								+ templateSet + "." + System.getProperty(LINE_SEPRATOR);
				    
				    message+= "Do you wish to edit ?";

					boolean yes = MessageDialog.openQuestion(Display.getCurrent().getActiveShell(), "SuperState in use", message);
					
					if (!yes) {
						close();
					}
				}
			}
		});

	}
	
	public void close() {
		Display display = getSite().getShell().getDisplay();
		display.asyncExec(new Runnable() {
			public void run() {
				getSite().getPage().closeEditor(NodeDefinitionEditor.this, false);
			}
		});
	}

	/**
	 * @see org.eclipse.wst.xml.ui.internal.tabletree.XMLMultiPageEditorPart#createPages()
	 */
	protected void createPages() {
		super.createPages();
		initSourcePage();
		initActionRegistry();
		initGraphPage();
		setActivePage(0);
	}

	/*
	 * Creates and adds a new page containing the given editor to this multi-page editor. The page
	 * is added at the given index. This also hooks a property change listener on the nested editor.
	 * 
	 * @param index the index at which to add the page (0-based) @param part the nested editor
	 * @param label the label of the editor
	 */
	private void addPage(int index, IEditorPart part, String label) {
		try {
			addPage(index, part, getEditorInput());
			setPageText(index, label);
		}
		catch (PartInitException e) {
			Logger.logError("Could not create editor page", e);
		}
	}

	/**
	 * @see org.eclipse.wst.xml.ui.internal.tabletree.XMLMultiPageEditorPart#isSaveOnCloseNeeded()
	 */
	public boolean isSaveOnCloseNeeded() {
		return isDirty();
	}

	/**
	 * @see org.eclipse.ui.views.properties.tabbed.ITabbedPropertySheetPageContributor#getContributorId()
	 */
	public String getContributorId() {
		return getSite().getId();
	}

	/**
	 * Gets the name of the node definition DTO.
	 * 
	 * @return the name of the node definition DTO
	 */
	public String getNodeDefinitionDTOName() {
		return (nodeDefinitionDTO != null) ? this.nodeDefinitionDTO.getName() : null;
	}

	/**
	 * Gets the id of the node definition DTO.
	 * 
	 * @return the id of the node definition DTO
	 */
	public long getNodeDefinitionDTOId() {
		return (nodeDefinitionDTO != null) ? this.nodeDefinitionDTO.getId() : 0L;
	}

	/*
	 * Initializes the action registry.
	 */
	private void initActionRegistry() {
		actionRegistry = new NodeDefinitionActionRegistry(this);
	}

	/*
	 * Initializes the edit domain.
	 */
	private void initEditDomain() {
		editDomain = new DefaultEditDomain(this);
	}

	/*
	 * Initializes the selectionSynchronizer (a selectionChangedListener).
	 */
	private void initSelectionSynchronizer() {
		selectionSynchronizer = createSelectionSynchronizer();
	}

	/*
	 * Initializes the command stack listener.
	 */
	private void initCommandStackListener() {
		commandStackListener = new CommandStackListener() {
			public void commandStackChanged(EventObject event) {
				handleCommandStackChanged();
			}
		};
		getCommandStack().addCommandStackListener(commandStackListener);
	}

	/*
	 * Initializes the selection listener for this editor.
	 */
	private void initSelectionListener() {
		selectionListener = new ISelectionListener() {
			public void selectionChanged(IWorkbenchPart part, ISelection selection) {
				getActionRegistry().updateEditPartActions();
			}
		};
		ISelectionService selectionService = this.getEditorSite()
				.getWorkbenchWindow()
				.getSelectionService();
		selectionService.addSelectionListener(selectionListener);
	}

	/**
	 * Initializes the graph page for this editor.
	 */
	protected void initGraphPage() {
		graphPage = new NodeDefinitionGraphPage(this);
		addPage(0, graphPage, "Diagram");
		getContentProvider().addNotationInfo(getRootContainer(), getEditorInput());
	}

	/**
	 * Initializes the source page.
	 */
	protected void initSourcePage() {
		int pageCount = getPageCount();
		for (int i = 0; i < pageCount; i++) {
			if (getEditor(i) instanceof StructuredTextEditor) {
				sourcePage = (StructuredTextEditor) getEditor(i);
				NodeDefinition nodeDefinition = (NodeDefinition) getSemanticElement(sourcePage);
				nodeDefinition.setNodeType(this.nodeDefinitionDTO.getType());
				NodeDefinitionCreationFactory factory = new NodeDefinitionCreationFactory(
						nodeDefinition.getElementId(), getSemanticElementFactory(),
						getNotationElementFactory());
				setRootContainer((RootContainer) factory.getNewObject());
				getRootContainer().setSemanticElement(nodeDefinition);
				nodeDefinition.addPropertyChangeListener(getRootContainer());
				setTransitionSources(nodeDefinition);
			}
		}
		if (sourcePage != null) {
			initSourcePageContextMenu(sourcePage.getTextViewer().getTextWidget());
		}
	}

	/*
	 * Sets the source for all transitions in the node definition.
	 */
	private void setTransitionSources(NodeDefinition nodeDefinition) {
		for (NodeElement node : nodeDefinition.getNodeElements()) {
			for (Transition transition : node.getTransitions()) {
				transition.setSource(node);
			}
		}
	}

	/*
	 * @param sourcePage the StructuredTextEditor from which to obtain the semantic element @return
	 * the semantic element represented by the xml in the sourcePage
	 */
	private SemanticElement getSemanticElement(StructuredTextEditor sourcePage) {
		Node node = getDocumentElement(sourcePage);
		XmlAdapterFactory factory = new XmlAdapterFactory(node.getOwnerDocument(),
				getSemanticElementFactory());
		XmlAdapter xmlAdapter = factory.adapt(node);
		SemanticElement semanticElement = createMainElement();
		xmlAdapter.initialize(semanticElement);
		return semanticElement;
	}

	/*
	 * @param sourcePage the StructuredTextEditor from which to obtain the document element @return
	 * the document element represented by the xml in the sourcePage
	 */
	private Node getDocumentElement(StructuredTextEditor sourcePage) {
		Node result = null;
		Document document = (Document) sourcePage.getAdapter(Document.class);
		if (document != null) {
			result = (Node) document.getDocumentElement();
		}
		return result;
	}

	/*
	 * Initializes the source page context menu.
	 * 
	 * @param control
	 */
	private void initSourcePageContextMenu(final Control control) {
		sourceContextMenuManager = new MenuManager("#PopupMenu");
		sourceContextMenuManager.setRemoveAllWhenShown(true);
		sourceContextMenuManager.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager m) {
				fillContextMenu(m);
			}
		});
		Menu menu = sourceContextMenuManager.createContextMenu(control);
		getSite().registerContextMenu("org.jbpm.ui.editor.DesignerEditor.SourcePopupMenu",
				sourceContextMenuManager, getSite().getSelectionProvider());

		control.setMenu(menu);
	}

	/**
	 * Fills the context menu for the menu manager.
	 * 
	 * @param menuManager
	 */
	private void fillContextMenu(final IMenuManager menuManager) {
		menuManager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	/*
	 * Handler for the command stack listener. Updates all the actions in the action registry and
	 * the dirty flag
	 */
	private void handleCommandStackChanged() {
		getActionRegistry().updateStackActions();
		if (!isDirty() && getCommandStack().isDirty()) {
			isDirty = true;
			firePropertyChange(IEditorPart.PROP_DIRTY);
		}
	}

	/**
	 * @param dirty
	 *            the value to set the dirty flag
	 */
	public void setDirty(boolean dirty) {
		isDirty = dirty;
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#isDirty()
	 */
	public boolean isDirty() {
		return (isDirty || super.isDirty());
	}

	/**
	 * Gets the action registry for this editor.
	 * 
	 * @return the action registry
	 */
	public NodeDefinitionActionRegistry getActionRegistry() {
		return this.actionRegistry;
	}

	/**
	 * Gets the graph page.
	 * 
	 * @return the graph page
	 */
	public NodeDefinitionGraphPage getGraphPage() {
		return graphPage;
	}

	/**
	 * Gets the edit domain for this editor.
	 * 
	 * @return the edit domain for this editor
	 */
	public EditDomain getEditDomain() {
		return editDomain;
	}

	/**
	 * Gets the command stack for this editor.
	 * 
	 * @return the command stack for this editor
	 */
	public CommandStack getCommandStack() {
		return editDomain.getCommandStack();
	}

	/**
	 * @param selectionSynchronizer
	 *            the selection synchronizer to set
	 */
	public void setSelectionSynchronizer(SelectionSynchronizer selectionSynchronizer) {
		this.selectionSynchronizer = selectionSynchronizer;
	}

	/**
	 * Gets the selection synchronizer for this editor.
	 * 
	 * @return the selection synchronizer for this editor
	 */
	public SelectionSynchronizer getSelectionSynchronizer() {
		return selectionSynchronizer;
	}

	/**
	 * Gets the semantic element factory for this editor.
	 * 
	 * @return the semantic element factory for this editor
	 */
	public SemanticElementFactory getSemanticElementFactory() {
		if (semanticElementFactory == null) {
			semanticElementFactory = new SemanticElementFactory(getContributorId());
		}
		return semanticElementFactory;
	}

	/**
	 * Gets the notation element factory for this editor.
	 * 
	 * @return the notation element factory for this editor
	 */
	public NotationElementFactory getNotationElementFactory() {
		if (notationElementFactory == null) {
			notationElementFactory = new NotationElementFactory();
		}
		return notationElementFactory;
	}

	/**
	 * Gets the root container for this editor.
	 * 
	 * @return the root container for this editor
	 */
    @Override
	public RootContainer getRootContainer() {
		return rootContainer;
	}

	/**
	 * Gets the node definition currently open for this editor.
	 * 
	 * @return the node definition currently open for this editor
	 */
	public NodeDefinition getNodeDefinition() {
		if (this.rootContainer == null)
			return null;
		return (NodeDefinition) rootContainer.getSemanticElement();
	}

	/**
	 * @param rootContainer
	 *            the root container to set for this editor
	 */
	public void setRootContainer(RootContainer rootContainer) {
		this.rootContainer = rootContainer;
	}

	/**
	 * Gets the content provider for this editor.
	 * 
	 * @return the content provider for this editor
	 */
	public ContentProvider getContentProvider() {
		if (contentProvider == null) {
			contentProvider = createContentProvider();
		}
		return contentProvider;
	}

	/**
	 * Gets the graphical viewer for this editor.
	 * 
	 * @return the graphical viewer for this editor
	 */
	public NodeDefinitionGraphicalViewer getGraphicalViewer() {
		return getGraphPage().getDesignerViewer();
	}

	/**
	 * Gets the outline viewer for this editor.
	 * 
	 * @return the outline viewer for this editor
	 */
	public OutlineViewer getOutlineViewer() {
		return getGraphPage().getOutlineViewer();
	}

	/**
	 * @see org.eclipse.wst.xml.ui.internal.tabletree.XMLMultiPageEditorPart#getAdapter(java.lang.Class)
	 */
	@SuppressWarnings("rawtypes")
	public Object getAdapter(Class adapter) {
		Object result = null;
		if (adapter == CommandStack.class) {
			result = getCommandStack();
		}
		else if (adapter == IContentOutlinePage.class) {
			return getOutlineViewer();
		}
		else if (adapter == IPropertySheetPage.class) {
			return new TabbedPropertySheetPage(this);
		}
		else if (adapter == org.eclipse.gef.GraphicalViewer.class) {
			return getGraphicalViewer();
		}
		else {
			result = super.getAdapter(adapter);
		}
		return result;
	}

	/**
	 * @see org.eclipse.wst.xml.ui.internal.tabletree.XMLMultiPageEditorPart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) {
		/* First save the xml files */
		super.doSave(monitor);
		getGraphPage().doSave(monitor);
		boolean saved = getContentProvider().saveToInput(getEditorInput(), getRootContainer());
		if (saved) {
			getCommandStack().markSaveLocation();
			setDirty(false);
			firePropertyChange(IEditorPart.PROP_DIRTY);
		}
	}

	/**
	 * @see org.eclipse.wst.xml.ui.internal.tabletree.XMLMultiPageEditorPart#dispose()
	 */
	public void dispose() {
		super.dispose();

		// Unregister from all services
		this.getEditorSite().getWorkbenchWindow().getSelectionService().removeSelectionListener(
				selectionListener);
		getCommandStack().removeCommandStackListener(commandStackListener);
	}

	/**
	 * Gets the image file name for the image stored of this node definition.
	 * 
	 * @return image file name
	 */
    @Override
	public String getDefaultImageFileName() {
		return NODEDEFINITIONIMAGE;
	}

	/**
	 * @see org.eclipse.ui.part.WorkbenchPart#getPartName()
	 */
	@Override
	public String getPartName() {
		if (this.getRootContainer() == null || this.getNodeDefinition().getName() == null)
			return super.getPartName();
		return this.getNodeDefinition().getName();
	}

	/**
	 * @return a new selection synchronizer
	 */
	protected SelectionSynchronizer createSelectionSynchronizer() {
		return new SelectionSynchronizer();
	}

	/**
	 * Creates a new instance of the content provider for this editor.
	 * 
	 * @return a new instance of the content provider for this editor
	 */
	protected ContentProvider createContentProvider() {
		return new NodeDefinitionContentProvider();
	}

	/**
	 * Create a NodeDefinition instance.
	 * 
	 * @return a NodeDefinition instance
	 */
	protected SemanticElement createMainElement() {
		return getSemanticElementFactory().createById(
				"com.tandbergtv.watchpoint.studio.nodeDefinition");
	}

	/**
	 * Creates a NodeDefinitionGraphicalViewer.
	 * 
	 * @return a NodeDefinitionGraphicalViewer instance
	 */
	protected NodeDefinitionGraphicalViewer createGraphicalViewer() {
		return new NodeDefinitionGraphicalViewer(this) {
			protected void initEditPartFactory() {
				setEditPartFactory(new GraphicalEditPartFactory(getEditor().getEditorInput()));
			}
		};
	}
}
