package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.editor.expression.ExpressionEditor;
import com.tandbergtv.watchpoint.studio.ui.model.AssignNode;
import com.tandbergtv.watchpoint.studio.ui.model.IExpression;
import com.tandbergtv.watchpoint.studio.ui.model.IValidatableElement;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

public class AssignNodeSectionComposite {

    private TabbedPropertySheetWidgetFactory widgetFactory;
    private Composite parent;
    private Composite composite;

    private Table table;
    private TableViewer tableViewer;
    private Button buttonAdd;
    private Button buttonRemove;

    private String[] columnNames = new String[] { "name", "type", "value" }; //$NON-NLS-1$ $NON-NLS-2$ $NON-NLS-3$
    private AssignNodeCellModifier cellModifier;
    private ExpressionEditor expressionEditor;

    public AssignNodeSectionComposite(Composite parent, TabbedPropertySheetWidgetFactory widgetFactory) {
        this.widgetFactory = widgetFactory;
        this.parent = parent;

        createComponents();
    }

    private void createComponents() {
        composite = widgetFactory.createFlatFormComposite(parent);

        createButtons();
        createTable();
        createTableViewer();
    }

    private void createTable() {
        //table will be created in a exclusive parent in order to use TableColumnLayout
        Composite exclusiveTableParent = widgetFactory.createFlatFormComposite(composite);

        int tableStyle = SWT.SINGLE | SWT.BORDER | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.HIDE_SELECTION;
        table = widgetFactory.createTable(exclusiveTableParent, tableStyle);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        FormData formData = new FormData();
        formData.left = new FormAttachment(0, 0);
        formData.bottom = new FormAttachment(100, 0);
        formData.top = new FormAttachment(0, 0);
        formData.right = new FormAttachment(buttonAdd, 0, SWT.LEFT);
        formData.width = 600; // trick: it has to be set in order to have TableColumnLayout working properly
        exclusiveTableParent.setLayoutData(formData);

        TableColumn column1 = new TableColumn(table, SWT.LEFT, 0);
        column1.setText(PropertiesMessages.AssignNodeSectionComposite_template_variable_label);
        column1.setResizable(true);

        table.setSortDirection(SWT.DOWN);
        table.setSortColumn(column1);
        column1.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (table.getSortDirection() == SWT.DOWN) {
                    table.setSortDirection(SWT.UP);
                } else {
                    table.setSortDirection(SWT.DOWN);
                }

                tableViewer.setComparator(new ViewerSorter() {
                    @Override
                    public int compare(Viewer viewer, Object e1, Object e2) {
                        int directionVar = 1;
                        if (table.getSortDirection() == SWT.UP) {
                            directionVar = -1;
                        }

                        Variable v1 = (Variable) e1;
                        Variable v2 = (Variable) e2;

                        if (e1 == null) return directionVar * 1;
                        if (v1.getName() == null) return directionVar * 1;
                        if (e2 == null) return directionVar * -1;
                        if (v2.getName() == null) return directionVar * -1;

                        return directionVar * v1.getName().compareTo(v2.getName());
                    }
                });
            }
        });

        TableColumn column2 = new TableColumn(table, SWT.LEFT, 1);
        column2.setText(PropertiesMessages.AssignNodeSectionComposite_type_label);
        column2.setResizable(true);

        TableColumn column3 = new TableColumn(table, SWT.LEFT, 2);
        column3.setText(PropertiesMessages.AssignNodeSectionComposite_value_label);
        column3.setResizable(true);

        TableColumnLayout layout = new TableColumnLayout();
        exclusiveTableParent.setLayout(layout);
        layout.setColumnData(column1, new ColumnWeightData(30));
        layout.setColumnData(column2, new ColumnWeightData(20));
        layout.setColumnData(column3, new ColumnWeightData(30));
    }

    private void createTableViewer() {
        tableViewer = new TableViewer(table);
        
        tableViewer.setUseHashlookup(true);
        tableViewer.setColumnProperties(columnNames);
        tableViewer.setContentProvider(new AssignNodeContentProvider());
        tableViewer.setLabelProvider(new AssignNodeLabelProvider());

        CellEditor[] editors = new CellEditor[columnNames.length];
        editors[0] = new ComboBoxCellEditor(table, new String[0], SWT.BORDER | SWT.READ_ONLY);
        editors[1] = null;
        editors[2] = new TextCellEditor(table);

        expressionEditor = new ExpressionEditor(tableViewer, 2);

        // Assign the cell editors to the viewer 
        tableViewer.setCellEditors(editors);

        cellModifier = new AssignNodeCellModifier(tableViewer, columnNames);

        // Set the cell modifier for the viewer
        tableViewer.setCellModifier(cellModifier);
    }

    private void createButtons() {
        buttonAdd = widgetFactory.createButton(composite, "+", SWT.PUSH);
        FormData formData = new FormData();
        formData.top = new FormAttachment(0, 0);
        formData.right = new FormAttachment(100, -35);
        formData.width = 25;
        buttonAdd.setLayoutData(formData);
        buttonAdd.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                AssignNode assignNode = (AssignNode) tableViewer.getInput();
                WPVariable variable = assignNode.addEmptyVariable();

                tableViewer.add(variable);
            }
        });

        buttonRemove = widgetFactory.createButton(composite, "-", SWT.PUSH);
        formData = new FormData();
        formData.top = new FormAttachment(buttonAdd);
        formData.right = new FormAttachment(buttonAdd, 0, SWT.RIGHT);
        formData.width = 25;
        buttonRemove.setLayoutData(formData);
        buttonRemove.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (((IStructuredSelection) tableViewer.getSelection()).getFirstElement() != null) {
                    AssignNode assignNode = (AssignNode) tableViewer.getInput();
                    WPVariable variable = (WPVariable) ((IStructuredSelection) tableViewer.getSelection())
                            .getFirstElement();

                    assignNode.removeVariable(variable);

                    tableViewer.remove(variable);
                }
            }
        });
    }

    public void setInput(AssignNode model, List<WPVariable> variables, IProject project) {
        tableViewer.setInput(model);
        cellModifier.setVariables(variables);
        ((ComboBoxCellEditor)tableViewer.getCellEditors()[0]).setItems(Utility.toNameArray(variables));

        expressionEditor.setupAndInitialize(variables, project, new CustomExpression());
    }

    class CustomExpression implements IExpression {
        private IValidatableElement model;

        @Override
        public void setModel(IValidatableElement model) {
            this.model = model;
        }

        @Override
        public IValidatableElement getModel() {
            return model;
        }

        @Override
        public String getText() {
            WPVariable variable = (WPVariable) model;
            return variable.getValue();
        }

        @Override
        public DataType getReturnType() {
            WPVariable variable = (WPVariable) model;
            return DataType.valueOf(variable.getType());
        }

        @Override
        public boolean isValidatable() {
            return true;
        }
    };
}