/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties;

import static java.lang.Boolean.FALSE;
import static java.lang.Boolean.TRUE;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import com.tandbergtv.watchpoint.studio.util.TimeoutValidation;

/**
 * A list of data types for template variables.
 * 
 * @author Imran Naqvi
 */
public enum DataType {
	INT() {
		@Override
		public boolean match(String value) {
			try {
				Integer.parseInt(value);
				return true;
			} catch (NumberFormatException e) { return false; }
		}
		
		@Override
		public String getJavaMatchingType() {
			return Integer.class.getCanonicalName();
		}
	},
	STRING,
	BOOLEAN {
		@Override
		public boolean match(String value) {
			return TRUE.toString().equalsIgnoreCase(value) || FALSE.toString().equalsIgnoreCase(value);
		}
		
		@Override
		public String getJavaMatchingType() {
			return Boolean.class.getCanonicalName();
		}
	},
	DATE {
		@Override
		public boolean match(String value) {
			try {
				new SimpleDateFormat("yyyy-MM-dd").parse(value);
				return true;
			} catch (ParseException e) { return false; }
		}
		@Override
		public String getJavaMatchingType() {
			return Date.class.getCanonicalName();
		}
	},
	FILE,
	DURATION {
		@Override
		public boolean match(String value) {
			return TimeoutValidation.isTimeoutExpressionValid(value);
		}
	},
	URL,
	LIST {
		@Override
		public String getJavaMatchingType() {
			return List.class.getCanonicalName() + "<String>";
		}
	},
	OBJECT {
	    @Override
	    public String getJavaMatchingType() {
	        return Object.class.getCanonicalName();
	    }
	    
	    @Override
	    public boolean isListable() {
	        return false;
	    }
	};
	
	public boolean match(String value) {
		return true;
	}
	
	public String getJavaMatchingType() {
		return String.class.getCanonicalName();
	}
	
	public boolean isListable() {
	    return true;
	}
}
