/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties.nodedefinition;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.common.model.NamedElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.editor.NodeDefinitionEditor;
import com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer;
import com.tandbergtv.watchpoint.studio.ui.model.IResourceTypeElement;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * The composite for viewing/editing the uid of a IMessageNodeContainer.
 * 
 * @author Imran Naqvi
 * 
 */
public class UidConfigurationComposite implements SelectionListener {

	private TabbedPropertySheetWidgetFactory widgetFactory;

	private Composite parent;

	private CCombo uidCombo;

	private Label uidLabel;

	private IMessageNodeContainer messageContainerNode;

	private List<Message> messages;

	private AbstractPropertySection parentPropertySection;

	private UidConfigurationComposite() {
	}

	/**
	 * Creates a UidConfigurationComposite object in the parent object using the given widget
	 * factory.
	 * 
	 * @param parentSection
	 * @param widgetFactory
	 * @param parent
	 * @return The UidConfigurationComposite created
	 */
	public static UidConfigurationComposite create(AbstractPropertySection parentSection,
			TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		UidConfigurationComposite result = new UidConfigurationComposite();
		result.parentPropertySection = parentSection;
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}

	/**
	 * Sets the uid element and updates controls.
	 * 
	 * @param uidElement
	 */
	public void setUidElement(IMessageNodeContainer uidElement) {
		clearControls();
		this.messageContainerNode = uidElement;
		if (uidElement != null) {
			if (this.uidCombo.isEnabled())
				fillUidCombo();
			updateControls();
		}
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == uidCombo && this.messageContainerNode != null) {
			String uid = getUID(uidCombo.getText());
			if (!Utility.isOutgoingMessage(uid) && this.hasAttachment()) {
				MessageDialog.openError(
						Display.getCurrent().getActiveShell(),
						"Error",
						"This is an incoming message and can not have an attachment. Please remove the attachment variable first before selecting this message.");
				uidCombo.setText(getMessageName(this.messageContainerNode.getUid()));
				return;
			}
			
			this.messageContainerNode.setUid(uid);
		}
	}

	/*
	 * Gets the UID for a given message name.
	 */
	private String getUID(String messageName){
		for (Message message : this.messages)
			if (message.getName().equals(messageName))
				return message.getUid();
		return "";
	}
	
	/*
	 * Gets the message name for a given UID.
	 */
	private String getMessageName(String uid){
		for (Message message : this.messages)
			if (message.getUid().equals(uid))
				return message.getName();
		return "";
	}
	
	/*
	 * Clears the controls.
	 * 
	 */
	private void clearControls() {
		uidCombo.setText("");
	}

	/*
	 * Updates the controls.
	 * 
	 */
	private void updateControls() {
		IWorkbenchPart part = this.parentPropertySection.getPart();
		NodeElementContainer container = Utility.getNodeElementContainer(part);
		if (container == null)
			throw new UIException(
					"Current editor does not have a node element container associated with it.");
		/*
		 * Enable the uid combo only if the messageContainerNode is the nodeElementContainer in the
		 * node definition editor and has the same name
		 */
		boolean enable = ((NodeElementContainer) messageContainerNode).getElementId().equals(
				container.getElementId())
				&& (messageContainerNode instanceof NamedElement)
				&& (container instanceof NodeDefinition)
				&& ((NamedElement) messageContainerNode).getName().equals(
						((NamedElement) container).getName());
		uidCombo.setEnabled(enable);
		String uid = messageContainerNode.getUid();
		if (uid != null)
			setUIDInCombo();
	}

	/*
	 * Creates the widgets for this composite.
	 * 
	 */
	private void create() {
		Composite composite = widgetFactory.createFlatFormComposite(parent);
		uidLabel = new Label(composite, SWT.NONE);
		uidLabel.setLayoutData(createUidpLabelLayoutData());
		uidLabel.setText("Message");

		uidLabel.pack();

		uidCombo = widgetFactory.createCCombo(parent, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL
				| SWT.H_SCROLL | SWT.READ_ONLY);
		uidCombo.setLayoutData(createUidComboLayoutData());
		uidCombo.addSelectionListener(this);
		uidCombo.pack();
	}

	/*
	 * Creates layout data for the duration label.
	 * 
	 * @return
	 */
	private FormData createUidpLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(0, 0);
		return data;
	}

	/*
	 * Creates layout data for the duration text box.
	 * 
	 * @return
	 */
	private FormData createUidComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(uidLabel, 95);
		data.top = new FormAttachment(0, 0);
		data.width = 250;
		return data;
	}

	/*
	 * Fill the uid combo with messages for the resource type for the node definition.
	 */
	private void fillUidCombo() {
		this.uidCombo.removeAll();
		
		IResourceTypeService rtService = ServiceFactory.createFactory().createResourceTypeService();
		long id = ((IResourceTypeElement) this.messageContainerNode).getResourceType();
		long definitionId = getNodeDefinitionId();
		if (definitionId == 0)
			messages = rtService.getUnusedResourceTypeMessages(id);
		else
			messages = rtService.getUnusedResourceTypeMessages(id, definitionId);
		if (this.messageContainerNode != null && this.messageContainerNode.getUid() != null) {
			if (messages == null)
				messages = new ArrayList<Message>();
		}

		// sorting messages based on their name (case-insensitive)
		Collections.sort(messages, new Comparator<Message>() {
			public int compare(Message m1, Message m2) {
				// assumed that the message objects will not be null
				if (m1.getName() == null && m2.getName() == null)
					return 0;
				if (m1.getName() == null)
					return -1;
				if (m2.getName() == null)
					return 1;
				return m1.getName().compareToIgnoreCase(m2.getName());
			}
		});

		for (Message message : messages)
			this.uidCombo.add(message.getName());
		
	}

	/*
	 * Sets the uid in the uid combo.
	 */
	private void setUIDInCombo() {
		if (this.messageContainerNode == null || this.messageContainerNode.getUid() == null)
			return;
		IResourceTypeService rtService = ServiceFactory.createFactory().createResourceTypeService();
		Message msg = rtService.findMessageByUID(((NodeDefinition) this.messageContainerNode).getUid());
		
		if (msg != null) {
			this.uidCombo.setText(msg.getName());
		} else {
			this.uidCombo.setText( ((NodeDefinition) messageContainerNode).getDefinitionName() );
		}
		return;
	}

	/*
	 * Gets the Id of the node definition DTO of the NodeDefinitionEditor
	 */
	private long getNodeDefinitionId() {
		if (this.parentPropertySection.getPart() instanceof NodeDefinitionEditor)
			return ((NodeDefinitionEditor) this.parentPropertySection.getPart()).getNodeDefinitionDTOId();
		return 0L;
	}

	/*
	 * Checks whether this IMessageNodeContianer has an attachment variable.
	 */
	private boolean hasAttachment() {
		List<WPVariable> vars = this.messageContainerNode.getOutgoingVariables();
		for (WPVariable var : vars) {
			if (var.isAttachment())
				return true;
		}
		return false;
	}
}
