/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties.template;

import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.jbpm.gd.common.model.GenericElement;

import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.AbstractTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.properties.TaskNodeActionElementComposite;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;

/**
 * Item wrapper for Arguments in tables.
 * 
 * @author Devender Ramasahayam
 * 
 */
public class ArgumentTableItemWrapper implements DisposeListener, FocusListener {

	private TableItem tableItem;

	private Text nameText;

	private Text valueText;

	private TaskNodeActionElementComposite parent;

	private GenericElement genericElement;

	private IValidationService validationService;

	/**
	 * Constructor
	 * 
	 * @param parent
	 * @param table
	 * @param genericElement
	 * @param isEditable
	 */
	public ArgumentTableItemWrapper(TaskNodeActionElementComposite parent, Table table,
			GenericElement genericElement, boolean isEditable) {
		this.parent = parent;
		this.genericElement = genericElement;
		this.validationService = ValidationServiceFactory.createFactory().createValidationService();

		initializeTableItem(table);
		initializeNameText(table);
		initializeValueText(table);
		nameText.setEnabled(isEditable);
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e) {
		if (e.widget == this.nameText) {
			if (genericElement.getName().equals(this.nameText.getText())) {
				return;
			} if (isNameValid()) {
				genericElement.setName(this.nameText.getText());
				// HACK to mark the template as dirty when the GenericElement's name property is changed.
				parent.updateGenericElement();
			} else { 
				this.nameText.setText(this.genericElement.getName());
			}
		}
		else if (e.widget == this.valueText) {
			if (genericElement.getValue().equals(this.valueText.getText())) {
				return;
			} if (isValueValid()) {
				genericElement.setValue(this.valueText.getText());
			} else {
				this.valueText.setText(this.genericElement.getValue());
			}
		}
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e) {
		if (e.widget == this.nameText || e.widget == this.valueText) {
			this.tableItem.getParent().setSelection(this.tableItem);
			if (nameText.isEnabled())
				this.parent.setRemoveButtonEnable(true);
		}
	}

	/**
	 * @see org.eclipse.swt.events.DisposeListener#widgetDisposed(org.eclipse.swt.events.DisposeEvent)
	 */
	public void widgetDisposed(DisposeEvent e) {
		if (nameText == null || nameText.isDisposed())
			return;
		if (valueText == null || valueText.isDisposed())
			return;
		nameText.removeFocusListener(this);
		nameText.dispose();
		valueText.removeFocusListener(this);
		valueText.dispose();
	}

	/**
	 * Gets the table item.
	 * 
	 * @return table item for this item wrapper
	 */
	public TableItem getTableItem() {
		return tableItem;
	}

	/**
	 * Initializes the table item.
	 * 
	 * @param table
	 */
	protected void initializeTableItem(Table table) {
		tableItem = new TableItem(table, SWT.NONE);
		tableItem.addDisposeListener(this);
		tableItem.setData(this);
	}

	/**
	 * Initializes the mapped name text box.
	 * 
	 * @param table
	 *            the table in which the text box is being added
	 */
	protected void initializeNameText(Table table) {
		nameText = createText(table, 0);
		FormData data = new FormData();
		nameText.setLayoutData(data);
		nameText.addFocusListener(this);
		nameText.setText(getArgumentName());
		nameText.setEditable(true);
	}

	/**
	 * Initializes the Argument Value text box.
	 * 
	 * @param table
	 *            the table in which the text box is being added
	 */
	protected void initializeValueText(Table table) {
		valueText = createText(table, 1);
		FormData data = new FormData();
		valueText.setLayoutData(data);
		valueText.addFocusListener(this);
		valueText.setText(getArgumentValue());
		valueText.setEditable(true);
	}

	/**
	 * Creates a combo box in the given table in the column specified by the given index.
	 * 
	 * @param table
	 *            the table in which the combo box is being created
	 * @param index
	 *            index of the column in which to create the combo box
	 * @return the combo box created
	 */
	protected Text createText(Table table, int index) {
		TableEditor textEditor = new TableEditor(table);
		Text result = new Text(table, SWT.NORMAL);
		result.setVisible(true);
		result.pack();
		textEditor.minimumWidth = result.getSize().x;
		textEditor.horizontalAlignment = SWT.LEFT;
		textEditor.grabHorizontal = true;
		textEditor.setEditor(result, tableItem, index);
		return result;
	}

	/**
	 * Get the Argument name.
	 * 
	 * @return Argument name
	 */
	public String getArgumentName() {
		return genericElement.getName();
	}

	/**
	 * Get the Argument value.
	 * 
	 * @return Argument value
	 */
	public String getArgumentValue() {
		return genericElement.getValue();
	}

	/*
	 * Checks to see if the variable name is valid.
	 */
	private boolean isNameValid() {
		String variableName = this.nameText.getText();
		GenericElement element = new GenericElement();
		element.setName(variableName);

		List<ValidationMessage> messages = null;
		messages = this.validationService.validateActionVariableName(element);
		boolean valid = this.isValid(messages);
		boolean duplicate = false;
		if (!valid) {
			this.displayValidationErrors(messages);
		}
		else{
			duplicate = checkDuplicateName(variableName);
			if (duplicate) {
				MessageDialog.openError(Display.getCurrent().getActiveShell(), "Error", variableName
						+ " is a duplicate name. Duplicate names are not allowed.");
			}
		}
		return valid && !duplicate;
	}

	/*
	 * Checks to see if the variable value is valid.
	 */
	private boolean isValueValid() {
		boolean valid = true;
		AbstractTaskNode node = this.parent.getTaskNode();
		if (!(node instanceof AutomaticTaskNode))
			return valid;

		AutomaticTaskNode automaticTaskNode = (AutomaticTaskNode) node;
		GenericElement element = new GenericElement();
		element.setName(this.nameText.getText());
		element.setValue(this.valueText.getText());

		List<ValidationMessage> messages = null;
		messages = this.validationService.validateActionVariableValue(automaticTaskNode, element);
		valid = this.isValid(messages);
		if (!valid) {
			this.displayValidationErrors(messages);
		}

		return valid;
	}

	/* Check if the list of validation messages signals a validation error */
	private boolean isValid(List<ValidationMessage> messages) {
		return (messages == null || messages.size() == 0);
	}

	/*
	 * Display the list of validation messages provided in a single message dialog.
	 */
	private void displayValidationErrors(List<ValidationMessage> messages) {
		Shell shell = Display.getCurrent().getActiveShell();
		StringBuilder buf = new StringBuilder();
		ValidationMessages validationMessages = null;

		try {
			validationMessages = ValidationMessages.getInstance();
		}
		catch (Exception e) {
			throw new UIException("Could not load the validation messages.", e);
		}

		boolean first = true;
		for (ValidationMessage message : messages) {
			if (!first)
				buf.append(System.getProperty("line.separator"));
			else
				first = false;

			buf.append(validationMessages.getMessage(message));
		}

		MessageDialog.openError(shell, "Error", buf.toString());
	}

	/*
	 * Checks to see if the given name already exists in the action's generic elements.
	 */
	private boolean checkDuplicateName(String name) {
		for (GenericElement element : this.parent.getTaskNode().getAction().getGenericElements()) {
			if (element.getName().equals(name))
				return true;
		}
		return false;
	}
}
