package com.tandbergtv.watchpoint.studio.ui.wizard.export;

import static com.tandbergtv.watchpoint.studio.ui.wizard.export.ResourceTypeExportConstants.BIN_INCLUDES_KEY;
import static com.tandbergtv.watchpoint.studio.ui.wizard.export.ResourceTypeExportConstants.BUILD_PROPERTIES_FILE;
import static com.tandbergtv.watchpoint.studio.ui.wizard.export.ResourceTypeExportConstants.PLUGIN_XML_FILE;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.ui.statushandlers.StatusManager;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.util.FileUtil;
import com.tandbergtv.watchpoint.studio.util.ResourceTypePluginFileGenerator;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

public class ResourceTypeExportOperation extends Job{

	// The Logger
	private static final Logger logger = Logger.getLogger(ResourceTypeExportOperation.class);
	private List<String> excludes;
	protected FeatureExportInfo fInfo;

	public ResourceTypeExportOperation(FeatureExportInfo info, String name) {
		super(name);
		this.fInfo = info;
	}

	@Override
	protected IStatus run(IProgressMonitor monitor) {
		try {
			createDestination();
			
			// Updates the plugin.xml version number
			updateVersionNumber();
			
			exportFiles(monitor);
			
			if (monitor.isCanceled()) {
				return Status.CANCEL_STATUS;
			}
			
			exportResourceGroups();
			
			return Status.OK_STATUS;
		} catch (IOException e) {
			return new Status(IStatus.ERROR, "WatchPoint Studio", "Problem during Resource Type export", e);
		} catch (CoreException e) {
			return e.getStatus();
		} catch (InvocationTargetException e) {
			return new Status(IStatus.ERROR, "WatchPoint Studio",  "Problem during Resource Type export", e.getTargetException());
		} catch (ExportFailureException e) {
			return new Status(IStatus.WARNING, "WatchPoint Studio", "Problem during Resource Group export", e);
		} finally {
			monitor.done();
		}
	}
	
	protected void createDestination() throws InvocationTargetException {
		File file = new File(fInfo.destinationDirectory);
		if (!file.exists() || !file.isDirectory()) {
			if (!file.mkdirs())
				throw new InvocationTargetException(new Exception("Specified directory could not be created."));
		}
	}
	
	protected void updateVersionNumber(){
		for (int i = 0; i < fInfo.items.length; i++) {
			IProject project = (IProject)fInfo.items[i];
			
			try {
				// Writes the plugin.xml file in workspace project
				IFile pluginFile = project.getFile(PLUGIN_XML_FILE);
				if (pluginFile.exists()) {
					Document pluginDoc = XMLDocumentUtility.loadFile(pluginFile.getContents(), false);
					
					XPath xpath = XPathFactory.newInstance().newXPath();
					Node versionNode = (Node)xpath.evaluate("/plugin/@version", pluginDoc.getDocumentElement(), XPathConstants.NODE);
					String versionValue = versionNode.getNodeValue();
					int versionNumber = Integer.parseInt(versionValue);
					versionNode.setNodeValue(Integer.toString(++versionNumber));
					
					ByteArrayOutputStream plugin = ResourceTypePluginFileGenerator.createWritablePluginDocument(pluginDoc, new File(""));
					byte[] pluginContentArray = plugin.toByteArray();
					pluginFile.setContents(new ByteArrayInputStream(pluginContentArray), true, false, null);
				}
				
			} catch (Exception e) {
				String errorMessage = "Error updating Resource Type " + project.getName() + " version number";
				Status status = new Status(IStatus.ERROR, "WatchPoint Studio",IStatus.ERROR, errorMessage, e);
				StatusManager.getManager().handle(status, StatusManager.SHOW);
			}
		}
	}
	
	private void initExcludes(){
		this.excludes = new ArrayList<String>();
		this.excludes.add("CVS");
	}
	
	private void exportFiles(IProgressMonitor monitor) throws IOException, CoreException {
		Map<String, String[]> projectIncludes = new HashMap<String, String[]>();
		initExcludes();
		monitor.beginTask("Exporting ...", fInfo.items.length + 1);
		
		for (int i = 0; i < fInfo.items.length; i++) {
			IProject project = (IProject) fInfo.items[i];
			String projectName = project.getName();

			if (project.exists()) {
				IFile buildPropertiesFile = project.getFile(BUILD_PROPERTIES_FILE);
				Properties prop = new Properties();
				prop.load(buildPropertiesFile.getContents());
				String includes = prop.getProperty(BIN_INCLUDES_KEY);

				String[] exportFiles = includes.split(",");

				projectIncludes.put(projectName, exportFiles);
			}
		}

		exportToDirectory(projectIncludes, monitor);
	}
	
	private void exportToDirectory(Map<String, String[]> projectIncludes, IProgressMonitor monitor){
		for (Entry<?, ?> entry : projectIncludes.entrySet()) {
			String projectName = (String) entry.getKey();
			String[] exportFiles = (String[]) entry.getValue();
			
			monitor.setTaskName("Exporting project : " + projectName);
			
			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
			String projectPath = project.getLocation().toOSString();
			
			File destinationPluginFolder = new File(fInfo.destinationDirectory + File.separator
					+ projectName);
			if(!destinationPluginFolder.exists()){
				destinationPluginFolder.mkdir();
			}
			
			FileUtil.deleteFolderContents(destinationPluginFolder, true);
			
			for (int j = 0; j < exportFiles.length; j++) {
				String exportFile = exportFiles[j];
				if (exportFile.equals("."))
					continue;
				
				File source = new File(projectPath + File.separator + exportFile);
				File destination = new File(fInfo.destinationDirectory + File.separator + projectName
						+ File.separator + exportFile);
				
				copyExportFiles(source, destination, true, true);
			}
			monitor.worked(1);
		}
	}
	
	public boolean copyExportFiles(File source, File target, boolean copySubFolders, boolean overwrite)
	{
		boolean result = true;

		if (source.isDirectory())
		{
			if (target.exists() && !target.isDirectory() && !overwrite)
			{
				String msg = "The target location is an existing file, failed to copy folder: "
						+ source.getAbsolutePath() + " to: " + target.getAbsolutePath();
				logger.warn(msg);
				return false;
			}
			else 
				if (!target.exists() && !target.mkdirs())
			{
				String msg = "Failed to create folder: " + target.getAbsolutePath()
						+ " when attempting to copy folder: " + source.getAbsolutePath();
				logger.warn(msg);
				return false;
			}

			File[] contents = source.listFiles();
			if (contents != null)
			{
				for (File content : contents)
				{
					if ((content.isDirectory() && !copySubFolders) || excludes.contains(content.getName()) )
						continue;
					
					File targetContent = new File(target, content.getName());
					if (!copyExportFiles(content, targetContent, copySubFolders, overwrite))
					{
						result = false;
						break;
					}
				}
			}
		}
		else
		{
			if (target.exists() && !overwrite)
			{
				String msg = "The target location already exists, failed to copy file: "
						+ source.getAbsolutePath() + " to: " + target.getAbsolutePath();
				logger.warn(msg);
				return false;
			}
			try
			{
				if(!target.isDirectory() && !target.getParentFile().exists()){
					target.getParentFile().mkdirs();
				}
				FileUtil.copyFile(source, target);
			}
			catch (IOException ex)
			{
				String msg = "Failed when copying file from: " + source.getAbsolutePath() + " to: "
						+ target.getAbsolutePath();
				logger.warn(msg, ex);
				result = false;
			}
		}

		return result;
	}
	
	/*
	 * Exports all ResourceGroups associated to this ResourceType
	 * 
	 */
	private void exportResourceGroups() throws ExportFailureException {
		ExportFailureException exception = null;
		if (fInfo.exportGroups) {
			for (int i = 0; i < fInfo.items.length; i++) {
				IResourceTypeService resourceTypeService = ServiceFactory.createFactory().createResourceTypeService();
				IResourceGroupService resourceGroupService = ServiceFactory.createFactory().createResourceGroupService();
				IProject project = (IProject) fInfo.items[i];
				
				List<ResourceType> resourceTypes = resourceTypeService.getResourceTypeByProject(project.getName());
				
				for (ResourceType resourceType : resourceTypes) {
					Collection<ResourceGroup> resourceGroups = resourceGroupService.getResourceGroupsByResourceType(resourceType.getSystemId());
					for (ResourceGroup group : resourceGroups) {
				        try {
							Map<String, Object> exportInputAttrs = new HashMap<String, Object>();
							File pluginsFolder = new File(fInfo.destinationDirectory);
							/* 
							 * If ResourceType export location is 
							 * /opt/tandbergtv/cms/workflow/plugins/subsystems/MyRT/
							 * Resource group should be:
							 * /opt/tandbergtv/cms/workflow/plugins/groups/MyGroup/
							*/
							pluginsFolder = new File(pluginsFolder.getParentFile(), "groups/" + group.getName());
							// Create destination folder
							if (!pluginsFolder.exists()) {
								pluginsFolder.mkdirs();
							}
							
					        exportInputAttrs.put(ExporterInputAttributes.EXPORT_FOLDER_PATH, pluginsFolder.getAbsolutePath());
					        exportInputAttrs.put(ExporterInputAttributes.PLUGIN_CLASSES_FILE_PATHS, Collections.EMPTY_LIST);
					        exportInputAttrs.put(ExporterInputAttributes.PLUGIN_RESOURCE_FILE_PATHS, Collections.EMPTY_LIST);
					        exportInputAttrs.put(ExporterInputAttributes.PLUGIN_JAR_FILE_PATHS, Collections.EMPTY_LIST);
		
					        resourceGroupService.exportResourceGroup(group, exportInputAttrs);
						} catch (ExportFailureException e) {
							// Error exporting group, ignore...
							exception = e;
							logger.error("Error while trying to auto-export ResourceGroup", e);
						}
					}
				}
			}
		}
		if (exception != null) {
			// had exception, throw-it
			throw exception;
		}
	}
}
