/**
 * ImportWizardSummaryPage.java
 * Created May 5, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.imports;

import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.ADD;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.CHANGE;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.REMOVE;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.jbpm.gd.jpdl.Logger;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNode;
import com.tandbergtv.watchpoint.studio.ui.sync.DiffKind;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiffModel;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.DTONameDecorator;
import com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWizardPage;

/**
 * Displays a summary of changes to be made after the items to import have been selected
 * 
 * @author Sahil Verma
 */
public class ImportWizardSummaryPage extends AbstractWizardPage {

	/**
	 * Type of items being imported
	 */
	private Class<? extends IWatchPointDTO> clazz;
	
	private ImportWizardListPage page;
	
	/**
	 * Displays the differences
	 */
	private TreeViewer viewer;
	
	/**
	 * The diff model
	 */
	private IDiffModel diffModel;
	
	/**
	 * @param pageName
	 */
	public ImportWizardSummaryPage(String pageName, ImportWizardListPage page, Class<? extends IWatchPointDTO> clazz) {
		super(pageName);
		setTitle("Confirmation");
		setMessage("Examine list of differences and confirm");
		this.clazz = clazz;
		this.page = page;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isPageComplete() {
		return isCurrentPage();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		
		if (visible)
			loadDiffModel();
	}

	/**
	 * @return the tree
	 */
	public IDiffModel getDiffModel() {
		return this.diffModel;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void saveSettings(IDialogSettings settings) {
	}
	
	protected void loadDiffModel() {
		final Collection<? extends IWatchPointDTO> models = page.getSelectedModels();
		
		IRunnableWithProgress op = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) {
				monitor.beginTask("Calculating differences...", models.size());
				
				diffModel = Importers.getImporter(clazz).getDiffModel(models, monitor);
				
				monitor.done();
			}
		};
		
		try {
			getContainer().run(true, false, op);
			this.viewer.setInput(diffModel.getDifferences());
		} catch (InvocationTargetException e) {
			Logger.logError(e.getCause());
		} catch (InterruptedException e) {
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void createControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		
		layout.numColumns = 1;
		layout.horizontalSpacing = 5;
		layout.verticalSpacing = 0;
		
		container.setLayout(layout);
		container.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		createTreeViewPart(container);
		setControl(container);
		Dialog.applyDialogFont(container);
	}
	
	private void createTreeViewPart(Composite parent) {
		Composite container = new Composite(parent, SWT.NONE);
		
		container.setLayout(new GridLayout());
		container.setLayoutData(new GridData(GridData.FILL_BOTH));

		this.viewer = new TreeViewer(container, SWT.BORDER);
		
		GridData gd = new GridData(GridData.FILL_BOTH | SWT.H_SCROLL | SWT.V_SCROLL);
		gd.widthHint = 225;
		
		this.viewer.getTree().setLayoutData(gd);
		this.viewer.setContentProvider(new DiffContentProvider());
		this.viewer.setLabelProvider(new DiffLabelProvider());
		this.viewer.setComparator(new DiffComparer());
	}
	
	@SuppressWarnings("unchecked")
	class DiffLabelProvider extends CellLabelProvider {
		public DiffLabelProvider() {
			super();
			putImage(ADD, "added.png");
			putImage(REMOVE, "removed.png");
			putImage(CHANGE, "changed.png");
		}

		@Override
		public void update(ViewerCell cell) {
			ITreeNode<IDiff> node = (ITreeNode<IDiff>) cell.getElement();
			IDiff diff = node.getData();
			DiffKind kind = diff.getKind();
			String text = getText(diff);
			String decoration = getDecoration(diff);
			StyleRange [] styles = new StyleRange[2];
			Display display = cell.getControl().getDisplay();
			
			cell.setText(text + decoration);
			cell.setImage(getImage(kind));
			
			if (kind == REMOVE)
				cell.setForeground(display.getSystemColor(SWT.COLOR_DARK_GRAY));
			
			StyleRange style = new StyleRange();
			
			style.start = 0;
			style.length = text.length();
			
			styles[0] = style;
			
			StyleRange style2 = new StyleRange();
			
			style2.start = text.length() + 1;
			style2.length = decoration.length();
			style2.foreground = display.getSystemColor(SWT.COLOR_CYAN);
			
			styles[1] = style2;

			cell.setStyleRanges(styles);
		}
		
		private String getDecoration(IDiff diff) {
			if (diff.getModel() != null)
				return " (" + new DTONameDecorator(diff.getModel()).getType() + ")";
			
			return " (" + diff.getElement().getLabel() + ")";
		}

		private String getText(IDiff diff) {
			if (diff.getModel() != null)
				return new DTONameDecorator(diff.getModel()).getName();
			return diff.getElement().toString();
		}

		private Image getImage(DiffKind kind) {
			return JFaceResources.getImageRegistry().get(kind.name());
		}
		
		private void putImage(DiffKind kind, String image) {
			ImageRegistry registry = JFaceResources.getImageRegistry();
			
			if (registry.get(kind.name()) == null)
				registry.put(kind.name(), getImageDescriptor(image));
		}
		
		private ImageDescriptor getImageDescriptor(String image) {
			return Utility.getImageDescriptor(image);
		}
	}
	
	@SuppressWarnings("unchecked")
	class DiffContentProvider extends ArrayContentProvider implements ITreeContentProvider {
		public Object[] getElements(Object inputElement) {
			ITree<IDiff> tree = (ITree<IDiff>) inputElement;
			return tree.getRoot().getChildren().toArray();
		}

		public Object[] getChildren(Object arg0) {
			ITreeNode<IDiff> node = (ITreeNode<IDiff>) arg0;
			ArrayList<ITreeNode<IDiff>> list = new ArrayList<ITreeNode<IDiff>>(node.getChildren());
			Iterator<ITreeNode<IDiff>> i = list.iterator();
			
			while (i.hasNext()) {
				ITreeNode<IDiff> child = i.next();
				IDiff diff = child.getData();
				
				if (diff.getKind() == DiffKind.NONE)
					i.remove();
			}
			
			return list.toArray();
		}

		public Object getParent(Object arg0) {
			return ((ITreeNode<IDiff>) arg0).getParent();
		}

		public boolean hasChildren(Object arg0) {
			return ((ITreeNode<IDiff>) arg0).hasChildren();
		}
	}
	
	@SuppressWarnings("unchecked")
	class DiffComparer extends ViewerComparator {
		@Override
		public int compare(Viewer viewer, Object e1, Object e2) {
			return getComparator().compare(getName(e1), getName(e2));
		}
		
		private String getName(Object obj) {
			ITreeNode<IDiff> node = (ITreeNode<IDiff>) obj;
			IDiff diff = node.getData();
			if (diff.getModel() != null)
				return new DTONameDecorator(diff.getModel()).getName();
			return diff.getElement().toString();
		}
	}
}
