package com.tandbergtv.watchpoint.studio.util;

import java.util.ArrayList;
import java.util.List;

import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

/**
 * @author Raj Prakash
 * 
 * Validator to validate Timer Expression used in Due Date & Duration.
 *
 */
public abstract class TimerExprValidator {
	
	private List<WPVariable> variables;
	
	protected TimerExprValidator(List<WPVariable> variables) {
		this.variables = (variables != null) ? variables : new ArrayList<WPVariable>();
	}
	
	protected ValidationMessage validate(String expression) {
		if(expression == null || expression.trim().length() == 0)
			return null;
		
		return validate(TimerExpressionMarshaller.unmarshall(expression));
	}

	protected ValidationMessage validate(TimerExpression te) {
		if(te == null)
			return null;
		
		if(te.getOperator() == null) {
			//unary expression
			//check if only first operand is present
			if(te.getOperand2() != null)
				return getGenericInvalidTimerExprVM();
			
			return validateUnaryExpression(te.getOperand1());
		}
		else if(te.getOperator().equals(TimerExpression.ADDITION_OPERATOR)) {
			//addition expression
			return validateAdditionExpression(te.getOperand1(), te.getOperand2());
		}
		else if(te.getOperator().equals(TimerExpression.MULTIPLICATION_OPERATOR)) {
			//multiplication expression
			return validateMultiplicationExpression(te.getOperand1(), te.getOperand2());
		}
		return null;
	}
	
	protected abstract ValidationMessage getInvalidAdditionExprVM();
	protected abstract ValidationMessage getInvalidMultiplicationExprVM();
	protected abstract ValidationMessage getInvalidUnaryExprVM();
	protected abstract ValidationMessage getGenericInvalidTimerExprVM();
	
	private ValidationMessage validateAdditionExpression(String operand1, String operand2) {
		if(! (isValidShortDuration(operand1) && isValidShortDuration(operand2)))
			return getInvalidAdditionExprVM();
		else
			return null;
	}
	
	private ValidationMessage validateMultiplicationExpression(String operand1, String operand2) {
		if(! (isValidShortDuration(operand1) && isValidMultiplier(operand2)))
			return getInvalidMultiplicationExprVM();
		else
			return null;
	}
	
	private ValidationMessage validateUnaryExpression(String operand) {
		if(!TimeoutValidation.isTimeoutExpressionValid(operand) )
			return getInvalidUnaryExprVM();
		else
			return null;
	}

	private boolean isValidShortDuration(String operand) {
		return (isShortDurationConstant(operand) || isVariableOfTypeDuration(operand));
	}
	
	private boolean isShortDurationConstant(String constant) {
		return TimeoutValidation.isTimeOutFormatValid("HH:mm:ss", constant);
	}
	
	private boolean isVariableOfTypeDuration(String varName) {
		for(WPVariable variable : variables) {
			if(variable.getName() != null && variable.getName().equals(varName)) {
				if(variable.getType() != null && variable.getType().equals("DURATION"))
					return true;
				else
					return false;
			}
		}
		return false;
	}
	
	private boolean isValidMultiplier(String operand) {
		return (isValidMultiplicativeConstant(operand) || isVariableOfTypeStringOrInt(operand));
	}
	
	private boolean isValidMultiplicativeConstant(String constant) {
		try {
			return (Double.parseDouble(constant) > 0);
		} catch(NumberFormatException e) {
			return false;
		}
	}

	private boolean isVariableOfTypeStringOrInt(String varName) {
		for(WPVariable variable : variables) {
			if(variable.getName() != null && variable.getName().equals(varName)) {
				if(variable.getType() != null && 
						(variable.getType().equals("STRING") || variable.getType().equals("INT")))
					return true;
				else
					return false;
			}
		}
		return false;
	}
	
}
