package com.tandbergtv.watchpoint.studio.validation.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.jbpm.gd.jpdl.model.Condition;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.ProcessState;
import org.jbpm.gd.jpdl.model.Transition;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.model.AssignNode;
import com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.MailNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementVisitor;
import com.tandbergtv.watchpoint.studio.ui.model.WPAbstractNode;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.DecisionExprValidationUtil;

/**
 * Each visit removes the used variable from the remaining list. The variables remaining are the ones not used.
 * 
 * These are the nodes that contain expressions and can be using variables:
 * assignNode (variable value field)
 * nodeDefinition (variable name field)
 * subProcess (variable name field)
 * TODO: use java expressions for those.
 * 
 * These are the other variable container nodes, which does not have expressions:
 * loopNode
 * automaticTaskNode
 * manualTaskNode
 * decisionNode
 * superState (node group)
 * mailNode
 */
public class CheckVariableUsageVisitor extends SemanticElementVisitor {

    private final List<Variable> remainingVariables;

    public CheckVariableUsageVisitor(Variable[] variables) {
        this.remainingVariables = new ArrayList<Variable>(Arrays.asList(variables));
    }

    public List<Variable> getUnusedVariables() {
        return remainingVariables;
    }

    public void visit(AssignNode semanticElement) {
        String path = Utility.getTemplateLocation(semanticElement);
        IProject project = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(path)).getProject();

        List<String> usedVariables = new ArrayList<String>();
        if (semanticElement.getTasks() != null && semanticElement.getTasks().length > 0 && semanticElement.getTasks()[0].getController() != null) {
            WPVariable wpVariable = null;
            for (Variable variable : semanticElement.getTasks()[0].getController().getVariables()) {
                wpVariable = (WPVariable) variable;
                usedVariables.add(wpVariable.getName());

                DataType dataType = wpVariable.getType() != null ? DataType.valueOf(wpVariable.getType()) : null;
                Set<String> usedVariablesInExpression = Utility.getUsedVariablesInScript(project, dataType, wpVariable.getValue(),
                        Utility.getVariables(semanticElement).toArray(new WPVariable[0]));

                usedVariables.addAll(usedVariablesInExpression);
            }
        }

        removeUsedVariables(usedVariables);
    }

    public void visit(ProcessState semanticElement) {
        String path = Utility.getTemplateLocation(semanticElement);
        IProject project = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(path)).getProject();

        List<String> usedVariables = new ArrayList<String>();
        WPVariable wpVariable = null;
        for (Variable variable : semanticElement.getVariables()) {
            wpVariable = (WPVariable) variable;

            if (!variable.isWritable()) {
                DataType dataType = wpVariable.getType() != null ? DataType.valueOf(wpVariable.getType()) : null;
                Set<String> usedVariablesInExpression = Utility.getUsedVariablesInScript(project, dataType, wpVariable.getName(),
                        Utility.getVariables(semanticElement).toArray(new WPVariable[0]));

                usedVariables.addAll(usedVariablesInExpression);
            } else {
                usedVariables.add(variable.getName());
            }
        }

        removeUsedVariables(usedVariables);
    }

    public void visit(LoopNode semanticElement) {
        List<String> usedVariables = new ArrayList<String>();
        if (semanticElement.getIndex() != null) {
            usedVariables.add(semanticElement.getIndex());
        }

        if (semanticElement.getListName() != null) {
            usedVariables.add(semanticElement.getListName());
        }

        Map<Integer, String> operands = DecisionExprValidationUtil.getOperands(semanticElement.getExpression());

        for (String operand : operands.values()) {
            usedVariables.add(operand);
        }

        removeUsedVariables(usedVariables);
    }

    /**
     * For ManualTaskNode, AutomaticTaskNode and NodeDefinition.
     */
    public void visit(IWPVariableContainer semanticElement) {
        IProject project = null;
        if (semanticElement instanceof NodeDefinition) {
            String path = Utility.getTemplateLocation((NodeDefinition)semanticElement);
            project = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(path)).getProject();
        }

        List<String> usedVariables = new ArrayList<String>();
        for (WPVariable wpVariable : semanticElement.getVariables()) {
            if (semanticElement instanceof NodeDefinition && wpVariable.isReadable()) {
                DataType dataType = wpVariable.getType() != null ? DataType.valueOf(wpVariable.getType()) : null;

                Set<String> usedVariablesInExpression = Utility.getUsedVariablesInScript(project, dataType,
                        wpVariable.getName(),
                        Utility.getVariables((WPAbstractNode) semanticElement).toArray(new WPVariable[0]));

                usedVariables.addAll(usedVariablesInExpression);

            } else {
                usedVariables.add(wpVariable.getName());

            }
        }

        removeUsedVariables(usedVariables);
    }

    public void visit(Decision semanticElement) {
        List<String> usedVariables = new ArrayList<String>();
        if (semanticElement.getExpression() != null) {
            usedVariables.addAll(getOperandVariables(semanticElement.getExpression()));
        } else {
            for (Transition transition : semanticElement.getTransitions()) {
                Condition condition = transition.getCondition();

                if (condition != null && condition.getExpression() != null) {
                    usedVariables.addAll(getOperandVariables(condition.getExpression()));
                }
            }
        }

        removeUsedVariables(usedVariables);
    }

    public void visit(MailNode semanticElement) {
        List<String> usedVariables = new ArrayList<String>();

        usedVariables.add(semanticElement.getTo());
        if (semanticElement.getText() != null) {
            usedVariables.add(semanticElement.getText().getText());
        }
        usedVariables.add(semanticElement.getTemplate());
        if (semanticElement.getSubject() != null) {
            usedVariables.add(semanticElement.getSubject().getSubject());
        }
        usedVariables.add(semanticElement.getActors());

        removeUsedVariables(usedVariables);
    }

    private List<String> getOperandVariables(String expression) {
        List<String> usedVariables = new ArrayList<String>();
        Map<Integer, String> operands = DecisionExprValidationUtil.getOperands(expression);

        for (String operand : operands.values()) {
            usedVariables.add(operand);
        }
        return usedVariables;
    }

    private void removeUsedVariables(List<String> usedVariables) {
        Variable variable = null;
        for (String variableName : usedVariables) {
            for (Iterator<Variable> it = remainingVariables.iterator(); it.hasNext();) {
                variable = it.next();
                if (variable.getName().equals(variableName)) {
                    it.remove();
                }
            }
        }
    }
}