/**
 * 
 */
package com.tandbergtv.watchpoint.studio.validation.rules.nodedefinition.graph;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.EndState;
import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.graph.NodeDefinitionGraph;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

import edu.uci.ics.jung.graph.Vertex;

/**
 * Rule that validates that the Node Definition Graph contains all the required nodes (with the
 * correct multiplicity). Checks that the graph contains atleast one End node, and at
 * least one other type of node.
 * 
 * @author Imran Naqvi
 *
 */
public class RequiredNodeDefinitionNodesRule extends ValidationRule<NodeDefinitionGraph> {

	/**
	 * Validates that the Graph element contains the required nodes.
	 * 
	 * @param graph
	 *            The object being validated
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateRule(NodeDefinitionGraph graph)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		int endNodeCount = 0;
		boolean otherNodesExist = false;

		Set<Vertex> vertices = graph.getVertices();
		for (Vertex vertex : vertices)
		{
			NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);

			if (node instanceof EndState)
			{
				endNodeCount++;
			}
			else
			{
				otherNodesExist = true;
			}
		}

		Object graphElement = WatchPointGraphUtils.getElement(graph);

		this.validateEndNodeCount(endNodeCount, messages, graphElement);
		this.validateNodeCount(otherNodesExist, messages, graphElement);

		return messages;
	}

	/*
	 * Validate the End Node Count (must be >= 1).
	 */
	private void validateEndNodeCount(int count, List<ValidationMessage> messages, Object element)
	{
		/* Check if the End node is missing */
        if (count == 0) {
            ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.NODEDEFINITION_MISSING_END_NODE);
        }
	}

	/*
	 * Validate the Node Count of all nodes other than the Start and End nodes (must be > 0).
	 */
	private void validateNodeCount(boolean nodeExists, List<ValidationMessage> messages,
			Object element)
	{
		/* Check if there are no other nodes present */
        if (!nodeExists) {
            ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.NODEDEFINITION_MISSING_NODE);
        }
	}

}
