/*
 * Created on Nov 13, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement;

import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.ROLLBACK_VARIABLE_MAPPING_INVALID_IN_TEMPLATE;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.ROLLBACK_VARIABLE_NOT_MAPPED_IN_TEMPLATE;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.ROLLBACK_VARIABLE_TYPE_MISMATCH_IN_TEMPLATE;

import java.util.ArrayList;
import java.util.List;

import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.model.ContainerNode;

/**
 * Validates the Variable Mapping for Node Elements that implement IWPVariableContainer.
 * 
 * @author Vijay Silva
 */
public class RollbackVariableContainerVariableMappingRule extends NodeVariablesMappingRule {
	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesMappingRule#getVariables(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected List<WPVariable> getVariables(NodeElement node) {
		List<WPVariable> variables = new ArrayList<WPVariable>();

		if (node instanceof NodeDefinition) {
			NodeDefinition nd = (NodeDefinition) node;
			List<WPVariable> nodeVariables = nd.getRollbackVariables();
			// For Rollback only outgoing messages are supported,
			// so all variables are required.
			if (nodeVariables != null)
				variables.addAll(nodeVariables);
		}

		return variables;
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/**
	 * Gets the Validation Message Code if the variable is not mapped to an existing container
	 * variable.
	 * 
	 * @param node
	 *            The node being validated.
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLE_NOT_MAPPED_IN_TEMPLATE} if the container
	 *         is a template, or
	 *         {@link ValidationMessageCode#NODE_VARIABLE_NOT_MAPPED_IN_NODE_DEFINITION} if the
	 *         container is a Node Definition.
	 */
	protected ValidationMessageCode getMappingMissingCode(ContainerNode node) {
		return ROLLBACK_VARIABLE_NOT_MAPPED_IN_TEMPLATE;
	}

	/**
	 * Gets the Validation Message Code if the variable is mapped to a non-existing container
	 * variable.
	 * 
	 * @param node
	 *            The node being validated.
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLE_MAPPING_INVALID_IN_TEMPLATE} if the
	 *         container is a template, or
	 *         {@link ValidationMessageCode#NODE_VARIABLE_MAPPING_INVALID_IN_NODE_DEFINITION} if the
	 *         container is a Node Definition.
	 */
	protected ValidationMessageCode getMappingInvalidCode(ContainerNode node) {
		return ROLLBACK_VARIABLE_MAPPING_INVALID_IN_TEMPLATE;
	}

	/**
	 * Gets the Validation Message Code if the variable type mismatches with the container variable
	 * type.
	 * 
	 * @param node
	 *            The node being validated.
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLE_TYPE_MISMATCH_IN_TEMPLATE} if the
	 *         container is a template, or
	 *         {@link ValidationMessageCode#NODE_VARIABLE_TYPE_MISMATCH_IN_NODE_DEFINITION} if the
	 *         container is a Node Definition.
	 */
	protected ValidationMessageCode getMappingTypeMismatchCode(ContainerNode node) {
		return ROLLBACK_VARIABLE_TYPE_MISMATCH_IN_TEMPLATE;
	}
}
