/*
 * Created on Nov 9, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.decision;

import java.util.ArrayList;
import java.util.List;

import org.jbpm.gd.jpdl.model.Condition;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Rule that validates that a Decision Node is configured to either have the expression at the
 * decision node, or at each of the outgoing Transitions. The expression cannot be set at both or be
 * missing in both.
 * 
 * @author Vijay Silva
 */
public class DecisionConfigurationRule extends ValidationRule<Decision>
{
	/**
	 * Validates that the decision is correctly configured. The decision can either be configured at
	 * the decision node, or at each of the Transition nodes.
	 * 
	 * @param target
	 *            The decision node to validate
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(Decision target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		boolean nodeConfigured = false;
		boolean transitionConfigured = false;

		/* Check if the node is configured */
		String expression = target.getExpression();
		nodeConfigured = (expression != null);

		/* Check if any and all the transitions are configured */
		Transition[] transitions = target.getTransitions();
		if (transitions != null)
		{
			for (Transition transition : transitions)
			{
				Condition condition = transition.getCondition();
				expression = (condition != null) ? condition.getExpression() : null;
				if (expression != null)
					transitionConfigured = true;
			}
		}

		/* Validate that either the node or the transitions are configured */
		this.validateConfiguration(target, nodeConfigured, transitionConfigured, messages);

		/* If transitions are configured, validate that all transitions are configured */
		if (transitionConfigured && !nodeConfigured)
			this.validateTransitionConfiguration(target, messages);

		return messages;
	}

	/*
	 * Validate that either the node or the transitions are configured.
	 */
	private void validateConfiguration(Decision target, boolean node, boolean transition,
			List<ValidationMessage> messages)
	{
		if (node == transition)
		{ // Invalid Configuration
			ValidationMessageCode code = (node)
					? ValidationMessageCode.NODE_DECISION_CONFIGURATION_DUPLICATED
					: ValidationMessageCode.NODE_DECISION_CONFIGURATION_MISSING;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, target, code);
		}
	}

	/*
	 * Validate that all the outgoing transitions from the Decision node have an expression
	 * configured.
	 */
	private void validateTransitionConfiguration(Decision target, List<ValidationMessage> messages)
	{
		for (Transition transition : target.getTransitions())
		{
			Condition condition = transition.getCondition();
			String expression = (condition != null) ? condition.getExpression() : null;
			if (expression == null || expression.trim().length() == 0)
			{
				ValidationMessageCode code = ValidationMessageCode.NODE_TRANSITION_CONDITION_MISSING;
				List<String> parameters = new ArrayList<String>();
				parameters.add(target.getName());
				ValidationMessageAdder.getInstance().addValidationMessage(messages, transition, code, parameters);
			}
		}
	}
}
