/*
 * Created on Jul 17, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.EndState;
import org.jbpm.gd.jpdl.model.Fork;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.StartState;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

import edu.uci.ics.jung.graph.Vertex;
import edu.uci.ics.jung.graph.impl.DirectedSparseGraph;

/**
 * Validation rule that ensures that the Node In-Degree and Out-Degree of every node in the node container
 * is valid based on the type of the node.
 * 
 * @author Vijay Silva
 */
public class GraphNodeDegreeRule extends ValidationRule<DirectedSparseGraph>
{
	/**
	 * Validates that the Graph element contains nodes with the correct in-degree and out-degree.
	 * 
	 * @param graph
	 *            The object being validated
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateRule(DirectedSparseGraph graph)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		Set<Vertex> vertices = graph.getVertices();

		for (Vertex vertex : vertices)
		{
			this.validateNodeInDegree(vertex, messages);
			this.validateNodeOutDegree(vertex, messages);
		}

		return messages;
	}

	/*
	 * Validate the In-Degree of the Vertex depending on the type of Node it contains.
	 */
	protected void validateNodeInDegree(Vertex vertex, final List<ValidationMessage> messages)
	{
		NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);
		int inDegree = vertex.inDegree();

		if (node instanceof StartState)
		{
			if (inDegree != 0)
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.INVALID_START_NODE_IN_DEGREE);
			}
		}
		else
		{
			if (inDegree < 1)
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.INVALID_NODE_IN_DEGREE);
			}
		}
	}

	/*
	 * Validate the Out-Degree of the Vertex depending on the type of Node it contains.
	 */
	protected void validateNodeOutDegree(Vertex vertex, final List<ValidationMessage> messages)
	{
		NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);
		int outDegree = vertex.outDegree();

		if (node instanceof EndState)
		{
			if (outDegree != 0)
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.INVALID_END_NODE_OUT_DEGREE);
			}
		}
		else if (node instanceof Decision)
		{
			if (outDegree < 2)
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.INVALID_DECISION_NODE_OUT_DEGREE);
			}
		}
		else if (node instanceof Fork)
		{
			if (outDegree < 2)
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.INVALID_FORK_NODE_OUT_DEGREE);
			}
		}
		else
		{
			if (outDegree == 0)
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.NODE_OUT_DEGREE_LOW);
			}
			else if (outDegree > 1)
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.NODE_OUT_DEGREE_HIGH);
			}
		}
	}
}
