/**
 * NodeGroup.java
 * Created Feb 15, 2007
 * Copyright (C) Tandberg Television 2007
 */
package com.tandbergtv.watchpoint.studio.debugger.core.graph;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.dom4j.Element;
import org.jbpm.graph.def.Node;
import org.jbpm.graph.def.SuperState;
import org.jbpm.graph.def.Transition;
import org.jbpm.jpdl.xml.JpdlXmlReader;

/**
 * Extends the JBPM concept of a group of nodes to add resource awareness
 * 
 * @author Sahil Verma
 */
public class SuperState2 extends SuperState {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1214915925481664560L;

	private Long resourceGroupID;
	
	private String duration;
	
	private String taskDuration;
	
	private static final Long DEFAULT_RESOURCE_GROUP_ID = -1L;
	
	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<Node> getNodes() {
		return super.getNodes();
	}

	/**
	 * @return the duration
	 */
	public String getDuration() {
		return this.duration;
	}

	/**
	 * @param duration the duration to set
	 */
	public void setDuration(String duration) {
		this.duration = duration;
	}

	/**
	 * @return the resourceGroupID
	 */
	public Long getResourceGroupID() {
		if (this.resourceGroupID.equals(DEFAULT_RESOURCE_GROUP_ID))
			return (Long)null;
		
		return this.resourceGroupID;
	}

	/**
	 * @param resourceGroupID the resourceGroupID to set
	 */
	public void setResourceGroupID(Long resourceGroupID) {
		this.resourceGroupID = (resourceGroupID == null) ? DEFAULT_RESOURCE_GROUP_ID : resourceGroupID;
	}

	/**
	 * @return the taskDuration
	 */
	public String getTaskDuration() {
		return this.taskDuration;
	}

	/**
	 * @param taskDuration the taskDuration to set
	 */
	public void setTaskDuration(String taskDuration) {
		this.taskDuration = taskDuration;
	}

	/* (non-Javadoc)
	 * @see org.jbpm.graph.def.SuperState#read(org.dom4j.Element, org.jbpm.jpdl.xml.JpdlXmlReader)
	 */
	@Override
	public void read(Element element, JpdlXmlReader jpdlReader) {
		action = jpdlReader.readSingleAction(element);
		setDuration(element.attributeValue("duration"));
		setTaskDuration(element.attributeValue("taskDuration"));
		super.read(element, jpdlReader);
	}

	/**
	 * Returns the first node in this group
	 * @see SuperState2#last()
	 * 
	 * @return
	 */
	public Node first() {
		for (Object obj : this.nodes) {
			Node node = (Node) obj;
			
			if (node.getArrivingTransitions() == null || node.getArrivingTransitions().isEmpty())
				return node;
			
			for (Object transObj : node.getArrivingTransitions()) {
				Transition transition = (Transition) transObj;
				SuperState superstate = transition.getFrom().getSuperState();
				
				if (superstate == null || !(superstate.equals(this)))
					return node;
			}
		}
		
		return null;
	}
	
	/**
	 * Returns the last node. The last node is defined as a child node that has at least 
	 * one leaving transition out of this super-state.
	 * 
	 * Note that the nodes aren't always ordered correctly - the last node in the collection of
	 * children is not guaranteed to be the 'last'.
	 * 
	 * @see SuperState2#first()
	 * @return the 'last' node
	 */
	public Node last() {
		return (Node)nodes.get(nodes.size() - 1);
	}

	/**
	 * A superstate doesn't really have any leaving transitions, get the transitions of children that 
	 * have leaving transitions to another node
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<?> getLeavingTransitions() {
		List<Transition> list = new ArrayList<Transition>();
		
		for (Node child  : getNodes()) {
			if (child.getLeavingTransitions() == null)
				continue;
			
			for (Transition transition : (Collection<Transition>) child.getLeavingTransitions()) {
				if (!this.equals(transition.getTo().getSuperState()))
					list.add(transition);
			}
		}
		
		return list;
	}

	/**
	 * A superstate doesn't really have arriving transitions, get the transitions of a child that
	 * has arriving transitions from another node.
	 */
	@Override
	public Set<?> getArrivingTransitions() {
		Set<Transition> set = new HashSet<Transition>();
		Node first = first();
		
		if (first.getArrivingTransitions() == null) // True only if parent is loop
			return set;
		
		for (Object obj : first.getArrivingTransitions()) {
			Transition transition = Transition.class.cast(obj);
			
			if (!this.equals(transition.getFrom().getSuperState()))
				set.add(transition);
		}
		
		return set;
	}
	
}
