package com.tandbergtv.watchpoint.studio.debugger.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.jbpm.gd.jpdl.model.Variable;

/**
 * 		The Table that shows the Template start nodes.
 *  
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class VariableValuesTable {
	
    protected Table table;
	protected TableViewer tableViewer;
	private List<SimulationModifyListener> simulationModifyListeners = new ArrayList<SimulationModifyListener>();
    protected static final int VALUE_COLUMN_INDEX = 2;
	
	protected String lastSortedColumn;
	protected boolean sortDesc;
	
	public final static String VARIABLE_COLUMN = "variable";
	public final static String TYPE_COLUMN = "type";
	public final static String VALUE_COLUMN = "value";
	public static final String[] COLUMN_NAMES = new String[] { 
		VARIABLE_COLUMN,
		TYPE_COLUMN,
		VALUE_COLUMN,
	};
	
	public VariableValuesTable(Composite parent) {
		this.createControls(parent);
	}
	
	/**
	 * 		Adds the Table controls to the composite.
	 * 
	 * @param composite
	 */
	private void createControls(Composite composite) {
		// Create the table 
		createTable(composite);
		// Create and setup the TableViewer
		createTableViewer();
	}
	
	public void setVariablesData(Collection<Variable> variables) {
		tableViewer.setInput(variables);
	}
	
	@SuppressWarnings("unchecked")
	public Collection<Variable> getVariablesData() {
		return (Collection<Variable>) tableViewer.getInput();
	}
	
	/**
	 * Create the Table
	 */
	private void createTable(Composite parent) {
		int style = SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | 
					SWT.FULL_SELECTION | SWT.HIDE_SELECTION;

		table = new Table(parent, style);
		
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 2;
		table.setLayoutData(gridData);		
					
		table.setLinesVisible(true);
		table.setHeaderVisible(true);

		// 1st column with Variable
		TableColumn column = new TableColumn(table, SWT.LEFT, 0);
		column.setText("Variable");
		column.setWidth(150);
		// Add listener for sorting the column  
		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				sortTable(VARIABLE_COLUMN);
			}
		});

		// 2nd column with Type
		column = new TableColumn(table, SWT.LEFT, 1);
		column.setText("Type");
		column.setWidth(150);
		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				sortTable(TYPE_COLUMN);
			}
		});
		
		// 3rd column with Value 
		column = new TableColumn(table, SWT.LEFT, 2);
		column.setText("Value");
		column.setWidth(300);
		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				sortTable(VALUE_COLUMN);
			}
		});
	}

	protected void sortTable(String column) {
		if (column.equals(lastSortedColumn)) {
			sortDesc = !sortDesc;
		} else {
			sortDesc = false;
		}
		VariableValuesTableComparator comparator = new VariableValuesTableComparator(column, sortDesc, (ITableLabelProvider) tableViewer.getLabelProvider());
		tableViewer.setComparator(comparator);
		lastSortedColumn = column;
	}

	/**
	 * Create the TableViewer 
	 */
	private void createTableViewer() {
		tableViewer = new TableViewer(table);
		tableViewer.setUseHashlookup(true);
		tableViewer.setColumnProperties(COLUMN_NAMES);

		// Create the cell editors
		CellEditor[] editors = new CellEditor[COLUMN_NAMES.length];
		// Column 2 : Value
		editors[VALUE_COLUMN_INDEX] = new TextCellEditor(table);

		// Assign the cell editors to the viewer 
		tableViewer.setCellEditors(editors);
		// Set the cell modifier for the viewer
		tableViewer.setCellModifier(new VariableValuesTableCellModifier(this));
		tableViewer.setContentProvider(ArrayContentProvider.getInstance());
		tableViewer.setLabelProvider(new VariableValuesTableLabelProvider());
	}

	/*
	 * Close the window and dispose of resources
	 */
	public void close() {
		Shell shell = table.getShell();

		if (shell != null && !shell.isDisposed())
			shell.dispose();
	}

	/**
	 * Return the column names in a collection
	 * 
	 * @return List  containing column names
	 */
	public List<String> getColumnNames() {
		return Arrays.asList(COLUMN_NAMES);
	}

	/**
	 * @return currently selected item
	 */
	public ISelection getSelection() {
		return tableViewer.getSelection();
	}

	/**
	 * Return the parent composite
	 */
	public Control getControl() {
		return table.getParent();
	}
	
	public void updateNode(Variable var) {
		tableViewer.update(var, null);
		for (SimulationModifyListener varModifyListener : simulationModifyListeners) {
			varModifyListener.modifySimulation(var);
		}
	}
	
	public void refresh() {
		tableViewer.refresh();
	}
	
	public void addSelectionListener(SelectionListener listener) {
		table.addSelectionListener(listener);
	}
	
	public void addSimulationModifyListener(SimulationModifyListener listener) {
		simulationModifyListeners.add(listener);
	}
	
	public void removeSelectionListener(SelectionListener listener) {
		table.removeSelectionListener(listener);
	}
	
	public void setVisible(boolean visible) {
		table.setVisible(visible);
	}
	
	public void setEnabled(boolean enabled) {
		table.setEnabled(enabled);
	}
	
	/**
	 * Release resources
	 */
	public void dispose() {
		// Tell the label provider to release its ressources
		tableViewer.getLabelProvider().dispose();
	}

	public void setLabelProvider(ITableLabelProvider labelProvider) {
		tableViewer.setLabelProvider(labelProvider);
	}
}
