package com.tandbergtv.watchpoint.studio.builder;

import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.resources.ProjectExplorer;

import com.tandbergtv.watchpoint.studio.ui.model.DataCollectionManager;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer;

/**
 * Resource type builder
 */
public class WatchpointResourceTypeBuilder extends IncrementalProjectBuilder {

	public static final String BUILDER_ID = "org.jbpm.gd.jpdl.resourceTypeBuilder";

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("rawtypes")
	@Override
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		if (kind == FULL_BUILD) {
			fullBuild(monitor);
		} else {
			IResourceDelta delta = getDelta(getProject());
			if (delta == null) {
				fullBuild(monitor);
			} else {
				monitor.beginTask("Building resource type", IProgressMonitor.UNKNOWN);
				incrementalBuild(delta, monitor);
				monitor.worked(1);
				monitor.done();
			}
		}
		
		refreshViews();
		
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void clean(IProgressMonitor monitor) throws CoreException {
		IProject project = getProject();
		new ResourceGroupVisitor().delete(project);
		new WatchpointResourceTypeVisitor().delete(project);
		DataCollectionManager.getInstance().removeAll(project.getName());
	}

	protected void fullBuild(final IProgressMonitor monitor) throws CoreException {
		try {
			monitor.beginTask("Building resource type", IProgressMonitor.UNKNOWN);

			getProject().accept(new WatchpointResourceTypeVisitor());
			getProject().accept(new ResourceTypeDataCollectorVisitor());

			monitor.worked(1);
			monitor.done();
		} catch (CoreException e) {
		}
	}

	protected void incrementalBuild(IResourceDelta delta, IProgressMonitor monitor) throws CoreException {
		// the visitor does the work.
		monitor.subTask("Building Resource Type");
		delta.accept(new WatchpointResourceTypeVisitor());
        delta.accept(new ResourceTypeDataCollectorVisitor());
		monitor.done();
	}

	public static void refreshViews() {
		PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
			public void run() {
				IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				// may be null during eclipse shutdown...
				if (dwindow == null)
					return;

				IWorkbenchPage page = dwindow.getActivePage();
				
				if (page != null) {
					IViewPart part = page.findView(Utility.ID_RESOURCE_TYPE);
					
					if (part != null)
						AbstractTreeViewExplorer.class.cast(part).refresh();
					
					part = page.findView(Utility.ID_RESOURCE_GROUP);
					
					if (part != null)
						AbstractTreeViewExplorer.class.cast(part).refresh();

					part = page.findView(ProjectExplorer.VIEW_ID);
					
					if (part != null) {
						TreeViewer viewer = (TreeViewer) part.getAdapter(CommonViewer.class);
						TreePath[] treePaths = viewer.getExpandedTreePaths();
				        
				        viewer.refresh();
				        viewer.setExpandedTreePaths(treePaths);
					}
				}
			}
		});
	}
}
