/*
 * Created on Jun 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess;

import java.util.Collection;
import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;

/**
 * The Data Access Interface for the Resource Group entity.
 * 
 * @author Vijay Silva
 */
public interface ResourceGroupDAI extends DataAccessInterface<ResourceGroup, Long>
{
	/**
	 * Gets the list of Resource Groups that belong to the Resource Type with a matching System Id.
	 * 
	 * @param systemId
	 *            The Resource Type System Id
	 * 
	 * @return The List of Resource Groups with a matching Resource Type System Id
	 */
	List<ResourceGroup> findByResourceTypeSystemId(String systemId);

	/**
	 * Finds Resource Groups based on the given set of names.
	 * 
	 * @param names
	 *            The Collection of Resource Group Names
	 * 
	 * @return The List of Resource Groups
	 */
	List<ResourceGroup> findByNames(Collection<String> names);

	/**
	 * Gets the Resource Groups with a matching name
	 * 
	 * @param name
	 *            The Resource Group Name
	 * 
	 * @return The Resource Groups with matching name
	 */
	List<ResourceGroup> findByName(String name);
	
	/**
	 * Gets the Resource Groups with a matching project
	 * 
	 * @param name
	 *            The project name
	 * 
	 * @return The Resource Groups with matching project
	 */
	List<ResourceGroup> findByProject(String project);
	
	/**
	 * Gets the Resource Group with a matching path
	 * 
	 * @param path
	 *            The Resource Group path
	 * 
	 * @return The Resource Group with matching path
	 */
	ResourceGroup findByPath(String path);

	/**
	 * Gets the Protection Key for the Resource Group with a matching name
	 * 
	 * @param name
	 *            The Resource Group Name
	 * 
	 * @return The Protection Key of the Resource Group with matching name
	 */
	ProtectionKey findProtectionKeyByName(String name);

	/**
	 * Gets functional types used by all Resource Groups that are of the given Resource Type
	 * 
	 * @param resourceTypeSystemId
	 *            The Resource Type system Id
	 * 
	 * @return The List of Functional Types for the given Resource Type
	 */
	List<String> getAllFunctionalTypesForResourceType(String resourceTypeSystemId);

	/**
	 * Checks if a Resource Group with the given name (case insensitive) already exist
	 * 
	 * @param name
	 *            The Resource Group Name
	 * 
	 * @return true if a Resource Group with the given name (case insensitive) exist; false
	 *         otherwise
	 */
	boolean isNameExists(String name);

	/**
	 * Gets the Count of Resource Groups given the Resource Type they belong to.
	 * 
	 * @param resourceTypeId
	 *            The Resource Type system Id
	 * 
	 * @return The number of Resource Groups that are of matching Resource Type.
	 */
	int getCountByResourceType(String resourceTypeSystemId);

	/**
	 * Gets the Resource Groups for the Template specified
	 * 
	 * @param name
	 *            The project name
	 * 
	 * @return The Resource Groups with matching project
	 */
	List<ResourceGroup> findByTemplateUsage(String templatePath);

}
