/*
 * Created on Jun 26, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess.jpa;

import java.io.Serializable;

import com.tandbergtv.watchpoint.studio.application.StudioRuntimeException;
import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory;
import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.MessageDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI;
import com.tandbergtv.watchpoint.studio.dataaccess.ProtectionKeyDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dto.IPersistable;

/**
 * Data Access Object Factory for JPA Data Access Objects.
 * 
 * @author Vijay Silva
 */
public class JPADataAccessFactory extends DataAccessFactory
{
	/**
	 * Constructor
	 */
	public JPADataAccessFactory()
	{
		super();
	}

	// ========================================================================
	// ==================== GENERIC OBJECT CREATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#createDataAccessObject(com.tandbergtv.watchpoint.studio.dto.IPersistable,
	 *      com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <DO extends IPersistable, DOK extends Serializable> DataAccessInterface<DO, DOK> createDataAccessObject(
			DO entity, IPersistenceContext context)
	{
		Class<? extends IPersistable> entityClass = entity.getClass();
		return (DataAccessInterface<DO, DOK>) this.createDataAccessObject(entityClass, context);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#createDataAccessObject(java.lang.Class,
	 *      com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <DO extends IPersistable, DOK extends Serializable> DataAccessInterface<DO, DOK> createDataAccessObject(
			Class<DO> entityClass, IPersistenceContext context)
	{
		// Assumptions:
		// 1. All JPA DAOs are present in the same package as this factory
		// 2. The naming convention for a JPA DAO class is <EntityName>PDAO
		// 3. Concrete JPA DAO classes have constructor with args: (JPAPersistenceContext context);

		String entityClassName = entityClass.getName();
		String daoClassName = null;

		try
		{
			String simpleEntityClassName = entityClass.getSimpleName();
			String daoPackageName = this.getClass().getPackage().getName();
			daoClassName = daoPackageName + "." + simpleEntityClassName + "PDAO";
			Class<?> clazz = Class.forName(daoClassName);
			Object instance = clazz.getConstructor(context.getClass()).newInstance(context);

			return (DataAccessInterface<DO, DOK>) instance;
		}
		catch (Exception ex)
		{
			String errMsg = "Failed to instantiate JPA Data Access Object: " + daoClassName
					+ " for entity class: " + entityClassName;
			throw new StudioRuntimeException(errMsg, ex);
		}
	}

	// ========================================================================
	// ==================== SPECIFIC OBJECT CREATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#getResourceTypeDAO(IPersistenceContext)
	 */
	@Override
	public ResourceTypeDAI getResourceTypeDAO(IPersistenceContext context)
	{
		return new ResourceTypePDAO((JPAPersistenceContext) context);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#getMessageDAO(IPersistenceContext)
	 */
	@Override
	public MessageDAI getMessageDAO(IPersistenceContext context)
	{
		return new MessagePDAO((JPAPersistenceContext) context);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#getResourceGroupDAO(IPersistenceContext)
	 */
	@Override
	public ResourceGroupDAI getResourceGroupDAO(IPersistenceContext context)
	{
		return new ResourceGroupPDAO((JPAPersistenceContext) context);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#getProtectionKeyDAO(IPersistenceContext)
	 */
	@Override
	public ProtectionKeyDAI getProtectionKeyDAO(IPersistenceContext context)
	{
		return new ProtectionKeyPDAO((JPAPersistenceContext) context);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#getNodeDefinitionDTODAO(IPersistenceContext)
	 */
	@Override
	public NodeDefinitionDTODAI getNodeDefinitionDTODAO(IPersistenceContext context)
	{
		return new NodeDefinitionDTOPDAO((JPAPersistenceContext) context);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory#getWorkflowTemplateDTODAO(IPersistenceContext)
	 */
	@Override
	public WorkflowTemplateDTODAI getWorkflowTemplateDTODAO(IPersistenceContext context)
	{
		return new WorkflowTemplateDTOPDAO((JPAPersistenceContext) context);
	}
}
