/*
 * Created on Mar 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.jpf;

import java.io.File;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import org.java.plugin.registry.Extension;
import org.java.plugin.registry.Extension.Parameter;

import com.tandbergtv.watchpoint.studio.external.jpf.model.AdaptorDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ManagementDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.MessageDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ResourceConnectionType;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ResourceTypeDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ScopeDescriptor;

/**
 * Utility class that parses the JPF Extension containing a Resource Type and produces the
 * ResourceTypeDescriptor object.
 * 
 * @author Vijay Silva
 */
public final class ExtensionParser {
	// Cannot instantiate
	private ExtensionParser() {
	}

	/**
	 * Utility method to parse the Resource Type Extension and produce a ResourceTypeDescriptor
	 * object that contains all the required data.
	 * 
	 * @param extension
	 *            The Resource Type Plugin Extension
	 * @param loader
	 *            The ClassLoader for the the Plugin
	 * 
	 * @return The ResourceTypeDescriptor object
	 */
	public static final ResourceTypeDescriptor parseResourceTypeExtension(Extension extension,
			ClassLoader loader) {
		ResourceTypeDescriptor descriptor = new ResourceTypeDescriptor();

		descriptor.setExtensionId(extension.getId());
		descriptor.setPluginId(extension.getDeclaringPluginDescriptor().getId());

		try {
			File file = new File(extension.getDeclaringPluginDescriptor().getLocation().toURI());
			String location = file.getParent();
			
			descriptor.setLocation(location);
		} catch (URISyntaxException e) {
			throw new RuntimeException(e);
		}

		Parameter param = extension.getParameter(PluginParameters.SYSTEM_ID);
		descriptor.setSystemId(param.valueAsString());

		param = extension.getParameter(PluginParameters.NAME);
		descriptor.setName(param.valueAsString());

		param = extension.getParameter(PluginParameters.CREATE_DATE);
		descriptor.setCreateDate(param.valueAsDate());

		param = extension.getParameter(PluginParameters.DESCRIPTION);
		if (param != null)
			descriptor.setDescription(param.valueAsString());

		param = extension.getParameter(PluginParameters.COMMUNICATION);
		if (param != null)
			parseCommunication(param, descriptor);

		param = extension.getParameter(PluginParameters.MANAGEMENT);
		parseManagement(param, descriptor);

		int version = extension.getDeclaringPluginDescriptor().getVersion().getMajor();
		descriptor.setVersion(version);

		descriptor.setClassLoader(loader);

		return descriptor;
	}

	/*
	 * Parse the top-level 'Communication' Parameter
	 */
	private static final void parseCommunication(Parameter parameter,
			ResourceTypeDescriptor descriptor) {
		Parameter param = parameter.getSubParameter(PluginParameters.MESSAGES);
		// Messages are not required anymore. Can exist a plugin.xml file with communication
		// parameter and without messages
		if(param != null){
			parseMessages(param, descriptor);
		}

		param = parameter.getSubParameter(PluginParameters.ADAPTOR);
		parseAdaptor(param, descriptor);
	}

	/*
	 * Parse the 'Messages' Parameter by reading the default protocol and all child messages.
	 */
	private static final void parseMessages(Parameter parameter, ResourceTypeDescriptor descriptor) {
		Map<String, MessageDescriptor> messageMap = new Hashtable<String, MessageDescriptor>();

		Parameter param = parameter.getSubParameter(PluginParameters.DEFAULT_PROTOCOL);
		String defaultProtocol = param.valueAsString();
		descriptor.setDefaultProtocol(defaultProtocol);

		int missingUIDCounter = 0;
		Collection<Parameter> messages = parameter.getSubParameters(PluginParameters.MESSAGE);
		for (Parameter message : messages) {
			MessageDescriptor messageDesc = new MessageDescriptor();

			param = message.getSubParameter(PluginParameters.MESSAGE_NAME);
			messageDesc.setName(param.valueAsString());

			param = message.getSubParameter(PluginParameters.MESSAGE_UID);
			messageDesc.setUID(param.valueAsString());

			param = message.getSubParameter(PluginParameters.MESSAGE_PROTOCOL);
			String protocol = (param != null) ? param.valueAsString() : defaultProtocol;
			messageDesc.setProtocol(protocol);

			param = message.getSubParameter(PluginParameters.MESSAGE_DESC);
			if (param != null)
				messageDesc.setDescription(param.valueAsString());

			// Check if the UID is not specified
			if (messageDesc.getUID() == null)
				messageDesc.setUID("UNKNOWN_UID_" + Integer.toString(missingUIDCounter++));

			messageMap.put(messageDesc.getUID(), messageDesc);
		}

		descriptor.setMessages(messageMap);
	}

	/*
	 * Parse the 'Adaptor' parameter to read the adaptor configuration
	 */
	private static final void parseAdaptor(Parameter parameter, ResourceTypeDescriptor descriptor) {
		AdaptorDescriptor adaptorDesc = new AdaptorDescriptor();

		Parameter param = parameter.getSubParameter(PluginParameters.ADAPTOR_DEFAULT);
		if (param != null) { // Read the default adaptor config
			Parameter child = param.getSubParameter(PluginParameters.ADAPTOR_MESSAGE_IDENTIFIER);
			if (child != null)
				adaptorDesc.setMessageIdentifierClass(child.valueAsString());
		} else {
			param = parameter.getSubParameter(PluginParameters.ADAPTOR_CUSTOM);
			Parameter child = param.getSubParameter(PluginParameters.ADAPTOR_CLASS);
			adaptorDesc.setAdaptorClass(child.valueAsString());

			child = param.getSubParameter(PluginParameters.ADAPTOR_SCOPE);
			String scope = (child != null) ? child.valueAsString() : ScopeDescriptor.APPLICATION
					.toString();
			adaptorDesc.setScope(ScopeDescriptor.valueOf(scope.toUpperCase()));

			child = param.getSubParameter(PluginParameters.ADAPTOR_PARAMETERS);
			if (child != null) {
				Map<String, String> properties = parseProperties(child);
				adaptorDesc.setProperties(properties);
			}
		}

		descriptor.setAdaptor(adaptorDesc);
	}

	/*
	 * Parse the 'Management' parameter
	 */
	private static final void parseManagement(Parameter parameter, ResourceTypeDescriptor descriptor) {
		ManagementDescriptor management = new ManagementDescriptor();

		Parameter param = parameter.getSubParameter(PluginParameters.CONNECTION_TYPE);
		management.setConnectionType(parseConnectionType(param.valueAsString()));

		param = parameter.getSubParameter(PluginParameters.INITIALIZATION);
		if (param != null) {
			Parameter child = param.getSubParameter(PluginParameters.INITIALIZATION_STRATEGY);
			management.setInitializationStrategyClass(child.valueAsString());

			child = param.getSubParameter(PluginParameters.INITIALIZATION_PARAMETERS);
			if (child != null) {
				Map<String, String> properties = parseProperties(child);
				management.setInitializationProperties(properties);
			}
		}

		param = parameter.getSubParameter(PluginParameters.HEARTBEAT);
		if (param != null) {
			Parameter child = param.getSubParameter(PluginParameters.HEARTBEAT_CONNECTION_TYPE);
			management.setHeartbeatConnectionType(parseConnectionType(child.valueAsString()));

			child = param.getSubParameter(PluginParameters.HEARTBEAT_STRATEGY);
			management.setHeartbeatStrategyClass(child.valueAsString());

			child = param.getSubParameter(PluginParameters.HEARTBEAT_PARAMETERS);
			if (child != null) {
				Map<String, String> hbProperties = parseProperties(child);
				management.setHeartbeatProperties(hbProperties);
			}
		}

		descriptor.setManagement(management);
	}

	/*
	 * Method to parse the ConnectionType value in the plugin to the ResourceConnectionType
	 * enumeration value.
	 */
	private static ResourceConnectionType parseConnectionType(String connectionType) {
		ResourceConnectionType result = null;

		if (connectionType != null) {
			if (connectionType.equals(PluginParameters.CONNECTION_TYPE_URL)) {
				result = ResourceConnectionType.URL;
			} else if (connectionType.equals(PluginParameters.CONNECTION_TYPE_IP)) {
				result = ResourceConnectionType.IP_PORT;
			} else if (connectionType.equals(PluginParameters.CONNECTION_TYPE_EMAIL)) {
				result = ResourceConnectionType.EMAIL;
			}
		}

		return result;
	}

	/*
	 * Parse the Properties that can be attached to a parameter
	 */
	private static Map<String, String> parseProperties(Parameter parameter) {
		Map<String, String> properties = new HashMap<String, String>();

		for (Parameter property : parameter.getSubParameters(PluginParameters.PARAMETER)) {
			Parameter key = property.getSubParameter(PluginParameters.PARAMETER_KEY);
			Parameter value = property.getSubParameter(PluginParameters.PARAMETER_VALUE);

			if (key.valueAsString() != null)
				properties.put(key.valueAsString(), value.valueAsString());
		}

		return properties;
	}
}
