/*
 * Created on Jul 5, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service.impl;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.eclipse.core.runtime.CoreException;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI;
import com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dto.IPersistable;
import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.ReferenceDeletionException;
import com.tandbergtv.watchpoint.studio.service.ServiceErrorCode;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceValidationException;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * The INodeDefinitionService implementation.
 * 
 * @author Vijay Silva
 */
public class NodeDefinitionService extends ServiceImpl implements INodeDefinitionService {

	// The XPath expression for any node definition element
	private static final String NODE_DEFINITION_ID_XPATH = "//nodeDefinition/@id";

	// The XPath expression for outgoing message uids
	private static final String SEND_MESSAGE_UID_XPATH = "//action/send";

	// The XPath expression for incoming message uids
	private static final String RECEIVE_MESSAGE_UID_XPATH = "//action/receive";
	
	private ResourceTypeFileSystemManager resourceTypeFileSystemManager;

	/**
	 * Constructor
	 */
	public NodeDefinitionService() {
		super();
		resourceTypeFileSystemManager = new ResourceTypeFileSystemManager();
	}

	// ========================================================================
	// ====== INODEDEFINITIONSERVICE METHODS
	// ========================================================================

	/**
	 * Get all available Node Definitions
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#getAllNodeDefinitions()
	 */
	@SuppressWarnings("unchecked")
	public List<NodeDefinitionDTO> getAllNodeDefinitions() {
		@SuppressWarnings("rawtypes")
		Class[] types = {};
		return (List<NodeDefinitionDTO>) super.performOperation(types);
	}

	/**
	 * Get all Node Definitions for a Resource Type.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#getNodeDefinitionsByResourceType(long)
	 */
	@SuppressWarnings("unchecked")
	public List<NodeDefinitionDTO> getNodeDefinitionsByResourceType(long resourceTypeId) {
		@SuppressWarnings("rawtypes")
		Class[] types = { long.class };
		return (List<NodeDefinitionDTO>) super.performOperation(types, resourceTypeId);
	}
	
	public NodeDefinitionDTO getNodeDefinitionDTOByPath(String path)
	{
		Class<?>[] types = { String.class };
		return (NodeDefinitionDTO) super.performOperation(types, path);
	}
	
	protected NodeDefinitionDTO getNodeDefinitionDTOByPath(String path, IPersistenceContext context)
	{
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return nodeDefinitionDAO.findByPath(path);
	}

	/**
	 * Get all Node Definitions for a Message.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#getNodeDefinitionsByMessageUID(String, com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<NodeDefinitionDTO> getNodeDefinitionsByMessageUID(String messageUID, NodeDefinitionType type) {
		@SuppressWarnings("rawtypes")
		Class[] types = { String.class, NodeDefinitionType.class};
		return (List<NodeDefinitionDTO>) super.performOperation(types, messageUID, type);
	}
	
	/**
	 * Get all Single Node Node Definitions for a Resource Type and filter by Message type
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#getSingleNodeDefinitionsByResourceTypeAndMessageType(long, boolean)
	 */
	@SuppressWarnings("unchecked")
	public List<NodeDefinitionDTO> getSingleNodeDefinitionsByResourceTypeAndMessageType(long resourceTypeId, boolean incoming) {
		@SuppressWarnings("rawtypes")
		Class[] types = { long.class, boolean.class };
		return (List<NodeDefinitionDTO>) super.performOperation(types, resourceTypeId, incoming);
	}

	public NodeDefinitionDTO getNodeDefinition(long nodeDefinitionId) {
		Class<?>[] types = { long.class };
		return (NodeDefinitionDTO) super.performOperation(types, nodeDefinitionId);
	}
	
	protected NodeDefinitionDTO getNodeDefinition(long nodeDefinitionId,
			IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return nodeDefinitionDAO.find(nodeDefinitionId);
	}
	
	/**
	 * Create a new Node Definition.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#createNodeDefinition(NodeDefinitionDTO,
	 *      String, boolean)
	 */
	public NodeDefinitionDTO createNodeDefinition(NodeDefinitionDTO nodeDefinition) {
		Class<?>[] types = { NodeDefinitionDTO.class };
		return (NodeDefinitionDTO) super.performOperation(types, nodeDefinition);
	}

	/**
	 * Delete a Node Definition specified by the given id.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#deleteNodeDefinition(long,
	 *      String, boolean)
	 */
	public void deleteNodeDefinition(long nodeDefinitionId, boolean removeReferences) {
		Class<?>[] types = { long.class, boolean.class };
		super.performOperation(types, nodeDefinitionId, removeReferences);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#nameExists(java.lang.String)
	 */
	public boolean nameExists(String name) {
		Class<?>[] types = { String.class };
		return (Boolean) super.performOperation(types, name);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.INodeDefinitionService#isUsedByNodeDefinition(long)
	 */
	public boolean isUsedByNodeDefinition(long nodeDefinitionId) {
		Class<?>[] types = { long.class };
		return (Boolean) super.performOperation(types, nodeDefinitionId);
	}

	// ========================================================================
	// ====== BUSINESS LOGIC IMPLEMENTATION (WITHOUT CONTEXT MANAGEMENT)
	// ========================================================================

	/**
	 * Implementation of the business logic for the getNodeDefinitions method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.NodeDefinitionService#getAllNodeDefinitions()
	 */
	protected List<NodeDefinitionDTO> getAllNodeDefinitions(IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return nodeDefinitionDAO.findAll();
	}

	/**
	 * Implementation of the business logic for the getNodeDefinitionsByResourceType method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.NodeDefinitionService#getNodeDefinitionsByResourceType(long)
	 */
	protected List<NodeDefinitionDTO> getNodeDefinitionsByResourceType(long resourceTypeId,
			IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return nodeDefinitionDAO.findByResourceType(resourceTypeId);
	}

	
	/**
	 * Implementation of the business logic for the getNodeDefinitionsByMessageUID method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.NodeDefinitionService#getNodeDefinitionsByMessageUID(String, com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType)
	 */
	public List<NodeDefinitionDTO> getNodeDefinitionsByMessageUID(String messageUID, NodeDefinitionType type, IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return nodeDefinitionDAO.findByMessageUID(messageUID, type);
	}
	
	/**
	 * Implementation of the business logic for the getSingleNodeDefinitionsByResourceType method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.NodeDefinitionService#getSingleNodeDefinitionsByResourceTypeAndMessageType(long, boolean)
	 */
	List<NodeDefinitionDTO> getSingleNodeDefinitionsByResourceTypeAndMessageType(long resourceTypeId, boolean incoming, 
			IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return nodeDefinitionDAO.findSingleNodeDefinitionsByResourceTypeAndMessageType(resourceTypeId, incoming);
	}

	/**
	 * Implementation of the business logic for the createNodeDefinition method.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#createTemplate(WorkflowTemplateDTO,
	 *      String, boolean)
	 */
	protected NodeDefinitionDTO createNodeDefinition(NodeDefinitionDTO nodeDefinition, IPersistenceContext context) {
		nodeDefinition.setId(IPersistable.DEFAULT_ID);
		
		// Code added in order to set the resource type id in the node definition XML
		// update: instead of using a dom engine to replace the string, it was manually 
		// replaced to reduce the build process' time. Has to be thoroughly tested before 
		// production.
		nodeDefinition.setXml(nodeDefinition.getXml().replaceFirst("resourceType=\"",
                "resourceType=\"" + nodeDefinition.getResourceType().getId()));

		// Get the xml document for this node definition
		Document nodeDefinitionDocument = this.loadNodeDefinitionDocument(nodeDefinition);

		// Trim the Node Definition name
		String nodeDefinitionName = nodeDefinition.getName();
		if (nodeDefinitionName != null)
			nodeDefinition.setName(nodeDefinitionName.trim());

		// Update the list of used node definitions
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		this.updateNodeDefinitionsUsed(nodeDefinition, nodeDefinitionDocument, nodeDefinitionDAO);

		// Update the used messages
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		this.updateMessagesUsed(nodeDefinition, nodeDefinitionDocument, resourceTypeDAO);
		
		// Validate the Node Definition being created
		this.validateNewNodeDefinition(nodeDefinition, context);

		// Save the Template to the database
		return nodeDefinitionDAO.create(nodeDefinition);
	}
	
	public void updateNodeDefinitionInFileSystem(NodeDefinitionDTO nodeDTO){
		try {
			resourceTypeFileSystemManager.updateNodeDefiniton(nodeDTO);
		} catch (Exception e) {
			throw new ServiceException(ServiceErrorCode.ND_UPDATE_IN_FILE_SYSTEM_ERROR, e);
		}
	}
	
	public void deleteNodeDefinitionInFileSystem(NodeDefinitionDTO nodeDTO, boolean removeReferences){
		IPersistenceContext context = this.createPersistenceContext();
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		
		NodeDefinitionDTO nodeDefinition = new NodeDefinitionDTO();
		
		ResourceType resourceType = createResourceTypeDAO(context).findByName(nodeDTO.getResourceType().getName());
		
		List<NodeDefinitionDTO> nodes = nodeDefinitionDAO.findByResourceType(resourceType.getId());
		for (NodeDefinitionDTO nodeDefinitionDTO : nodes) {
			if(nodeDefinitionDTO.getName().equals(nodeDTO.getName())){
				nodeDefinition = nodeDefinitionDTO;
				break;
			}
		}
		
		nodeDefinition = nodeDefinitionDAO.find(nodeDefinition.getId());

		/* Check if the node definition being deleted is being used by any template */
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		if (!removeReferences) {
			List<WorkflowTemplateDTO> templateList = new ArrayList<WorkflowTemplateDTO>();
			if (nodeDefinition.getType().equals(NodeDefinitionType.MessageNode)) {
				Collection<String> messageUIDs = new ArrayList<String>(); 
				for (Message msg : nodeDefinition.getMessages()) {
					messageUIDs.add(msg.getUid());
				}
				templateList = templateDAO.findByMessageUsage(messageUIDs, NodeDefinitionType.MessageNode);		
			} else {
				for (Message msg : nodeDefinition.getMessages()) {
					String systemID = Utility.getSystemIDByMessageUID(msg.getUid());
					templateList = templateDAO.findBySuperStateUsage(nodeDefinition.getName(), systemID);
					break;
				}
			}
			List<NodeDefinitionDTO> parentList = nodeDefinitionDAO.findByParentUsage(nodeDTO.getId());
			List<IWatchPointDTO> references = new ArrayList<IWatchPointDTO>();
			references.addAll(templateList);
			references.addAll(parentList);
			if (!references.isEmpty())
				throw new ReferenceDeletionException(ServiceErrorCode.ND_NODE_DEFN_REFERENCED,
						references);
		}
		
		try {
			resourceTypeFileSystemManager.deleteNodeDefinition(nodeDTO);
		} catch (CoreException e) {
			throw new ServiceException(ServiceErrorCode.ND_DELETE_IN_FILE_SYSTEM_ERROR, e);
		}
	}
	
	public void createNodeDefinitionInFileSystem(NodeDefinitionDTO nodeDTO){
		try {
			resourceTypeFileSystemManager.createNodeDefinition(nodeDTO);
		} catch (Exception e) {
			throw new ServiceException(ServiceErrorCode.ND_CREATE_IN_FILE_SYSTEM_ERROR, e);
		}
	}

	/**
	 * Implementation of the business logic for the deleteNodeDefinition method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.NodeDefinitionService#deleteNodeDefinition(long,
	 *      String, boolean)
	 */
	protected void deleteNodeDefinition(long nodeDefinitionId, boolean removeReferences, 
					IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		NodeDefinitionDTO nodeDefinition = nodeDefinitionDAO.find(nodeDefinitionId);

		// Check if the node definition's resource type is Out-of-the-box
		if(nodeDefinition.getResourceType().isOutOfTheBox()) {
			throw new ServiceException(ServiceErrorCode.ND_DELETE_RESTYPE_OOB_ERROR);
		}

		/* Check if the node definition being deleted is being used by any template */
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		if (!removeReferences) {
			List<WorkflowTemplateDTO> templateList = new ArrayList<WorkflowTemplateDTO>();
			if (nodeDefinition.getType().equals(NodeDefinitionType.MessageNode)) {
				Collection<String> messageUIDs = new ArrayList<String>(); 
				for (Message msg : nodeDefinition.getMessages()) {
					messageUIDs.add(msg.getUid());
				}
				templateList = templateDAO.findByMessageUsage(messageUIDs, NodeDefinitionType.MessageNode);		
			} else {
				for (Message msg : nodeDefinition.getMessages()) {
					String systemID = Utility.getSystemIDByMessageUID(msg.getUid());
					templateList = templateDAO.findBySuperStateUsage(nodeDefinition.getName(), systemID);
					break;
				}
			}
			List<NodeDefinitionDTO> parentList = nodeDefinitionDAO.findByParentUsage(nodeDefinitionId);
			List<IWatchPointDTO> references = new ArrayList<IWatchPointDTO>();
			references.addAll(templateList);
			references.addAll(parentList);
			if (!references.isEmpty())
				throw new ReferenceDeletionException(ServiceErrorCode.ND_NODE_DEFN_REFERENCED,
						references);
		}
		
		/* Delete references from parent node definitions */
		nodeDefinitionDAO.deleteNodeDefinitionReferences(nodeDefinitionId);
		
		nodeDefinition.getResourceType().removeNode(nodeDefinition);
		nodeDefinitionDAO.delete(nodeDefinitionId);
	}

	/**
	 * Implementation of the business logic for the nameExists method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.NodeDefinitionService#nameExists(String)
	 */
	protected boolean nameExists(String name, IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return nodeDefinitionDAO.getCountByName(name) > 0;
	}
	
	/**
	 * Implementation of the business logic
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.NodeDefinitionService#isUsedByNodeDefinition(long)
	 */
	protected boolean isUsedByNodeDefinition(long nodeDefinitionId, IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAO = this.createNodeDefinitionDAO(context);
		return (nodeDefinitionDAO.getCountByParentUsage(nodeDefinitionId) > 0);
	}
	
	// ========================================================================
	// ==================== HELPER METHODS
	// ========================================================================

	/**
	 * Create the data access object for the NodeDefinitionDTODAI
	 * 
	 * @param context
	 *            The Persistence Context
	 * 
	 * @return The NodeDefinitionDTO DAO.
	 */
	protected NodeDefinitionDTODAI createNodeDefinitionDAO(IPersistenceContext context) {
		Class<NodeDefinitionDTO> entityClass = NodeDefinitionDTO.class;
		DataAccessInterface<?, ?> dao = this.daFactory.createDataAccessObject(entityClass, context);
		return (NodeDefinitionDTODAI) dao;
	}

	/**
	 * Create the data access object for the ResourceTypeDAI
	 * 
	 * @param context
	 *            The Persistence Context
	 * 
	 * @return The ResourceType DAO.
	 */
	protected ResourceTypeDAI createResourceTypeDAO(IPersistenceContext context) {
		Class<ResourceType> entityClass = ResourceType.class;
		DataAccessInterface<?, ?> dao = this.daFactory.createDataAccessObject(entityClass, context);
		return (ResourceTypeDAI) dao;
	}

	/**
	 * Create the data access object for the WorkflowTemplateDTODAI
	 * 
	 * @param context
	 *            The Persistence Context
	 * 
	 * @return The WorkflowTemplateDTODAI DAO.
	 */
	protected WorkflowTemplateDTODAI createTemplateDAO(IPersistenceContext context) {
		Class<WorkflowTemplateDTO> entityClass = WorkflowTemplateDTO.class;
		DataAccessInterface<?, ?> dao = this.daFactory.createDataAccessObject(entityClass, context);
		return (WorkflowTemplateDTODAI) dao;
	}

	/*
	 * Load the Template XML as a Document.
	 * 
	 * @param nodeDefinition The Workflow Template
	 * 
	 * @return The Document containing the template xml.
	 */
	private Document loadNodeDefinitionDocument(NodeDefinitionDTO nodeDefinition) {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			factory.setNamespaceAware(true);
			StringReader reader = new StringReader(nodeDefinition.getXml());

			return factory.newDocumentBuilder().parse(new InputSource(reader));
		}
		catch (Exception ex) {
			String msg = "Failed to load the node definition XML.";
			throw new ServiceException(ServiceErrorCode.ND_INVALID_XML, msg, ex);
		}
	}

	/*
	 * Update the set of child Node Definitions used by the Node Definition by examining the XML
	 * contents and checking if the Node Definitions still exist in the database.
	 */
	private void updateNodeDefinitionsUsed(NodeDefinitionDTO nodeDefinition, Document document,
			NodeDefinitionDTODAI nodeDefinitionDAO) {
		Set<NodeDefinitionDTO> nodeDefinitions = new HashSet<NodeDefinitionDTO>();
		nodeDefinition.setChildren(nodeDefinitions);
		Set<Long> nodeDefinitionKeys = new HashSet<Long>();

		try {
			XPath xpath = XPathFactory.newInstance().newXPath();
			NodeList nodes = (NodeList) xpath.evaluate(NODE_DEFINITION_ID_XPATH, document,
					XPathConstants.NODESET);

			int nodeCount = (nodes != null) ? nodes.getLength() : 0;
			for (int i = 0; i < nodeCount; i++) {
				Node node = nodes.item(i);
				String nodeDefinitionIdValue = node.getNodeValue();
				if (nodeDefinitionIdValue != null && nodeDefinitionIdValue.trim().length() > 0) {
					nodeDefinitionKeys.add(Long.parseLong(nodeDefinitionIdValue));
				}
			}
		}
		catch (Exception ex) {
			String msg = "Failed to get the list of Node Definition Ids specified in the Node Definition XML.";
			throw new ServiceException(ServiceErrorCode.ND_NODE_DEFN_UPDATE_ERROR, msg, ex);
		}

		List<NodeDefinitionDTO> usedDefns = nodeDefinitionDAO.findByKeys(nodeDefinitionKeys);
		if (usedDefns != null)
			nodeDefinitions.addAll(usedDefns);
	}

	/*
	 * Update the set of Messages used by the Node Definition by examining the XML contents and
	 * checking if the Messages still exist in the database.
	 */
	private void updateMessagesUsed(NodeDefinitionDTO nodeDefinition, Document document,
			ResourceTypeDAI resourceTypeDAO) {
		Set<Message> messages = new HashSet<Message>();
		nodeDefinition.setMessages(messages);
		Set<String> messageUIDs = new HashSet<String>();

		try {
			XPath xpath = XPathFactory.newInstance().newXPath();
			NodeList sendNodes = (NodeList) xpath.evaluate(SEND_MESSAGE_UID_XPATH, document,
					XPathConstants.NODESET);
			NodeList receiveNodes = (NodeList) xpath.evaluate(RECEIVE_MESSAGE_UID_XPATH, document,
					XPathConstants.NODESET);
			addMessages(sendNodes, messageUIDs);
			addMessages(receiveNodes, messageUIDs);
		}
		catch (Exception ex) {
			String msg = "Failed to get the list of Message Uids specified in the Node Definition XML.";
			throw new ServiceException(ServiceErrorCode.ND_MESSAGE_UPDATE_ERROR, msg, ex);
		}

		List<Message> usedMessages = resourceTypeDAO.getMessagesByUid(messageUIDs);
		if (usedMessages != null)
			messages.addAll(usedMessages);
	}

	/*
	 * Adds the content of the nodes to the set of message uids.
	 */
	private void addMessages(NodeList nodes, Set<String> messageUIDs) {
		int nodeCount = (nodes != null) ? nodes.getLength() : 0;
		for (int i = 0; i < nodeCount; i++) {
			Node node = nodes.item(i);
			String uid = node.getTextContent().trim();
			if (uid != null && uid.trim().length() > 0) {
				messageUIDs.add(uid);
			}
		}
	}

	// ========================================================================
	// ==================== VALIDATION METHODS
	// ========================================================================

	/*
	 * Validates a Template before it is created.
	 */
	private void validateNewNodeDefinition(NodeDefinitionDTO nodeDefinition,
			IPersistenceContext context) {
		NodeDefinitionDTODAI nodeDefinitionDAI = this.createNodeDefinitionDAO(context);
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		String nodeDefinitionName = nodeDefinition.getName();
		if (nodeDefinitionName == null || nodeDefinitionName.length() == 0) { // Check that the
			// name is not
			// blank or empty
			ValidationMessageCode code = ValidationMessageCode.NODEDEFINITION_NAME_BLANK;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, nodeDefinition, code);
		}
		
		// Check that the node definition does not refer to Out-of-the-box resource type 
		ResourceTypeDAI resourceTypeDAO = this.createResourceTypeDAO(context);
		if(resourceTypeDAO.isOutOfTheBox(nodeDefinition.getResourceType().getId())) {
			ValidationMessageCode code = ValidationMessageCode.NODEDEFINITION_CANNOT_CREATE_OOBRESTYPE;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, nodeDefinition, code);
		}

		/*
		 * If this is a single-node node definition, check that no other single-node node definition
		 * uses the same message.
		 */
		if(nodeDefinition.getType() == NodeDefinitionType.MessageNode) {
			Set<Message> msgs = nodeDefinition.getMessages();
			if(msgs.size() != 0) {
				if(msgs.size() > 1) {
					throw new ServiceException(ServiceErrorCode.ND_INVALID_XML);
				}
				Message message = msgs.iterator().next();
				List<NodeDefinitionDTO> nodeDefs = nodeDefinitionDAI.findSingleNodeDefinitionsByMessage(message.getId());
				if(nodeDefs != null && !nodeDefs.isEmpty()) {
					ValidationMessageCode code = ValidationMessageCode.NODEDEFINITION_SAME_MESSAGE_USED;
					ValidationMessageAdder.getInstance().addValidationMessage(messages, nodeDefinition, code);
				}
			}
		}

		if (messages.size() > 0) {
			String msg = "The node definition failed validation during creation.";
			throw new ServiceValidationException(msg, messages);
		}
	}
}
