/*
 * Created on Jun 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathException;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.ProtectionKeyDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.fs.dao.TemplateParser;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter;
import com.tandbergtv.watchpoint.studio.external.wpexport.WatchPointDTOExporterFactory;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceErrorCode;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageType;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * The IWorkflowTemplateService implementation.
 * 
 * @author Vijay Silva
 */
public class WorkflowTemplateService extends ServiceImpl implements IWorkflowTemplateService
{
	// The Logger
	private static final Logger logger = Logger.getLogger(WorkflowTemplateService.class);

	// The XPath expression for the Process Definition element
	private static final String PROCESS_DEFINITION_XPATH = "/process-definition";

	// The name of the Process Definition 'version' attribute
	private static final String VERSION_ATTR_NAME = "version";

	/**
	 * Constructor
	 */
	public WorkflowTemplateService()
	{
		super();
	}

	/**
	 * Get the list of the latest versions of all active templates
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#getTemplateList()
	 */
	@SuppressWarnings("unchecked")
	public List<WorkflowTemplateDTO> getTemplateList()
	{
		Class<?>[] types = {};
		return (List<WorkflowTemplateDTO>) super.performOperation(types);
	}
	
	/**
	 * Get the list of Templates that references a specified template. 
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#getReferencingTemplates(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> getReferencingTemplates(String templateName)
	{
		Class<?>[] types = {String.class};
		return (List<WorkflowTemplateDTO>) super.performOperation(types, templateName);
	}
	
	/**
	 * 	Get the list of Templates that refers to a specified ResourceType.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#getTemplatesByResourceTypeUsage(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> getTemplatesByResourceTypeUsage(String systemID) {
		Class<?>[] types = {String.class};
		return (List<WorkflowTemplateDTO>) super.performOperation(types, systemID);
	}
	
	/**
	 * 	Get the list of Templates that refers to a specified ResourceGroup.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#getTemplatesByResourceGroupUsage(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> getTemplatesByResourceGroupUsage(String resourceGroupName) {
		Class<?>[] types = {String.class};
		return (List<WorkflowTemplateDTO>) super.performOperation(types, resourceGroupName);
	}

	/**
	 * Create a new Template.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#createTemplate(WorkflowTemplateDTO,
	 *      String, boolean)
	 */
	public WorkflowTemplateDTO createTemplate(WorkflowTemplateDTO template, String ownerId,
			boolean lockTemplate)
	{
		Class<?>[] types = { WorkflowTemplateDTO.class, String.class, boolean.class };
		return (WorkflowTemplateDTO) super.performOperation(types, template, ownerId, lockTemplate);
	}
	
	/**
	 * Parses a template.
	 * @throws XPathException 
	 * @throws TransformerException 
	 * @throws CoreException 
	 * @throws ExportFailureException 
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#exportTemplate(Document)
	 */
	public WorkflowTemplateDTO exportTemplate(WatchPointTemplateEditor editor) throws ExportFailureException {
		Document document = editor.getDocumentElement();

		WorkflowTemplateDTO dto = null;
		try {
			dto = TemplateParser.parse(editor);
		} catch (Exception e) {
			logger.debug("An error ocurred while trying to parse the template. ", e);
			throw new ExportFailureException("An error ocurred while trying to parse the template.", e);
		} 
		
		List<ValidationMessage> messages = validateTemplate(dto);
		
		// User can export if template have warnings or notifications, but not errors
		for (ValidationMessage message : messages) {
			if (message.getType() == ValidationMessageType.Error) {
				throw new ExportFailureException("Template contains validation errors. Please, resolve all the errors before you export.");
			}
		}
		
		int version = dto.getVersion();
		version++;
		updateTemplateVersion(document, version);
		editor.doSave(null);
		
		// After updating the JPDL's version, have to generate the runtime file again. 
		WatchPointDTOExporterFactory exporterFactory = WatchPointDTOExporterFactory.createFactory();
		IWatchPointDTOExporter<WorkflowTemplateDTO> exporter = exporterFactory.getExporter(WorkflowTemplateDTO.class);
		exporter.export(dto, null);

		return dto;
	}

	/**
	 * Get a Template.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#getTemplateByName(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	public List<WorkflowTemplateDTO> getTemplateByName(String templateName)
	{
		Class<?>[] types = { String.class };
		return (List<WorkflowTemplateDTO>) super.performOperation(types, templateName);
	}
	
	/**
	 * Implementation of the business logic for the getTemplate method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#getTemplateByName(java.lang.String)
	 */
	protected List<WorkflowTemplateDTO> getTemplateByName(String templateName, IPersistenceContext context)
	{
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		return templateDAO.findByName(templateName);
	}
	
	/**
	 * Get a Template.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#getTemplateByPath(java.lang.String)
	 */
	public WorkflowTemplateDTO getTemplateByPath(String path)
	{
		Class<?>[] types = { String.class };
		return (WorkflowTemplateDTO) super.performOperation(types, path);
	}
	
	/**
	 * Implementation of the business logic for the getTemplate method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#getTemplateByPath(java.lang.String)
	 */
	protected WorkflowTemplateDTO getTemplateByPath(String path, IPersistenceContext context)
	{
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		return templateDAO.findByPath(path);
	}
	
	/**
	 * Save a Template. The Template should already be locked.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#saveTemplate(WorkflowTemplateDTO,
	 *      String)
	 */
	public WorkflowTemplateDTO saveTemplate(WorkflowTemplateDTO template)
	{
		Class<?>[] types = { WorkflowTemplateDTO.class };
		return (WorkflowTemplateDTO) super.performOperation(types, template);
	}

	/**
	 * Delete a Template and all its versions with a matching name.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#deleteTemplate(String,
	 *      String)
	 */
	public void deleteTemplate(String templateName)
	{
		Class<?>[] types = { String.class };
		super.performOperation(types, templateName);
	}

	// ========================================================================
	// ====== BUSINESS LOGIC IMPLEMENTATION (WITHOUT CONTEXT MANAGEMENT)
	// ========================================================================

	/**
	 * Implementation of the business logic for the getTemplateList method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#getTemplateList()
	 */
	protected List<WorkflowTemplateDTO> getTemplateList(IPersistenceContext context)
	{
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		
		List<WorkflowTemplateDTO> templates = templateDAO.findAll();
		
		return templates;
	}
	
	/**
	 * Implementation of the business logic for the getTemplatesByResourceTypeUsage method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#getTemplatesByResourceTypeUsage(String)
	 */
	protected List<WorkflowTemplateDTO> getTemplatesByResourceTypeUsage(String systemID, IPersistenceContext context) {
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		
		List<WorkflowTemplateDTO> templates = templateDAO.findByResourceTypeUsage(systemID);
		
		return templates;
	}
	
	/**
	 * Implementation of the business logic for the getTemplatesByResourceGroupUsage method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#getTemplatesByResourceTypeUsage(String)
	 */
	protected List<WorkflowTemplateDTO> getTemplatesByResourceGroupUsage(String resourceGroupName, IPersistenceContext context) {
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		
		List<WorkflowTemplateDTO> templates = templateDAO.findByResourceGroupUsage(resourceGroupName);
		
		return templates;
	}
	
	/**
	 * Implementation of the business logic for the getReferencingTemplates method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#getReferencingTemplates(String)
	 */
	protected List<WorkflowTemplateDTO> getReferencingTemplates(String templateName, IPersistenceContext context)
	{
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		
		List<WorkflowTemplateDTO> templates = templateDAO.findByTemplateUsage(templateName);
		
		return templates;
	}

	/**
	 * Implementation of the business logic for the saveTemplate method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#saveTemplate(WorkflowTemplateDTO,
	 *      String)
	 */
	protected WorkflowTemplateDTO saveTemplate(WorkflowTemplateDTO template, IPersistenceContext context)
	{
		// Get the template already present in persistence layer
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		
		WorkflowTemplateDTO persistedTemplate = templateDAO.findByPath(template.getPath());

		if (persistedTemplate != null) {
			template.setId(persistedTemplate.getId());
		}
		
		// Update the used ProtectionKeys definitions
		this.updateProtectionKeysUsed(template, context);

		// Save the Template to the database with the appropriate version number
		if (persistedTemplate != null) {
			template = templateDAO.update(template);
		} else {
			try {
				template = templateDAO.create(template);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}

		return template;
	}

	/**
	 * Implementation of the business logic for the deleteTemplate method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#deleteTemplate(String,
	 *      String)
	 */
	protected void deleteTemplate(String templatePath, IPersistenceContext context) {
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		
		WorkflowTemplateDTO toDelete = templateDAO.findByPath(templatePath);
		
		if (toDelete != null) {
			templateDAO.delete(toDelete.getKey());
			logger.debug("Template deleted: " + templatePath);
		} else {
			logger.debug("Template not found to delete: " + templatePath);
		}

	}

	// ========================================================================
	// ==================== HELPER METHODS
	// ========================================================================

	/*
	 * Create the data access object for the WorkflowTemplateDTO
	 */
	private WorkflowTemplateDTODAI createTemplateDAO(IPersistenceContext context)
	{
		return this.daFactory.getWorkflowTemplateDTODAO(context);
	}

	/*
	 * Create the data access object for the NodeDefinitionDTODAI
	 */
	private ProtectionKeyDAI createProtectionKeyDAO(IPersistenceContext context)
	{
		Class<ProtectionKey> entityClass = ProtectionKey.class;
		DataAccessInterface<?, ?> dao = this.daFactory.createDataAccessObject(entityClass, context);
		return (ProtectionKeyDAI) dao;
	}

	/*
	 * Update all the Protection Keys used by the Template, creating Protection Keys that do not
	 * currently exist.
	 */
	private void updateProtectionKeysUsed(WorkflowTemplateDTO template, IPersistenceContext context)
	{
		Set<ProtectionKey> persistedProtectionKeys = new HashSet<ProtectionKey>();

		ProtectionKeyDAI protectionKeyDAO = createProtectionKeyDAO(context);
		for (ProtectionKey protectionKey : template.getProtectionKeys()) {
			ProtectionKey persistedPK = protectionKeyDAO.findByName(protectionKey.getName());
			if (persistedPK == null) {
				persistedPK = protectionKeyDAO.create(protectionKey);
			}
			persistedProtectionKeys.add(persistedPK);
		}
		
		template.setProtectionKeys(persistedProtectionKeys);
	}

	/*
	 * Update the Template version in the entity and the template xml document
	 */
	private Document updateTemplateVersion(Document document, int version) {
		try 		{
			// Write the version in the template xml
			XPath xpath = XPathFactory.newInstance().newXPath();
			Element processDefinitionElement = (Element) xpath.evaluate(PROCESS_DEFINITION_XPATH,
					document, XPathConstants.NODE);
			
			processDefinitionElement.setAttribute(VERSION_ATTR_NAME, Integer.toString(version));
			
		} catch (Exception ex) {
			String msg = "Failed to update the version attribute in the Workflow Template XML.";
			throw new ServiceException(ServiceErrorCode.WT_VERSION_UPDATE_ERROR, msg, ex);
		}
		
		return document;
	}
	// ========================================================================
	// ==================== VALIDATION METHODS
	// ========================================================================

	/*
	 * Validates a Template before it is created.
	 */
	private List<ValidationMessage> validateTemplateName(WorkflowTemplateDTO template, IPersistenceContext context) {
		WorkflowTemplateDTODAI templateDAI = this.createTemplateDAO(context);
		
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();
		String templateName = template.getName();
		if (templateName == null || templateName.length() == 0) { 
			// Check that the name is not blank or empty
			ValidationMessageCode code = ValidationMessageCode.TEMPLATE_NAME_BLANK;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, template, code);
		} else { // Check that the name is unique
			List<WorkflowTemplateDTO> templates = templateDAI.findByName(template.getName());
			for (WorkflowTemplateDTO existingTemplate : templates) {
				if (!template.getPath().equals(existingTemplate.getPath())) {
					ValidationMessageCode code = ValidationMessageCode.TEMPLATE_NAME_NOT_UNIQUE;
					ValidationMessageAdder.getInstance().addValidationMessage(messages, template, code);
				}
			}
		}
		
		return messages;
	}

	/*
	 * Validates a Template before it is created.
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateTemplate(WorkflowTemplateDTO template) {
		Class<?>[] types = { WorkflowTemplateDTO.class };
		return (List<ValidationMessage>) super.performOperation(types, template);
	}
	
	/*
	 * Validates a Template using the Validation framework.
	 */
	protected List<ValidationMessage> validateTemplate(WorkflowTemplateDTO template, IPersistenceContext context)  {
		ValidationServiceFactory factory = ValidationServiceFactory.createFactory();
		IValidationService validationService = factory.createValidationService();
		List<ValidationMessage> messages = validationService.validateTemplate(template);
		messages.addAll(validateTemplateName(template, context));
		
		return messages;
	}
	
	/**
	 *	Gets the templates for a specified project.
	 *
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService.getTemplatesByProject(String) 
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> getTemplatesByProject(String projectName) {
		Class<?>[] types = {String.class};
		return (List<WorkflowTemplateDTO>) super.performOperation(types, projectName);
	}
	
	protected List<WorkflowTemplateDTO> getTemplatesByProject(String projectName, IPersistenceContext context) {
		WorkflowTemplateDTODAI templateDAI = this.createTemplateDAO(context);
		
		return templateDAI.findByProjectName(projectName);
	}

	/**
	 * 	Get the list of Templates that refers to a specified ResourceType.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService#getTemplatesBySuperStateUsage(java.lang.String, java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> getTemplatesBySuperStateUsage(String nodeDefinitionName, String systemID) {
		Class<?>[] types = {String.class, String.class};
		return (List<WorkflowTemplateDTO>) super.performOperation(types, nodeDefinitionName, systemID);
	}
	
	
		
	/**
	 * Implementation of the business logic for the getTemplatesBySuperStateUsage method
	 * 
	 * @see com.tandbergtv.watchpoint.studio.service.impl.WorkflowTemplateService#getTemplatesBySuperStateUsage(String, String)
	 */
	protected List<WorkflowTemplateDTO> getTemplatesBySuperStateUsage(String nodeDefinitionName, String systemID, IPersistenceContext context) {
		WorkflowTemplateDTODAI templateDAO = this.createTemplateDAO(context);
		
		List<WorkflowTemplateDTO> templates = templateDAO.findBySuperStateUsage(nodeDefinitionName, systemID);
		
		return templates;
	}
	
	
}
